///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_CORE_H
#define MOIRA_CORE_H
///////////////////////////////////////////////////////////////////////

#ifndef NULL
#define NULL 0
#endif

#include <cstdlib>
#include <list>
#include <map>
#include <string>
#include <vector>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

/*! Byte value.
 */
typedef unsigned char Byte;

/*! Time value, in seconds.
 */
typedef double Time;

/*! String type.
 */
typedef std::string String;

/*! Generic function pointer type.
 */
typedef void (*EntryPoint)(void);

///////////////////////////////////////////////////////////////////////

/*! Base class for exceptions.
 */
class Exception : public std::exception
{
public:
  /*! Constructor.
   *  @param initMessage The message for this exception.
   */
  Exception(const char* initMessage);
  /*! @return The massage for this exception.
   */
  const char* what(void) const throw();
private:
  const char* message;
};

///////////////////////////////////////////////////////////////////////

/*! @brief Scoped automatic pointer.
 *
 *  Smart pointer. Manages a single object of type T.
 */
template <typename T>
class Ptr
{
public:
  /*! Default constructor.
   */
  inline Ptr(T* object = NULL);
  /*! Copy constructor.
   *  @param source The pointer object to inherit from.
   */
  inline Ptr(const Ptr<T>& source);
  /*! Destructor
   */
  inline virtual ~Ptr(void);
  /*! Detaches (orphans) the currently owned object.
   * @return The currently owned object.
   * @remarks Use with care.
   */
  inline T* detachObject(void);
  /*! Cast operator.
   */
  inline operator T* (void);
  /*! Cast operator.
   */
  inline operator const T* (void) const;
  /*! Member operator.
   */
  inline T* operator -> (void);
  /*! Member operator.
   */
  inline const T* operator -> (void) const;
  /*! Object assignment operator.
   */
  inline Ptr<T>& operator = (T* newObject);
  /*! Assignment operator. Transfers ownership to the lval object.
   */
  inline Ptr<T>& operator = (const Ptr<T>& source);
  /*! @return The currently owned object.
   */
  inline T* getObject(void);
  /*! @return The currently owned object.
   */
  inline const T* getObject(void) const;
private:
  T* object;
};

///////////////////////////////////////////////////////////////////////

/*! @brief Smart reference.
 *
 *  Pointer to objects that inherit from RefObject.
 */
template <typename T>
class Ref
{
public:
  /*! Default constructor.
   */
  inline Ref(T* object = NULL);
  /*! Copy constructor.
   *  @param source The pointer object to inherit from.
   */
  inline Ref(const Ref<T>& source);
  /*! Destructor
   */
  inline virtual ~Ref(void);
  /*! Cast operator.
   */
  inline operator T* (void) const;
  /*! Member operator.
   */
  inline T* operator -> (void) const;
  /*! Object assignment operator.
   */
  inline Ref<T>& operator = (T* newObject);
  /*! Assignment operator.
   */
  inline Ref<T>& operator = (const Ref<T>& source);
  /*! @return The currently owned object.
   */
  inline T* getObject(void) const;
private:
  T* object;
};

///////////////////////////////////////////////////////////////////////

/*! @brief Template mixin for reference counting.
 *
 *  @remarks No, there are no visible knobs on this class. Use the Ref class to
 *  point to objects derived from RefObject to enable reference counting.
 */
template <typename T>
class RefObject
{
  friend class Ref<T>;
public:
  /*! Constructor.
   */
  inline RefObject(void);
  /*! Copy constructor.
   */
  inline RefObject(const RefObject<T>& source);
  /*! Destructor.
   */
  inline virtual ~RefObject(void);
  /*! Assignment operator.
   */
  inline RefObject<T>& operator = (const RefObject<T>& source);
private:
  unsigned int count;
};

///////////////////////////////////////////////////////////////////////

/*! @brief %Singleton template mixin.
 *
 *  Inherit from this to become a compatible singleton.
 */
template <typename T>
class Singleton
{
public:
  /*! Deletes the singleton instance.
   */
  static inline void destroy(void);
  /*! @return The singleton instance if available, otherwise @c NULL.
   */
  static inline T* get(void);
protected:
  /*! Sets the singleton instance.
   *  @param object [in] The instance to set.
   */
  static inline void set(T* newObject);
private:
  static Ptr<T> object;
};

///////////////////////////////////////////////////////////////////////

/*! @brief %Managed object template mixin.
 *
 *  Base class for managed objects with unique literal names.
 *  Names must be unique within each class that derives from this class.
 */
template <typename T>
class Managed
{
public:
  /*! List of instances, as returned by Managed::getInstances.
   */
  typedef std::vector<T*> InstanceList;
  /*! Constructor.
   *  @param name The desired name of this object, or the empty string to
   *  request automatic name generation.
   *
   *  @remarks If the name you specified already exists, an exception will be
   *  thrown.
   */
  inline Managed(const String& name);
  /*! Copy constructor.
   *
   *  @remarks If a managed object is created through a copy constructor, it
   *  receives an automatically generated name.
   */
  inline Managed(const Managed<T>& source);
  /*! Destructor.
   */
  inline virtual ~Managed(void);
  /*! Assignment operator.
   *
   *  @remarks This does not change the name of the object being assigned to.
   */
  inline Managed<T>& operator = (const Managed<T>& source);
  /*! @return The unique name of this object.
   */
  inline const String& getName(void) const;
  /*! Sets the name of this object.
   *  @param[in] newName The desired new name of this object.
   *  @return @c true if this object was successfully renamed, or @c false if
   *  the name already existed.
   *  
   *  @note This feature is mainly for editor interfaces, which interact with
   *  users expecting to be able to rename objects at their leisure.  If you
   *  rename an object, make sure you change all the places where the previous
   *  name of the object is stored, as these will not be changed for you.
   *
   *  @note If you specify the empty string as the desired name, the object will
   *  get a new (guaranteed unique) automatically generated name.
   */
  inline bool setName(const String& newName);
  /*! Returns the instance with the specified name, if available.
   *  @param name The name of the desired instance.
   *  @return The instance with the specified name, or @c NULL if no such
   *  instance exists.
   */
  static inline T* findInstance(const String& name);
  /*! Destroys all instances of this type.
   *
   *  @remarks Use with care. This will backfire if you have objects of type T
   *  on the stack, or if higher-level objects own any of the destroyed
   *  instances.
   */
  static inline void destroyInstances(void);
  /*! Builds a list of all current instances of this type.
   *  @param instances The resulting list of instances.
   */
  static inline void getInstances(InstanceList& instances);
private:
  inline void generateName(String& name) const;
  typedef std::map<String, Managed<T>*> InstanceMap;
  String name;
  static InstanceMap instanceMap;
};

///////////////////////////////////////////////////////////////////////

/*! @brief Heap memory byte block container.
 */
class Block
{
public:
  /*! Default constructor.
   *  @param size The initial size, in bytes, of this data block.
   */
  explicit Block(size_t size = 0);
  /*! Constructor. Copies the specified number of bytes into this block
   *  @param source The data to copy into this block.
   *  @param sourceSize The size, in bytes, of the data.
   */
  Block(const Byte* source, size_t sourceSize);
  /*! Copy constructor.
   *  @remarks Performs a deep copy of data items.
   */
  Block(const Block& source);
  /*! Destructor.
   */
  ~Block(void);
  /*! Copies the specified number of bytes from this block, starting
   *  at the specified offset.
   *  @param target The target buffer for the data to be copied.
   *  @param targetSize The number of bytes to copy.
   *  @param offset The desired offset within this data block, in bytes,
   *  from which to start.
   */
  void copyTo(Byte* target, size_t targetSize, size_t offset = 0) const;
  /*! Copies the specified number of bytes into this block, starting
   *  at the specified offset.
   *  @param source The source buffer from which to copy.
   *  @param sourceSize The number of bytes to copy.
   *  @param offset The desired offset within this data block, in bytes,
   *  from which to start.
   */
  void copyFrom(const Byte* source, size_t sourceSize, size_t offset = 0);
  /*! Changes the number of bytes in this data block.
   *  @param newSize The new size, in bytes, of this data block.
   */
  void resize(size_t newSize);
  /*! Assures that the size of this block is at least the specified
   *  number of bytes.
   *  @param minSize The minimum desired size, in bytes.
   */
  void reserve(size_t minSize);
  void attach(Byte* newData, size_t newSize);
  Byte* detach(void);
  /*! Frees the currently allocated data block.
   */
  void destroy(void);
  /*! Cast operator.
   */
  operator Byte* (void);
  /*! Cast operator.
   */
  operator const Byte* (void) const;
  /*! Assignment operator.
   */
  Block& operator = (const Block& source);
  /*! @return The size, in bytes, of this data block.
   */
  size_t getSize(void) const;
  /*! @return The allocation granularity, or zero if granularity is disabled.
   */
  size_t getGrain(void) const;
  /*! Sets or disables the allocation granularity.
   *  @param grain The desired allocation granularity, or zero to disable.
   */
  void setGrain(size_t newGrain);
  /*! @return The data block.
   */
  Byte* getData(void);
  /*! @return The first item.
   */
  const Byte* getData(void) const;
private:
  size_t size;
  size_t grain;
  Byte* data;
};

///////////////////////////////////////////////////////////////////////

/*! @brief Bidirectional key mapper template.
 *
 *  This is a performance-wise highly suboptimal, but syntactically delicious,
 *  way of maintaining a bidirectional mapping between values, which is often
 *  useful when you wish to hide platform- or API-specific constants and
 *  enumerations, and need to convert back and forth between them and you
 *  public enumeration values.
 *
 *  @remarks For this to work, the types must be distinguishable.
 */
template <typename X, typename Y>
class Mapper
{
public:
  inline Mapper(void);
  inline Mapper(const X& defaultX, const Y& defaultY);
  inline X& operator [] (const Y& value);
  inline const X& operator [] (const Y& value) const;
  inline Y& operator [] (const X& value);
  inline const Y& operator [] (const X& value) const;
  inline bool isEmpty(void) const;
  inline bool hasKey(const X& key) const;
  inline bool hasKey(const Y& key) const;
  inline void setDefaults(const X& defaultX, const Y& defaultY);
private:
  struct Entry
  {
    X x;
    Y y;
  };
  typedef std::vector<Entry> EntryList;
  EntryList entries;
  Entry defaults;
};

///////////////////////////////////////////////////////////////////////

class Variant
{
public:
  Variant(void);
  Variant(const String& value);
  float asFloat(void) const;
  void setFloatValue(float newValue);
  int asInteger(void) const;
  void setIntegerValue(int newValue);
  bool asBoolean(void) const;
  void setBooleanValue(bool newValue);
  const String& asString(void) const;
  void setStringValue(const String& newValue);
  static float convertToFloat(const String& value);
  static int convertToInteger(const String& value);
  static bool convertToBoolean(const String& value);
  static void convertToString(String& result, float value);
  static void convertToString(String& result, int value);
  static void convertToString(String& result, bool value);
private:
  String value;
};

///////////////////////////////////////////////////////////////////////

/*! @brief File system path descriptor.
 *
 *  Represents the path to a single file or directory.
 */
class Path
{
public:
  /*! List of paths. Used to return child paths.
   */
  typedef std::vector<Path> List;
  /*! Constructor. Creates a path object with the specified name.
   */
  explicit Path(const String& name = "");
  /*! Constructor. Creates a path object with the specified name.
   */
  explicit Path(const char* format, ...);
  /*! Creates a directory with this path.
   *  @return @c true if successful, otherwise @c false.
   */
  bool createDirectory(void) const;
  /*! Destroys the directory with this path.
   *  @return @c true if successful, otherwise @c false.
   */
  bool destroyDirectory(void) const;
  /*! @return @c true if a file or directory with this path exists, otherwise
   *  @c false.
   */
  bool exists(void) const;
  /*! @return This path represented as a string.
   */
  const String& asString(void) const;
  /*! Append operator. Creates a path with this path as the directory part and
   *  the specified name as the leaf (file or directory) name.
   *  @param[in] child The desired leaf name.
   *  @return The resulting path.
   */
  Path operator + (const String& child) const;
  Path& operator += (const String& child);
  bool operator == (const Path& other) const;
  bool operator != (const Path& other) const;
  /*! Assignment operator.
   */
  Path& operator = (const String& newName);
  /*! @return @c true if the file or directory is readable, otherwise
   *  @c false.
   */
  bool isReadable(void) const;
  /*! @return @c true if the file or directory is writable, otherwise
   *  @c false.
   */
  bool isWritable(void) const;
  /*! @return @c true if the path represents a regular file, otherwise
   *  @c false.
   */
  bool isFile(void) const;
  /*! @return @c true if the path represents a directory, otherwise
   *  @c false.
   */
  bool isDirectory(void) const;
  /*! @return A path object representing the parent directory of this
   *  path object.
   *  @remarks The root directory is its own parent.
   */
  Path getParent(void) const;
  /*! Returns the paths of all files and directories in the directory with this
   *  path.
   *  @param[in,out] children The resulting list of paths.
   *  @return @c true if successful, otherwise @c false.
   */
  bool getChildren(List& children) const;
  /*! @return The suffix of the name of the represented path, or the empty
   *  string if no suffix is present.
   */
  String getSuffix(void) const;
private:
  String name;
};

///////////////////////////////////////////////////////////////////////

/*! @brief System log singleton interface.
 *
 *  Interface for the system log singleton object.
 */
class Log : public Singleton<Log>
{
public:
  /*! Log entry type enumeration.
   */
  enum EntryType
  {
    /*! The log entry is an error message.
     */
    ERROR,
    /*! The log entry is a warning, or a non-critical error message.
     */
    WARNING,
    /*! The log entry is an informational message.
     */
    INFORMATION,
  };
  /*! Destructor.
   */
  virtual ~Log(void);
  /*! Writes a log entry of the specified type to this log object.
   *  @param type [in] The type of log entry to write.
   *  @param format [in] The formatting string for the log entry.
   */
  virtual void write(EntryType type, const char* format, ...) = 0;
  /*! @return The literal name of the log entry type.
   */
  static const char* getTypeName(EntryType type);
  /*! Writes an error message log entry to the current log object,
   *  or to stderr if there is no current log object.
   *  @param format [in] The formatting string for the log entry.
   */
  static void writeError(const char* format, ...);
  /*! Writes a warning message log entry to the current log object,
   *  or to stderr if there is no current log object.
   *  @param format [in] The formatting string for the log entry.
   */
  static void writeWarning(const char* format, ...);
  /*! Writes an informational message log entry to the current log object,
   *  or to stderr if there is no current log object.
   *  @param format [in] The formatting string for the log entry.
   */
  static void writeInformation(const char* format, ...);
};

///////////////////////////////////////////////////////////////////////

/*! @brief File writer log singleton.
 */
class FileLog : public Log
{
public:
  ~FileLog(void);
  void write(EntryType type, const char* format, ...);
  static bool create(const Path& path);
private:
  FileLog(void);
  bool init(const Path& path);
  int fd;
};

///////////////////////////////////////////////////////////////////////

template <typename T>
inline Ptr<T>::Ptr(T* initObject):
  object(initObject)
{
}

template <typename T>
inline Ptr<T>::Ptr(const Ptr<T>& source):
  object(NULL)
{
}

template <typename T>
inline Ptr<T>::~Ptr(void)
{
  if (object)
    delete object;
}

template <typename T>
inline T* Ptr<T>::detachObject(void)
{
  T* temp;

  temp = object;
  object = NULL;
  return temp;
}

template <typename T>
inline Ptr<T>::operator T* (void)
{
  return object;
}

template <typename T>
inline Ptr<T>::operator const T* (void) const
{
  return object;
}

template <typename T>
inline T* Ptr<T>::operator -> (void)
{
  return object;
}

template <typename T>
inline const T* Ptr<T>::operator -> (void) const
{
  return object;
}

template <typename T>
inline Ptr<T>& Ptr<T>::operator = (T* newObject)
{
  if (object)
    delete object;

  object = newObject;
  return *this;
}

template <typename T>
inline Ptr<T>& Ptr<T>::operator = (const Ptr<T>& source)
{
  operator = (source.detachObject());
  
  return *this;
}

template <typename T>
inline T* Ptr<T>::getObject(void)
{
  return object;
}

template <typename T>
inline const T* Ptr<T>::getObject(void) const
{
  return object;
}

///////////////////////////////////////////////////////////////////////

template <typename T>
inline Ref<T>::Ref(T* initObject):
  object(NULL)
{
  operator = (initObject);
}

template <typename T>
inline Ref<T>::Ref(const Ref<T>& source):
  object(NULL)
{
  operator = (source);
}

template <typename T>
inline Ref<T>::~Ref(void)
{
  operator = (NULL);
}

template <typename T>
inline Ref<T>::operator T* (void) const
{
  return object;
}

template <typename T>
inline T* Ref<T>::operator -> (void) const
{
  return object;
}

template <typename T>
inline Ref<T>& Ref<T>::operator = (T* newObject)
{
  if (newObject)
    static_cast<RefObject<T>*>(newObject)->count++;

  if (object)
  {
    static_cast<RefObject<T>*>(object)->count--;
    if (static_cast<RefObject<T>*>(object)->count == 0)
      delete object;
  }

  object = newObject;
  return *this;
}

template <typename T>
inline Ref<T>& Ref<T>::operator = (const Ref<T>& source)
{
  return operator = (source.object);
}

template <typename T>
inline T* Ref<T>::getObject(void) const
{
  return object;
}

///////////////////////////////////////////////////////////////////////

template <typename T>
inline RefObject<T>::RefObject(void):
  count(0)
{
}

template <typename T>
inline RefObject<T>::RefObject(const RefObject<T>& source):
  count(0)
{
}

template <typename T>
inline RefObject<T>::~RefObject(void)
{
}

template <typename T>
inline RefObject<T>& RefObject<T>::operator = (const RefObject<T>& source)
{
  return *this;
}

///////////////////////////////////////////////////////////////////////

template <typename T>
inline void Singleton<T>::destroy(void)
{
  object = NULL;
}

template <typename T>
inline T* Singleton<T>::get(void)
{
  return object;
}

template <typename T>
inline void Singleton<T>::set(T* newObject)
{
  object = newObject;
}

template <typename T>
Ptr<T> Singleton<T>::object;

///////////////////////////////////////////////////////////////////////

template <typename T>
inline Managed<T>::Managed(const String& initName):
  name(initName)
{
  if (name.empty())
    generateName(name);
  else
  {
    if (findInstance(name))
      throw Exception("Duplicate name for managed object");
  }

  instanceMap[name] = this;
}

template <typename T>
inline Managed<T>::Managed(const Managed<T>& source)
{
  generateName(name);

  instanceMap[name] = this;
}

template <typename T>
inline Managed<T>::~Managed(void)
{
  instanceMap.erase(name);
}

template <typename T>
inline Managed<T>& Managed<T>::operator = (const Managed<T>& source)
{
  return *this;
}

template <typename T>
inline const String& Managed<T>::getName(void) const
{
  return name;
}

template <typename T>
inline bool Managed<T>::setName(const String& newName)
{
  if (newName.empty())
  {
    generateName(name);
    return true;
  }

  if (findInstance(newName))
    return false;

  name = newName;

  instanceMap.erase(name);
  instanceMap[name] = this;
  return true;
}

template <typename T>
inline T* Managed<T>::findInstance(const String& name)
{
  if (name.empty())
    return NULL;

  typename InstanceMap::iterator i = instanceMap.find(name);
  if (i == instanceMap.end())
    return NULL;

  return dynamic_cast<T*>((*i).second);
}

template <typename T>
inline void Managed<T>::destroyInstances(void)
{
  while (!instanceMap.empty())
  {
    typename InstanceMap::iterator i = instanceMap.begin();
    delete (*i).second;
    instanceMap.erase(i);
  }
}

template <typename T>
inline void Managed<T>::getInstances(InstanceList& instances)
{
  instances.clear();

  for (typename InstanceMap::const_iterator i = instanceMap.begin();  i != instanceMap.end();  i++)
    instances.push_back(dynamic_cast<T*>((*i).second));
}

template <typename T>
inline void Managed<T>::generateName(String& name) const
{
  do
  {
    name = "auto:";
    for (unsigned int i = 0;  i < 8;  i++)
      name.push_back('a' + std::rand() % ('z' - 'a' + 1));
  }
  while (findInstance(name));
}

template <typename T>
typename Managed<T>::InstanceMap Managed<T>::instanceMap;

///////////////////////////////////////////////////////////////////////

template <typename X, typename Y>
inline Mapper<X,Y>::Mapper(void)
{
}

template <typename X, typename Y>
inline Mapper<X,Y>::Mapper(const X& defaultX, const Y& defaultY)
{
  setDefaults(defaultX, defaultY);
}

template <typename X, typename Y>
inline X& Mapper<X,Y>::operator [] (const Y& key)
{
  for (typename EntryList::iterator i = entries.begin();  i != entries.end();  i++)
  {
    if ((*i).y == key)
      return (*i).x;
  }

  entries.push_back(Entry());
  Entry& entry = entries.back();

  entry.y = key;
  return entry.x;
}

template <typename X, typename Y>
inline const X& Mapper<X,Y>::operator [] (const Y& key) const
{
  for (typename EntryList::const_iterator i = entries.begin();  i != entries.end();  i++)
  {
    if ((*i).y == key)
      return (*i).x;
  }

  return defaults.x;
}

template <typename X, typename Y>
inline Y& Mapper<X,Y>::operator [] (const X& key)
{
  for (typename EntryList::iterator i = entries.begin();  i != entries.end();  i++)
  {
    if ((*i).x == key)
      return (*i).y;
  }

  entries.push_back(Entry());
  Entry& entry = entries.back();

  entry.x = key;
  return entry.y;
}

template <typename X, typename Y>
inline const Y& Mapper<X,Y>::operator [] (const X& key) const
{
  for (typename EntryList::const_iterator i = entries.begin();  i != entries.end();  i++)
  {
    if ((*i).x == key)
      return (*i).y;
  }

  return defaults.y;
}

template <typename X, typename Y>
inline bool Mapper<X,Y>::isEmpty(void) const
{
  return entries.empty();
}

template <typename X, typename Y>
inline bool Mapper<X,Y>::hasKey(const X& key) const
{
  for (typename EntryList::const_iterator i = entries.begin();  i != entries.end();  i++)
  {
    if ((*i).x == key)
      return true;
  }

  return false;
}

template <typename X, typename Y>
inline bool Mapper<X,Y>::hasKey(const Y& key) const
{
  for (typename EntryList::const_iterator i = entries.begin();  i != entries.end();  i++)
  {
    if ((*i).y == key)
      return true;
  }

  return false;
}

template <typename X, typename Y>
inline void Mapper<X,Y>::setDefaults(const X& defaultX, const Y& defaultY)
{
  defaults.x = defaultX;
  defaults.y = defaultY;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_CORE_H*/
///////////////////////////////////////////////////////////////////////
