;;;
;;; Snowflake by neon / darklite (geir@darkside.no)
;;;
;;; My 35 byte ZX Spectrum 48k entry for the Vintage Computing
;;; Christmas Challenge 2025.
;;;
;;; See source_no_comments.s if you want to figure out how the program
;;; works yourself.
;;;
;;; The program must be loaded at address 0xdc02 and the entrypoint is
;;; 0xdc0a.
;;;
;;; The program assumes the byte at 0xdc01 is 0x00 and that the
;;; foreground color is black and background color is gray (attribute
;;; 0x38). Both of these assumptions are valid after a clean reboot.
;;; These two assumptions save one byte each. This has been verified
;;; to work on real hardware.
;;;
;;; The program works by exploiting the symmetry in the snowflake.
;;; One quadrant of the snowflake is stored as a bitmap, and each
;;; one-bit in the bitmap indicates that a star should be
;;; plotted. This star is plotted four times, rotating the point 90
;;; degrees around the center of the snowflake each time.
;;;
;;;          *
;;;        * * .
;;;   * *   **.   . .
;;;    **    *    ..
;;;   ***  * * .  ...
;;;      *  **.  .
;;;       *  *  .
;;;  *  *  * * .  .  .
;;;   *  *  **.  .  .
;;; **********.........
;;;   .  .  ...  .  .
;;;  .  .  . . .  .  .
;;;       .  .  .
;;;      .  ...  .
;;;   ...  . . .  ...
;;;    ..    .    ..
;;;   . .   ...   . .
;;;        . . .
;;;          .
;;;
;;; The size of the snowflake is 19x19, so in theory 10x10 bits is
;;; needed as shown above. Fortunately there are some other properties
;;; of the snowflake that can be used to reduce the size.
;;;
;;; The right-most column in a quadrant (the vertical center line in
;;; the snowflake) is always a star, so it can be omitted from the
;;; bitmap. The bottom line in a quadrant (the horizontal center line)
;;; is the same as the right-most column rotated, so it can also be
;;; omitted. Because of this, the left-most column is always empty and
;;; is not needed.
;;;
;;; In the end a 10x8 bitmap is used, shown below. The x's are the
;;; stars that are always plotted. The value of the bottom byte,
;;; indicated by .'s, does not matter because of the rotation, but it
;;; still has to be processed because of the star in the center.
;;;
;;;  00000000x
;;;  00000010x
;;;  01010001x
;;;  00110000x
;;;  01110010x
;;;  00001001x
;;;  00000100x
;;;  10010010x
;;;  01001001x
;;;  ........x
;;;

	.z180
	.allow_undocumented
	;; This is the bitmap and is located at 0xdc02. The byte at
	;; 0xdc01 is assumed to be zero (see the assumptions mentioned
	;; earlier).
	.db	0b00000010
	.db	0b01010001
	.db	0b00110000
	.db	0b01110010
	.db	0b00001001
	.db	0b00000100
	.db	0b10010010
	.db	0b01001001
_start::
	;; This is the entrypoint and is located at address 0xdc01.
	;; The initial value of bc is this address.

	;; Set h to 0xdc. hl will be used to access the bitmap. This
	;; instruction is also the first byte of the bitmap.
	ld	h, b

	;; The x loop runs 10 times. The b register is used as the
	;; counter and as the x coordinate and counts down from 10 to
	;; 1.
	ld	b, #10

	;; The carry flag is set to always run the plot_loop in the
	;; first iteration of the x loop.
	scf
x_loop:
	;; Skip plotting if carry is not set. In the first iteration,
	;; carry is always set, but for the rest the carry is the bits
	;; in the current bitmap byte (except for the last iteration
	;; when it is always 0).
	jr	nc, skip_plot

	;; This is the plot loop. It plots a star at the coordinate in
	;; b,c and then rotates the point 90 degrees. If the number of
	;; iterations is a multiple of 4, the coordinate ends up where
	;; it started. e is used as a loop counter. The very first
	;; time this loop runs e can be anything, but since the
	;; coordinate is 10,10 so it rotating it does nothing. After
	;; the loop has run, the e register is zero so the next time
	;; the number of iterations is always a multiple of 4.
	;;
	;; This plot technique is inspired by Dr. Beep's VCCC 2022
	;; entry.
plot_loop:
	;; Set the plot position from bc, where b is x and c is y.
	call	0x200a

	;; Plot the star
	ld	a, #'*'
	rst	0x10

	;; Rotate the coordinate in b,c 90 degrees around the center
	;; of the snowflake (position 10,10)
	;;   b' = 10-c
	;;   c' = b
	;; The a register contains 0x38 (the attribute value) after
	;; plotting the star. Change the value to 10 by adding the
	;; high byte of the bitmap address, which is 0xdc.
	add	h
	sub	c
	ld	c, b
	ld	b, a

	;; Loop until e reaches zero.
	dec	e
	jr	nz, plot_loop

skip_plot:
	;; Set l to the y coordinate in l. hl now points to the bitmap.
	ld	l, c

	;; Shift the bitmap byte to the right. The carry flag is set
	;; to the bit that is shifted out.
	srl	(hl)

	;; Loop until b (x) reaches zero.
	djnz	x_loop

	;; Decrement c (y) and return when it reaches zero.
	dec	c
	ret	z

	;; Jump back to the top (basically the y loop). This will jump
	;; to the bitmap byte that has just been plotted. This
	;; byte will now be 0x00 (nop) because of the srl instruction.
	jp	(hl)
