#version 460

#define GEOMETRY_INFORMATION_STATIC 1
#define LIGHT_PROPERTIES_BINDING 1

#extension GL_EXT_debug_printf : enable

uniform sampler2D      s_NoiseRGBA;
uniform sampler2DArray s_BlueNoise;

#include <shaders/materials/commons.glsl>
#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/noise/noise3d.glsl>
#include <shaders/materials/commons_instancing_buffers.h>
#include <shaders/materials/commons_gradient.glsl>
#include <shaders/deferred/lighting/lighting_support.glsl>

#include <shaders/geometry_partitioning/raytrace_buffers.glsl>
#include <shaders/geometry_partitioning/raytrace_commons.glsl>

uniform sampler2D sImage1;
uniform sampler2D sImage2;

uniform sampler3D s_voxel_colors;
uniform sampler3D s_voxel_colors_filtered;
uniform sampler3D s_voxel_occupancy;
uniform sampler3D s_voxel_occupancy_filtered;
uniform sampler2D sDepth;

layout(location = 1) in struct
{
	vec3 vCoords;
	vec3 vNorm;
	vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vCameraRelativeWorldPos;
	vec4 vColor;
	vec2 vUV0;
} vtx_input;
layout(location = 0) in flat uint instanceID;

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

struct RaymarchParams
{
	vec3  function_scale;
	int   show_bounding_box;
	vec3  function_origin;
	int   clamp_to_volume;

	int   use_instance_origin;
	int   txt1_flip_y;
	int   txt2_flip_y;
	int   gradient_idx;
	vec4  near_far_plane;
	int   trace_inside;
	int   _pad0;
	int   _pad1;
	int   _pad2;
	vec4  camera_projection_params;
	vec2  frustum_shift;
	vec2  resolution;

	float animation_time;
	float prestep;
	float intersection_density_modifier;
	float density;
	float intensity;
	float opacity;
	float dithering;
	float sampling_mip;
};


#note rename param1 Animation Time
#note rename param2 Prestep
#note rename param3 Intersection Density Modifier
#note rename param4 Density
#note rename param5 Intensity
#note rename param6 Opacity
#note rename param7 Dithering
#note rename param8 Sampling Mip


layout(std140, row_major) uniform RaymarchParamsBuffer{
	RaymarchParams raymarch_params;
};

layout(std140, row_major) uniform BaseMaterialPropertiesBuffer
{
	vec4 colorDiffuse;
	int gUseDerivedNormal;
	int gMaterialMode;
	int materialId;
	int materialIndex;
	int componentTags;
};

// NOTE: we try to always render using base object front faces so we have conservative depth
//       when inside the object we will just pass the define and have 2 shaders and use the second
//       one to not use the conservative depth

// output for 2 buffers
#ifdef DEFERRED_PASS
layout(location = 0) out vec4 outAlbedo;
#ifndef DEFERRED_PASS_ALBEDO_ONLY
layout(location = 1) out uint outNormalMaterial;
layout(location = 2) out uvec4 outMetalnessRoughnessMeterialTags;
layout(location = 3) out vec4 outEmissive;
#endif
#endif

//

float linearizeDepth(float d)
{
	return raymarch_params.near_far_plane.z / (raymarch_params.near_far_plane.y + raymarch_params.near_far_plane.x - d * raymarch_params.near_far_plane.w);
}

vec3 get_view_direction(vec2 screen_pos)
{
	vec2 vd_pos = screen_pos - raymarch_params.frustum_shift.xy * raymarch_params.resolution.xy * vec2(0.5, -0.5);
	vec3 view_direction;

	view_direction.x = -raymarch_params.camera_projection_params.z + raymarch_params.camera_projection_params.x * vd_pos.x / raymarch_params.resolution.x;
	view_direction.y = -raymarch_params.camera_projection_params.w + raymarch_params.camera_projection_params.y * vd_pos.y / raymarch_params.resolution.y;
	view_direction.z = 1.0;

	#ifdef SPIRV_VULKAN
	view_direction.y = -view_direction.y;
	#endif

	return view_direction;
}

vec4 sample_voxelization_data(vec3 world_position, float color_lod, float occupancy_lod)
{
	vec3 voxelizer_bbox_origin = in_bbox_data.bbox_voxelize_min.xyz;
	vec3 voxelizer_bbox_size   = in_bbox_data.grid_size_voxelize.xyz * GRID_RES;

	vec3 source_voxel_position = (world_position - voxelizer_bbox_origin) / voxelizer_bbox_size;

	if (source_voxel_position.x < 0.0 || source_voxel_position.y < 0.0 || source_voxel_position.z < 0.0
	 || source_voxel_position.x >= 1.0 || source_voxel_position.y >= 1.0 || source_voxel_position.z >= 1.0)
	 return vec4(0.0);

	vec3  voxel_color =     color_convert_rgbm_rgb(textureLod(s_voxel_colors_filtered, source_voxel_position, color_lod));
	float voxel_occupancy = textureLod(s_voxel_occupancy_filtered, source_voxel_position, occupancy_lod).r;

	return vec4(voxel_color, voxel_occupancy);
}

// basic template based on the shadertoy framework template

float sdBox(vec3 p, vec3 b)
{
	vec3 q = abs(p) - b;
	return length(max(q,0.0)) + min(max(q.x,max(q.y,q.z)),0.0);
}

float sdTorus( vec3 p, vec2 t )
{
	vec2 q = vec2(length(p.xz) - t.x,p.y);
	return length(q)-t.y;
}

float raymarchBBox(vec3 ro, vec3 rd, vec3 function_scale)
{
	const float maxd = 20000.0;        // max trace distance
	const float precis = 0.001;        // precission of the intersection
	float h = precis*2.0;
	float t = 0.0;
	float res = -1.0;
	for(int i = 0; i < 48; i++)
	{
		if(h<precis || t > maxd) break;

		// yeah, this probably sucks;)
		h = max(sdBox(ro + rd * t, vec3(1000.0)), -sdBox(ro + rd * t, function_scale * 0.5 + 0.01));
		t += h;
	}

	if( t<maxd )
		res = t;

	return res;
}

#ifndef MARCHINGITERATIONS
#define MARCHINGITERATIONS 64
#endif

#if 1
// transparent and infinite object. based on shader from:
// https://www.shadertoy.com/view/lssGRX
// Created by Hazel Quantock - 2013
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

float noise( in vec3 x )
{
	vec3 p = floor(x);
	vec3 f = fract(x);
	f = f*f*(3.0-2.0*f);
	
	// there's an artefact because the y channel almost, but not exactly, matches the r channel shifted (37,17)
	// this artefact doesn't seem to show up in chrome, so I suspect firefox uses different texture compression.
	vec2 uv = (p.xy+vec2(37.0,17.0)*p.z) + f.xy;
	vec2 rg = textureLod(s_NoiseRGBA, (uv+ 0.5)/256.0, 0.0 ).yx;
	return mix( rg.x, rg.y, f.z );
}

vec2 density_map(vec3 p, vec3 p_noise)
{
	//return vec2(1.0, 0.0);
	float t = texture(sImage1, fract(p.xz / raymarch_params.function_scale.xy + 0.5)).g;
	float b = 1.0 - clamp(abs(0.5 - t) * 2.0, 0.0, 1.0);
	float n = noise(vec3(p_noise * 3.1341));

	// making it > 1.0 will amplify noise in the gradation area
	return vec2(t, b * t * clamp(n * 4.0, -0.5, 4.0));
}

vec4 map(vec3 p, vec3 world_p)
{
	vec4 voxel_color_occupancy = sample_voxelization_data(world_p, raymarch_params.sampling_mip, raymarch_params.sampling_mip);
	if (voxel_color_occupancy.a < 0.001)
		return vec4(0.0);

	//float den = 1.0;//-1.0 - (abs(p.y / (raymarch_params.clouds_height * 0.1) - 0.5)+0.5)/2.0;
	float den = -2.0;

	// clouds
	float cloud_time = raymarch_params.animation_time;
	vec3 q = p*.5                       - vec3(0.0,0.0,1.5) * cloud_time + vec3(-0.4,0.0,1.5) * cloud_time * 0.7031;

	vec2 texture_base_den = density_map(p, q);

	float f;
	f  = 0.250000*noise( q * 13.3127 );

	f += 0.50000*noise( q ); q = q*2.02 - vec3(0.0,0.0,1.4) * cloud_time;
	f += 0.25000*noise( q ); q = q*2.03 - vec3(0.0,0.0,1.6) * cloud_time;
	f += 0.12500*noise( q ); q = q*2.01 - vec3(0.0,0.0,1.8) * cloud_time;
	f += 0.06250*noise( q ); q = q*2.02 - vec3(0.0,0.0,1.0) * cloud_time;
	f += 0.03125*noise( q );

	den = den + 4.0 * f;
	//den *= (texture_base_den.x + texture_base_den.y);
	den = clamp(den, 0.0, 1.0);

	{
		//return vec4(1.0, 1.0, 1.0, den);
	}

	vec3 col = mix( vec3(1.0, 1.0, 1.0), vec3(0.6,0.5,0.4), den*.5);
	vec4 color_density = vec4( col, den*.7 * raymarch_params.density * 0.001 );

	color_density.a *= voxel_color_occupancy.a;

	return color_density;
}

vec4 map_simple( vec3 p )
{
	float den = 1.0;//-1.0 - (abs(p.y / (raymarch_params.clouds_height * 0.1) - 0.5)+0.5)/2.0;

	// clouds
	float f;
	float cloud_time = raymarch_params.animation_time;
	vec3 q = p*.5                       - vec3(0.0,0.0,1.5) * cloud_time + vec3(-0.4,0.0,1.5) * cloud_time * 0.7031;

	vec2 texture_base_den = density_map(p, q);

	f  = 0.50000*noise( q ); q = q*2.02 - vec3(0.0,0.0,1.4) * cloud_time;
	f += 0.25000*noise( q ); q = q*2.03 - vec3(0.0,0.0,1.6) * cloud_time;
	f += 0.12500*noise( q ); q = q*2.01 - vec3(0.0,0.0,1.8) * cloud_time;

	den = den + 4.0*f;
	//den *= (texture_base_den.x + texture_base_den.y);
	den = clamp(den, 0.0, 1.0);

	vec3 col = mix( vec3(1.0, 1.0, 1.0), vec3(0.6,0.5,0.4), den*.5);// + 0.05*sin(p);
	vec4 color_density = vec4( col, den*.7 * raymarch_params.density * 0.001 );

	return color_density;
}

#ifdef BUILTIN_LIGHTING
const vec3 sunDir = vec3(-1,0.5,0.5);
#else
#ifndef SPIRV_ENABLED
const vec3 sunDir = -normalize(lights.light_properties[0].direction.xyz);
#else
#define sunDir (normalize(lights.light_properties[0].direction.xyz))
#endif
#endif

float testshadow( vec3 p, float dither )
{
	float shadow = 1.0;
	float s = .05 * dither;
	for ( int j=0; j < 5; j++ )
	{
		vec3 shadpos = p + s*sunDir;

		vec3 world_pos = shadpos;
		world_pos -= raymarch_params.function_origin;
		world_pos /= raymarch_params.function_scale;
		world_pos = (transform_params.mModel * vec4(world_pos, 1.0)).xyz;

		shadow = shadow - map(shadpos, world_pos).a * shadow;
		
		s += .075;
	}
	return shadow;
}

// NOTE: This should be simplified!
float calculate_scene_light(vec3 world)
{
	float total_light = 0.0;
	int lights_num = 1;
	for (int light_idx = 0; light_idx < lights_num; light_idx++)
	{
		LightProperties light = lights.light_properties[light_idx];

		float shadow = 0.0;
		{
			bool calculate_shadows = (light.type & LightType_Shadowcasting) != 0;
			if ((light.type & (LightType_Spot | LightType_Directional)) != 0)
			{
				vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world.xyz, 1.0);

				float in_frustum = 0.0;
				shadow = sampleShadowPCF(
					LightShadowmapCmpSamplers[light.shadowmap_sampler0],
					vShadowCoords,
					in_frustum,
					1,
					materials.material_properties[materialIndex].shadowmap_bias);	// was bias = 0.0001
			}
		}

		shadow = 1.0 - shadow;
		if ((light.type & (LightType_Spot | LightType_Attenuating)) == (LightType_Spot | LightType_Attenuating))
		//if ((light.type & (LightType_Spot)) == (LightType_Spot))
		{
			float attenuation_color = light_calculate_spot_attenuation(light, world.xyz);
			shadow *= length(attenuation_color);	// simplified
		}

		total_light += shadow * light.intensity;
	}

	return total_light / float(lights_num);
}

float cubic(float y0, float y1, float y2, float y3, float mu)
{
   float a0, a1, a2, a3, mu2;

   mu2 = mu*mu;
   a0 = y3 - y2 - y0 + y1;
   a1 = y0 - y1 - a0;
   a2 = y2 - y0;
   a3 = y1;

   return a0 * mu * mu2 + a1 * mu2 + a2 * mu + a3;
}

vec3 cubic3(vec3 y0, vec3 y1, vec3 y2, vec3 y3, float mu)
{
   vec3  a0, a1, a2, a3;
   float mu2;

   mu2 = mu*mu;
   a0 = y3 - y2 - y0 + y1;
   a1 = y0 - y1 - a0;
   a2 = y2 - y0;
   a3 = y1;

   return a0 *mu *mu2 + a1 * mu2 + a2 * mu + a3;
}

vec4 raymarch( in vec3 ro, in vec3 rd, float max_t, out float intersection )
{
	vec4 sum = vec4( 0 );
	
	float t = 0.0;
	intersection = -1.0;

	// dithering	
	float t_step = 0.1;
	float dither = texelFetch(s_BlueNoise, ivec3(uvec2(gl_FragCoord.xy) % uvec2(128), 0), 0).r;
	t += raymarch_params.dithering * dither;
	
	float total_scene_light = 0.0;
	float shadow = 0.0;

	// pre-trace into the voxel fiels using low res mip

	for(int i = 0; i < 64; i++)
	{
		float trace_t = t +.2 * t * t;
		if (trace_t >= max_t) break;

		vec3 pos = ro + trace_t * rd;

		vec3 world_pos = pos;
		world_pos -= raymarch_params.function_origin;
		world_pos /= raymarch_params.function_scale;
		//world_pos = (transform_params.mModel * vec4(world_pos + dither * 0.02, 1.0)).xyz;	// NOTE: this dithering is a huge hack, but for now will do
		world_pos = vector_transform_by_mat43(world_pos, transform_params.mModel).xyz;

		vec4 voxel_color_occupancy = sample_voxelization_data(world_pos, raymarch_params.sampling_mip, raymarch_params.sampling_mip);
		if (voxel_color_occupancy.a > 0.01)
			break;

		t += t_step * raymarch_params.prestep;
	}

	// actual tracing
	for(int i = 0; i < MARCHINGITERATIONS; i++)
	{
		float trace_t = t +.2 * t * t;
		if (sum.a > 0.99 || trace_t >= max_t) break;
		
		vec3 pos = ro + trace_t * rd;

		vec3 world_pos = pos;
		world_pos -= raymarch_params.function_origin;
		world_pos /= raymarch_params.function_scale;
		//world_pos = (transform_params.mModel * vec4(world_pos + dither * 0.02, 1.0)).xyz;	// NOTE: this dithering is a huge hack, but for now will do
		world_pos = vector_transform_by_mat43(world_pos, transform_params.mModel).xyz;

		vec4 col = map(pos, world_pos);

		// this is huge hack
		#if 0
		if (i > 16)
		{
			float prev_shadow = shadow;
			if ((i & 3) == 0)
				shadow = testshadow(pos, dither);
			shadow = mix(prev_shadow, shadow, 0.75);
		}
		else
		{
			shadow = testshadow(pos, dither);
		}
		#else
		shadow = testshadow(pos, dither);
		#endif

		float scene_light = 1.0;
#ifndef BUILTIN_LIGHTING
		if (true)
		{
			vec3 world_pos = pos;
			world_pos -= raymarch_params.function_origin;
			world_pos /= raymarch_params.function_scale;
			world_pos = vector_transform_by_mat43(world_pos + dither * 0.02, transform_params.mModel).xyz;// NOTE: this dithering is a huge hack, but for now will do
			scene_light = calculate_scene_light(world_pos) * 0.01;
			total_scene_light += scene_light * col.a;
		}
#else
		if (true)
		{
		}
#endif

		#if 1
		
		vec3 cloud_color_for_shadow = vec3(1.0);
		if (raymarch_params.gradient_idx != -1)
			cloud_color_for_shadow = gradient_sample(raymarch_params.gradient_idx,  shadow).rgb;

		col.xyz *= cloud_color_for_shadow;

		#else

		col.xyz *= mix( vec3(0.2,0.27,0.4), vec3(0.7), shadow);

		#endif

		//col.xyz =  vec3(shadow);
		col.xyz *= (scene_light + 0.1);

		col.rgb *= col.a;

		sum.rgb += col.rgb * (1.0 - sum.a);
		sum.a    = min(sum.a + col.a, 1.0);

		t += t_step;
	}

	vec4 bg = mix( vec4(.3,.4,.5,0), vec4(.5,.7,1,0), smoothstep(-.4,.0,rd.y) ); // sky/ocean

	/*// floor
	if ( rd.y < -.2 )
	{
		vec3 pos = ro + rd*(ro.y+1.0)/(-rd.y);
		
		float shadow = testshadow(pos+sunDir/sunDir.y, dither);
		bg.xyz = mix( vec3(0), vec3(.5), shadow*.8+.2 );
	}*/

	sum.rgb += bg.rgb * (1.0 - sum.a) * (total_scene_light * 0.05);

	//intersection = t;
	intersection = t + .2 * t * t;
	return vec4(max(vec3(0.0), sum.xyz), sum.a);
}

float calcIntersection(in vec3 ro, in vec3 rd, float max_t, out vec4 color)
{
	float intersection;
	color.rgba = raymarch(ro, rd, max_t, intersection);

	return intersection;
}

vec3 doModelNormal(vec3 p)
{
	return vec3(1.0);
}

#endif

#ifdef RAYMARCH_OUTPUT_DEPTH
layout (depth_less) out float gl_FragDepth;
//out float gl_FragDepth;
#endif

mat3 transpose_mat3(mat3 matrix)
{
    vec3 row0 = matrix[0];
    vec3 row1 = matrix[1];
    vec3 row2 = matrix[2];
    mat3 result = mat3(
        vec3(row0.x, row1.x, row2.x),
        vec3(row0.y, row1.y, row2.y),
        vec3(row0.z, row1.z, row2.z)
    );
    return result;
}

float det(mat2 matrix) {
    return matrix[0].x * matrix[1].y - matrix[0].y * matrix[1].x;
}

mat3 inverse_mat3(mat3 matrix)
{
    vec3 row0 = matrix[0];
    vec3 row1 = matrix[1];
    vec3 row2 = matrix[2];

    vec3 minors0 = vec3(
        det(mat2(row1.y, row1.z, row2.y, row2.z)),
        det(mat2(row1.z, row1.x, row2.z, row2.x)),
        det(mat2(row1.x, row1.y, row2.x, row2.y))
    );
    vec3 minors1 = vec3(
        det(mat2(row2.y, row2.z, row0.y, row0.z)),
        det(mat2(row2.z, row2.x, row0.z, row0.x)),
        det(mat2(row2.x, row2.y, row0.x, row0.y))
    );
    vec3 minors2 = vec3(
        det(mat2(row0.y, row0.z, row1.y, row1.z)),
        det(mat2(row0.z, row0.x, row1.z, row1.x)),
        det(mat2(row0.x, row0.y, row1.x, row1.y))
    );

    mat3 adj = transpose_mat3(mat3(minors0, minors1, minors2));

    return (1.0 / dot(row0, minors0)) * adj;
}



void main() 
{
#ifdef MATERIAL_PROPERTIES_BINDING
	MaterialPropertiesGPU material = materials.material_properties[materialIndex];
#else
	MaterialPropertiesGPU material;
	material.diffuse = colorDiffuse.rgb;
	material.emmisive = vec3(0.0f);
	material.metalness = 0.0f;
	material.roughness = 0.5f;
	material.transparency = 0.0f;
	material.refraction = 0.0f;
	material.flags = 0;
#endif

#ifndef DEFERRED_PASS
	vec4 outAlbedo = vec4(1.0);	// this is dummy, will be optimized out
#endif

	// NOTE: Whole instancing support is pretty expensive, but maybe it doesn't matter when the marching code is going to cost 10x of it?
	mat4 mat_instance_model;
	mat4 mat_instance_model_inv;
	mat_instance_model[0] = vec4(1.0, 0.0, 0.0, 0.0);
	mat_instance_model[1] = vec4(0.0, 1.0, 0.0, 0.0);
	mat_instance_model[2] = vec4(0.0, 0.0, 1.0, 0.0);
	mat_instance_model[3] = vec4(0.0, 0.0, 0.0, 1.0);

	if (instance_params.stride > 0)
	{
		vec4 inst_m0 = instance_transform[instanceID * instance_params.stride + 0];
		vec4 inst_m1 = instance_transform[instanceID * instance_params.stride + 1];
		vec4 inst_m2 = instance_transform[instanceID * instance_params.stride + 2];

		mat_instance_model[0].xyz = vec3(inst_m0.x, inst_m1.x, inst_m2.x);
		mat_instance_model[1].xyz = vec3(inst_m0.y, inst_m1.y, inst_m2.y);
		mat_instance_model[2].xyz = vec3(inst_m0.z, inst_m1.z, inst_m2.z);
		mat_instance_model[3].xyz = vec3(inst_m0.w, inst_m1.w, inst_m2.w);
	}

	// if we want to have each instance to be selfcontained. in case of continuous function this can be left out
	//vec3 instance_function_origin = -(mat_instance_model * vec4(raymarch_params.function_origin, 1.0)).xyz;
	vec3 instance_function_origin = raymarch_params.function_origin;
	if (raymarch_params.use_instance_origin != 0)
	{
		instance_function_origin = (mat_instance_model * vec4(raymarch_params.function_origin, 1.0)).xyz * 0.001;
	}
	//instance_function_origin += raymarch_params.function_origin;

	mat_instance_model = transform_params.mModel * mat_instance_model;
	mat_instance_model_inv = mat_instance_model;
	{
		mat3 inv = inverse_mat3(mat3(mat_instance_model_inv));
		//mat3 inv = mat3(transform_params.mModelInv);
		mat_instance_model_inv[0].xyz = inv[0].xyz;
		mat_instance_model_inv[1].xyz = inv[1].xyz;
		mat_instance_model_inv[2].xyz = inv[2].xyz;
		//mat_instance_model_inv = transform_params.mModelInv;
		mat_instance_model_inv[3].x = -(inv[0].x * mat_instance_model[3].x + inv[1].x * mat_instance_model[3].y + inv[2].x * mat_instance_model[3].z);
		mat_instance_model_inv[3].y = -(inv[0].y * mat_instance_model[3].x + inv[1].y * mat_instance_model[3].y + inv[2].y * mat_instance_model[3].z);
		mat_instance_model_inv[3].z = -(inv[0].z * mat_instance_model[3].x + inv[1].z * mat_instance_model[3].y + inv[2].z * mat_instance_model[3].z);
	}

	float g = vtx_input.vNorm.z * 0.5 + 0.5;

#ifdef DEFERRED_PASS
	outMetalnessRoughnessMeterialTags.rg = encode_metalness_roughness_material(material.metalness, material.roughness, materialIndex);

	outEmissive.rgba = vec4(0.0);
#endif

	// NOTE: when rendering to the shadowmap we should be rendering backfaces
	//       so maybe just mirror the camera and render 'back' towards the real one
	#if 0
	vec3 ro = (transform_params.mModelInv * vec4(vtx_input.vCameraRelativeWorldPos.xyz + transform_params.vCameraPosition.xyz, 1.0)).xyz; // start tracing at the cube surface. still should clamp at the "outgoing face"
	vec3 rd = (transform_params.mModelInv * vec4(transform_params.vCameraPosition, 1.0)).xyz;
	#else
	vec3 ro = (mat_instance_model_inv * vec4(vtx_input.vCameraRelativeWorldPos.xyz + transform_params.vCameraPosition.xyz, 1.0)).xyz; // start tracing at the cube surface. still should clamp at the "outgoing face"
	vec3 rd = (mat_instance_model_inv * vec4(transform_params.vCameraPosition, 1.0)).xyz;
	#endif
	rd = normalize(ro - rd);
	if (raymarch_params.trace_inside != 0)
	{
		ro = (mat_instance_model_inv * vec4(transform_params.vCameraPosition, 1.0)).xyz;
	}

	ro *= raymarch_params.function_scale;
	vec3 ro_bounding_volume = ro;
	ro += instance_function_origin;

	float raw_depth = texelFetch(sDepth, int2(gl_FragCoord.xy), 0).r;
	float max_depth = linearizeDepth(raw_depth);

	vec4 color = vec4(1.0);
	float bounding_volume_intersection = 100000.0;
	if (raymarch_params.clamp_to_volume != 0)
		bounding_volume_intersection = raymarchBBox(ro_bounding_volume , rd, raymarch_params.function_scale);	// NOTE: yeah, we could just intersect analytically, but i want to have possibility to use custom shapes here

	// ok, this is a HACK!
	bounding_volume_intersection = max_depth;
	//bounding_volume_intersection = min((max_depth - vtx_input.vCoords.z) / (raymarch_params.intersection_density_modifier * raymarch_params.function_scale.x), bounding_volume_intersection);

	//outAlbedo.rgba = vec4((max_depth - vtx_input.vCoords.z) / 1000.0);
	//outAlbedo.rgba = vec4(bounding_volume_intersection * 3.0 / 1000.0);
	//return;

	{
		vec3 view_direction = get_view_direction(vec2(gl_FragCoord.xy));
		vec3 world          = (transform_params.mViewInv * vec4(view_direction * max_depth, 1.0)).xyz;
		float d             = length((transform_params.mModelInv * vec4(world - transform_params.vCameraPosition.xyz, 0.0)).xyz);
		//float d             = length(world - transform_params.vCameraPosition.xyz);

		// need to scale by model scale / function scale. this is approx
		//float s = transform_
		//bounding_volume_intersection = d / (3000.0 / 10.0);
		bounding_volume_intersection = d * raymarch_params.function_scale.z;
	}

	float intersection = calcIntersection(ro, rd, bounding_volume_intersection, color);
	if (intersection > 0.0)
	{
		vec3 ri = ro + intersection * rd;

#ifdef DEFERRED_PASS
		vec3 normal = doModelNormal(ri);
		vec3 world_normal = (mat_instance_model * vec4(normal, 0.0)).xyz;
		outNormalMaterial = encode_normal_material(normalize(world_normal), materialId);
#endif

		ri -= instance_function_origin;
		ri /= raymarch_params.function_scale;

		ri = (transform_params.mView * mat_instance_model * vec4(ri, 1.0)).xyz;

		// scale alpha value by distance to closest currently rendered object.
		{
			//float absorption = (max_depth - vtx_input.vCoords.z) / 1000.0;
			//absorption = clamp(absorption, 0.0, 1.0);
			//color.a = absorption;
		}

		outAlbedo.rgb = color.rgb * vec3(0, 50, 30) * raymarch_params.intensity;
		outAlbedo.a   = clamp(1.0 - (1.0 - exp(-color.a) * raymarch_params.opacity), 0.0, 1.0);
		
		//outAlbedo.a = pow(outAlbedo.a, 3.0);
		//outAlbedo.rgb = pow(outAlbedo.rgb, vec3(2.0));
		//outAlbedo.rgb = vec3(ri.z * 0.0001);
		//outAlbedo.rgb = vec3(color.a);
		//outAlbedo.rgb = vec3(0.1);

		// NOTE: These two should (roughly) match
		//outAlbedo.rgb = vec3(bounding_volume_intersection * 0.01);
		//outAlbedo.rgb = vec3(intersection * 0.01);

		{
			vec3 view_direction = get_view_direction(vec2(gl_FragCoord.xy));
			vec3 world          = (transform_params.mViewInv * vec4(view_direction * max_depth, 1.0)).xyz;
			float d             = length(transform_params.vCameraPosition.xyz - world);

			//outAlbedo.xyz = length(world - transform_params.vCameraPosition.xyz).xxx * 0.0001;
		}

	
#ifdef RAYMARCH_OUTPUT_DEPTH
		{
			float near = -1.0;	// this is depth range, not the projection
			float far  =  1.0;
			float depth = (transform_params.mProjection * vec4(ri, 1.0)).z / (transform_params.mProjection * vec4(ri, 1.0)).w;
			depth = (((far - near) * depth) + near + far) / 2.0;
			gl_FragDepth = depth;
		}
#endif
	}
	else
	{
		if (raymarch_params.show_bounding_box != 0)
		{
			if (intersection == -1.0)
				outAlbedo = vec4(0.4, 0.2, 0.0, 1.0);
			else
				outAlbedo = vec4(5.0, 0.0, 0.0, 1.0);

			outAlbedo.rgb = fract(vtx_input.vCoords.xyz * 0.01);
		}
		else
		{
			//outAlbedo.rgb = vec3(1.0, 0.0, 0.0);
			outAlbedo.a = 1.0;
			discard;
		}
	}
}
