//
// I2S Audio driver (for UDA1334A)
// Modified implementation, based on pico-extras/pico-audio, Copyright (c) 2020 Raspberry Pi (Trading) Ltd.
//

#include <stdio.h>

#include "audio_i2s.h"
#include "audio_i2s.pio.h"
#include "hardware/pio.h"
#include "hardware/gpio.h"
#include "hardware/dma.h"
#include "hardware/irq.h"
#include "hardware/clocks.h"

CU_REGISTER_DEBUG_PINS(audio_timing)

#define i2s_dma_configure_size DMA_SIZE_32

#define audio_pio       __CONCAT(pio, PICO_AUDIO_I2S_PIO)
#define GPIO_FUNC_PIOx  __CONCAT(GPIO_FUNC_PIO, PICO_AUDIO_I2S_PIO)
#define DREQ_PIOx_TX0   __CONCAT(__CONCAT(DREQ_PIO, PICO_AUDIO_I2S_PIO), _TX0)

#define AUDIO_STATIC_BUFFER_SIZE_CONSUMER 256

struct
{
	uint32_t freq;
	uint8_t pio_sm;
	uint8_t dma_channel;
} shared_state;

static audio_format_t pio_i2s_consumer_format;
static audio_buffer_format_t pio_i2s_consumer_buffer_format =
{
		.format = &pio_i2s_consumer_format,
};

static struct buffer_copying_on_consumer_call_connection m2s_audio_i2s_ct_connection;
static bool audio_enabled;

static void __isr __time_critical_func(audio_i2s_dma_irq_handler)();

const audio_format_t *audio_i2s_setup(const audio_format_t *intended_audio_format, const audio_i2s_config_t *config)
{
	uint func = GPIO_FUNC_PIOx;

	gpio_set_function(config->data_pin, func);
	gpio_set_function(config->clock_pin_base, func);
	gpio_set_function(config->clock_pin_base + 1, func);

	#if PICO_PIO_USE_GPIO_BASE
	if (config->data_pin >= 32 || config->clock_pin_base + 1 >= 32)
	{
		assert(config->data_pin >= 16 && config->clock_pin_base >= 16);
		pio_set_gpio_base(audio_pio, 16);
	}
	#endif

	uint8_t sm = shared_state.pio_sm = config->pio_sm;
	pio_sm_claim(audio_pio, sm);

	const struct pio_program *program =
		#if PICO_AUDIO_I2S_CLOCK_PINS_SWAPPED
		& audio_i2s_swapped_program
		#else
		& audio_i2s_program
		#endif
		;

	uint offset = pio_add_program(audio_pio, program);
				  audio_i2s_program_init(audio_pio, sm, offset, config->data_pin, config->clock_pin_base);
	__mem_fence_release();

	uint8_t dma_channel = config->dma_channel;

	// Automatic asignment if 0xFF, claim channel by index in any other case
	if (dma_channel == 0xFF)
		dma_channel = dma_claim_unused_channel(true);
	else
		dma_channel_claim(dma_channel);

	shared_state.dma_channel = dma_channel;

	dma_channel_config dma_config = dma_channel_get_default_config(dma_channel);

	channel_config_set_dreq(&dma_config, DREQ_PIOx_TX0 + sm);

	channel_config_set_transfer_data_size(&dma_config, i2s_dma_configure_size);

	dma_channel_configure(dma_channel,
						  &dma_config,
						  &audio_pio->txf[sm],  // dest
						  NULL,                 // src
						  0,                    // count
						  false);               // trigger

	irq_add_shared_handler(DMA_IRQ_0 + PICO_AUDIO_I2S_DMA_IRQ, audio_i2s_dma_irq_handler, PICO_SHARED_IRQ_HANDLER_DEFAULT_ORDER_PRIORITY);
	dma_irqn_set_channel_enabled(PICO_AUDIO_I2S_DMA_IRQ, dma_channel, 1);

	return intended_audio_format;
}

void audio_i2s_cleanup()
{
	if (dma_channel_is_claimed(shared_state.dma_channel)) 
	{
		dma_channel_abort(shared_state.dma_channel);
		dma_channel_unclaim(shared_state.dma_channel);
		shared_state.dma_channel = 0xFF;
	}

	return;
}

static void update_pio_frequency(uint32_t sample_freq)
{
	uint32_t system_clock_frequency = clock_get_hz(clk_sys);
	assert(system_clock_frequency < 0x40000000);
	uint32_t divider = system_clock_frequency * 4 / sample_freq; // avoid arithmetic overflow
	assert(divider < 0x1000000);
	pio_sm_set_clkdiv_int_frac(audio_pio, shared_state.pio_sm, divider >> 8u, divider & 0xffu);
	shared_state.freq = sample_freq;
}

bool audio_i2s_connect(void (*producer)(void *, uint), uint32_t freq)
{
	pio_i2s_consumer_format.format = AUDIO_BUFFER_FORMAT_PCM_S16;
	pio_i2s_consumer_format.sample_freq = freq;
	pio_i2s_consumer_format.channel_count = 2;
	pio_i2s_consumer_buffer_format.sample_stride = 4;

	update_pio_frequency(freq);

	__mem_fence_release();

	audio_connection_t *connection = &m2s_audio_i2s_ct_connection.core;
						connection->producer_call = producer;

	return true;
}

static inline void audio_start_dma_transfer()
{
	static uint16_t ScrapBuffer[2][AUDIO_STATIC_BUFFER_SIZE_CONSUMER * 2];
	static uint8_t  ScrapFlip;

	dma_channel_config c = dma_get_channel_config(shared_state.dma_channel);
	channel_config_set_read_increment(&c, true);
	dma_channel_set_config(shared_state.dma_channel, &c, false);
	dma_channel_transfer_from_buffer_now(shared_state.dma_channel, ScrapBuffer[ScrapFlip], AUDIO_STATIC_BUFFER_SIZE_CONSUMER);

	ScrapFlip ^= 1;

	audio_connection_t *connection = &m2s_audio_i2s_ct_connection.core;
	if (connection->producer_call)
		connection->producer_call(ScrapBuffer[ScrapFlip], AUDIO_STATIC_BUFFER_SIZE_CONSUMER * 2 * sizeof(int16_t));

	return;
}

void __isr __time_critical_func(audio_i2s_dma_irq_handler)()
{
	uint dma_channel = shared_state.dma_channel;
	if (dma_irqn_get_channel_status(PICO_AUDIO_I2S_DMA_IRQ, dma_channel))
	{
		dma_irqn_acknowledge_channel(PICO_AUDIO_I2S_DMA_IRQ, dma_channel);
		DEBUG_PINS_SET(audio_timing, 4);
		audio_start_dma_transfer();
		DEBUG_PINS_CLR(audio_timing, 4);
	}
}

void audio_i2s_set_enabled(bool enabled)
{
	if (enabled != audio_enabled)
	{
		irq_set_enabled(DMA_IRQ_0 + PICO_AUDIO_I2S_DMA_IRQ, enabled);

		if (enabled)
			audio_start_dma_transfer();

		pio_sm_set_enabled(audio_pio, shared_state.pio_sm, enabled);

		audio_enabled = enabled;
	}

	return;
}
