//
// I2S Audio driver (for UDA1334A)
// Modified implementation, based on pico-extras/pico-audio, Copyright (c) 2020 Raspberry Pi (Trading) Ltd.
//

#include "pico.h"
#include "hardware/sync.h"

#ifndef _PICO_AUDIO_I2S_H
#define _PICO_AUDIO_I2S_H

//#include "audio.h"

/** \file audio_i2s.h
 *  \defgroup pico_audio_i2s pico_audio_i2s
 *  I2S audio output using the PIO
 *
 * This library uses the \ref hardware_pio system to implement a I2S audio interface
 *
 */

#ifdef __cplusplus
extern "C" {
#endif

#define AUDIO_BUFFER_FORMAT_PCM_S16 1 ///< signed 16bit PCM (this is only supported format for now)

#ifndef PICO_AUDIO_I2S_DMA_IRQ
#ifdef PICO_AUDIO_DMA_IRQ
#define PICO_AUDIO_I2S_DMA_IRQ PICO_AUDIO_DMA_IRQ
#else
#define PICO_AUDIO_I2S_DMA_IRQ 0
#endif
#endif

#ifndef PICO_AUDIO_I2S_PIO
#ifdef PICO_AUDIO_PIO
#define PICO_AUDIO_I2S_PIO PICO_AUDIO_PIO
#else
#define PICO_AUDIO_I2S_PIO 0
#endif
#endif

#if !(PICO_AUDIO_I2S_DMA_IRQ == 0 || PICO_AUDIO_I2S_DMA_IRQ == 1)
#error PICO_AUDIO_I2S_DMA_IRQ must be 0 or 1
#endif

#if !(PICO_AUDIO_I2S_PIO == 0 || PICO_AUDIO_I2S_PIO == 1)
#error PICO_AUDIO_I2S_PIO ust be 0 or 1
#endif

// The default order is CLOCK_PIN_BASE=LRCLK, CLOCK_PIN_BASE+1=BCLK
// The swapped order is CLOCK_PIN_BASE=BCLK,  CLOCK_PIN_BASE+1=LRCLK
#ifndef PICO_AUDIO_I2S_CLOCK_PINS_SWAPPED
#define PICO_AUDIO_I2S_CLOCK_PINS_SWAPPED 0
#endif

/** \brief Audio format definition
 */
typedef struct audio_format 
{
    uint32_t sample_freq;               ///< Sample frequency in Hz
    uint16_t format;                    ///< Audio format \ref audio_formats
    uint16_t channel_count;             ///< Number of channels
} audio_format_t;

/** \brief Audio buffer format definition
 */
typedef struct audio_buffer_format 
{
    const audio_format_t *format;      ///< Audio format
    uint16_t sample_stride;            ///< Sample stride
} audio_buffer_format_t;

typedef struct audio_connection audio_connection_t;

struct audio_connection 
{
    void (*producer_call)(void *, uint);
};

struct buffer_copying_on_consumer_call_connection 
{
    struct audio_connection core;
};

/** \brief Base configuration structure used when setting up
 * \ingroup pico_audio_i2s
 */
typedef struct audio_i2s_config 
{
    uint8_t data_pin;
    uint8_t clock_pin_base;
    uint8_t dma_channel;
    uint8_t pio_sm;
} audio_i2s_config_t;

/** \brief Set up system to output I2S audio
 * \ingroup pico_audio_i2s
 *
 * \param intended_audio_format
 * \param config The configuration to apply.
 */
const audio_format_t *audio_i2s_setup(const audio_format_t *intended_audio_format, const audio_i2s_config_t *config);

/**
 * \ingroup pico_audio_i2s
 *
 * \param producer
 * \param buffer_on_give
 * \param buffer_count
 * \param samples_per_buffer
 * \param connection
 * \return
 */
bool audio_i2s_connect(void (*producer)(void *, uint), uint32_t freq);

/** \brief Set up system to output I2S audio
 * \ingroup pico_audio_i2s
 *
 * \param enable true to enable I2S audio, false to disable.
 */
void audio_i2s_set_enabled(bool enabled);


/** \brief Shut down audio and do cleanup
 * \ingroup pico_audio_i2s
 * 
 */
void audio_i2s_cleanup();

#ifdef __cplusplus
}
#endif

#endif //_AUDIO_I2S_H
