// (c) MX^Add

#define _CRT_SECURE_NO_WARNINGS

#include "SDLMain.h"
#include "DemoMain.h"
#include "Renderer/FrameBuffer.h"
#include "ModPlayer/ModPlayer.h"
#include "Renderer/PostEfx.h"
#include "Font/BitmapFont.h"
#include "RendererTypes/Scene.h"

#define WIN32_LEAN_AND_MEAN
#define NOGDI
#define NOMINMAX
#include <windows.h>
#include <mmsystem.h>

#pragma comment(lib, "winmm.lib")
#pragma comment(lib, "SDL2.lib")
#pragma comment(lib, "SDL2main.lib")

static uint16 FrontBuffer[RasterizerSizeX * RasterizerSizeY];
static uint16 DebugBuffer[RasterizerSizeX * 20];
static bool   DemoIsPaused;

static constexpr uint32 MusicLenInSeconds = 152;
static constexpr uint32 MusicSampleRate   = 44100;
static constexpr uint32 MusicSeekRate     = 100;
static constexpr uint32 MusicTotalSamples = MusicSampleRate * MusicLenInSeconds;
static sint16           MusicDataBuffer     [(MusicTotalSamples * 2) + 1024];
static HWAVEOUT         MusicWaveOut;

static WAVEFORMATEX WaveFMT =
{
    WAVE_FORMAT_PCM,
    2,                                   // channels
    MusicSampleRate,                     // samples per sec
    MusicSampleRate*sizeof(sint16) * 2,  // bytes per sec
    sizeof(sint16) * 2,                  // block alignment;
    sizeof(sint16) * 8,                  // bits per sample
    0                                    // extension not needed
};

static WAVEHDR WaveHDR =
{
    (LPSTR)MusicDataBuffer, MusicTotalSamples * sizeof(sint16) * 2, 0, 0, WHDR_PREPARED, 0, 0, 0
};

static void MusicInitialize()
{
    MODInitializePlayerData();
    MODFetchSamples(MusicDataBuffer, MusicTotalSamples);
	waveOutOpen(&MusicWaveOut, WAVE_MAPPER, &WaveFMT, NULL, 0, CALLBACK_NULL);
	waveOutWrite(MusicWaveOut, &WaveHDR, sizeof(WaveHDR));

    return;
}

static void MusicDeinitialize()
{
    waveOutClose(MusicWaveOut);
    return;
}

#ifdef DEMO_EDITOR
static uint32      DebugSceneFrame;
static const char *DebugSceneCam;
extern uint32      DebugTrissPass;
extern uint32      DebugTrissRass;

uint32 DebugTrissPass;
uint32 DebugTrissRass;

void __DbgMarker(uint32 f, const char *n)
{
    DebugSceneFrame = f;
    DebugSceneCam   = n;
    return;
}
#endif

static void DoImage(uint32* PhysicalScreen, uint32 Tick)
{
    //
    // Render scene at given frame
    //
    const uint32 PostEfx = DemoDo(Tick);
    DoPostEfx(PostEfx, FrontBuffer);

    #ifdef DEMO_EDITOR
    //
    // Print debugs
    //
    {
        char Buff[128];

        memset(DebugBuffer, 0, RasterizerSizeX * 20 * sizeof(uint16));
        sprintf_s(Buff, 128, "F:%i  T:%3.3f", Tick/33, float(Tick)/1000.0f);
        PrintString(Buff, strlen(Buff), 0, 0, 0xFFFF, DebugBuffer);

        if (DebugSceneCam)
        {
            sprintf_s(Buff, 128, "%s %i", DebugSceneCam, DebugSceneFrame);
            PrintString(Buff, strlen(Buff), 320 - (strlen(Buff) * 8 + 4), 0, 0xFFFF, DebugBuffer);
        }

        {
                  uint32 *Dest = PhysicalScreen;
            const uint32 *Src  = (uint32 *)DebugBuffer;

            for (uint32 i = 0; i < RasterizerSizeX * 20; i += 2, Dest += 2, Src++)
            {
                uint16 c;
                uint32 v = *Src;
                       v = ((v & 0x00FF00FF) << 8) | ((v & 0xFF00FF00) >> 8);

                c       = uint16(v & 0xFFFF);
                Dest[0] = 0xFF000000 | uint32((c >> 11) << 3) << 16 | uint32(((c >> 5) & 0x3F) << 2) << 8 | uint32((c & 0x1F) << 3);

                c       = uint16(v >> 16);
                Dest[1] = 0xFF000000 | uint32((c >> 11) << 3) << 16 | uint32(((c >> 5) & 0x3F) << 2) << 8 | uint32((c & 0x1F) << 3);
            }
        }

        memset(DebugBuffer, 0, RasterizerSizeX * 20 * sizeof(uint16));
        if (DemoIsPaused)
            PrintString("paused", 6, 4, 0, 0xFFFF, DebugBuffer);

        sprintf_s(Buff, 128, "T:%i R:%i", DebugTrissPass, DebugTrissRass);
        PrintString(Buff, strlen(Buff), 320 - (strlen(Buff) * 8 + 4), 0, 0xFFFF, DebugBuffer);

        DebugTrissPass = 0;
        DebugTrissRass = 0;

        sint32 QLen = CLAMP(sint32((double(Tick) / (MusicLenInSeconds * 1000)) * RasterizerSizeX), 0, RasterizerSizeX);

        if (QLen > 0)
        {
            for (sint32 i = 0; i < 4; i++)
                memset(&DebugBuffer[RasterizerSizeX * (16 + i)], 0xFFFFFFFF, QLen * sizeof(uint16));
        }

        {
                  uint32 *Dest = PhysicalScreen + (PhysicalScreenSizeY - 20) * RasterizerSizeX;
            const uint32 *Src  = (uint32 *)DebugBuffer;

            for (uint32 i = 0; i < RasterizerSizeX * 20; i += 2, Dest += 2, Src++)
            {
                uint16 c;
                uint32 v = *Src;
                       v = ((v & 0x00FF00FF) << 8) | ((v & 0xFF00FF00) >> 8);

                c       = uint16(v & 0xFFFF);
                Dest[0] = 0xFF000000 | uint32((c >> 11) << 3) << 16 | uint32(((c >> 5) & 0x3F) << 2) << 8 | uint32((c & 0x1F) << 3);

                c       = uint16(v >> 16);
                Dest[1] = 0xFF000000 | uint32((c >> 11) << 3) << 16 | uint32(((c >> 5) & 0x3F) << 2) << 8 | uint32((c & 0x1F) << 3);
            }
        }
    }
    #endif
    
    //
    // Copy image to physical screen, doing color conversion
    //
          uint32 *Dest = PhysicalScreen + ((PhysicalScreenSizeY - RasterizerSizeY) / 2) * RasterizerSizeX;
    const uint32 *Src  = (uint32 *)FrontBuffer;
   
    for (uint32 i = 0; i < RasterizerSizeX * RasterizerSizeY; i += 2, Dest += 2, Src++)
    {
        uint16 c;
        uint32 v = *Src;
               v = ((v & 0x00FF00FF) << 8) | ((v & 0xFF00FF00) >> 8);

        c       = uint16(v & 0xFFFF);
        Dest[0] = 0xFF000000 | uint32((c >> 11) << 3) << 16 | uint32(((c >> 5) & 0x3F) << 2) << 8 | uint32((c & 0x1F) << 3);

        c       = uint16(v >> 16);
        Dest[1] = 0xFF000000 | uint32((c >> 11) << 3) << 16 | uint32(((c >> 5) & 0x3F) << 2) << 8 | uint32((c & 0x1F) << 3);
    }

    return;
}

#if ((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER)
#include <vector>
#include <string>
#include <map>

class FVisibilityFaceInfo
{
public:

    FVector3D a, b, c;
    uint32    p;
};

class FVisibilityMeshInfo
{
public:

    std::vector<FVisibilityFaceInfo> FaceInfo;
    const FMesh                     *Mesh;
    const void                      *VPtr;
    const uint16                    *FPtr;

    FVisibilityMeshInfo(const FMesh *Msh) : Mesh(Msh), VPtr(nullptr), FPtr(nullptr)
    {
        return;
    }

    ~FVisibilityMeshInfo()
    {
        return;
    }

    void ReportFace(const void *VtxPtr, const uint16 *FcsPtr, uint16 Face, uint32 Pixels)
    {
        if (!VPtr)
        {
            VPtr = VtxPtr;
            FPtr = FcsPtr;
        }

        FVector3D a, b, c;

        const uint16 fa = FcsPtr[Face*3+0];
        const uint16 fb = FcsPtr[Face*3+1];
        const uint16 fc = FcsPtr[Face*3+2];

        switch (Mesh->VertexFormat)
        {
            case FMesh::EVertexFormat::Simple:
                {
                    const FMesh::FVertex *Vtxs = (const FMesh::FVertex *)VtxPtr;

                    a = Vtxs[fa].Pos;
                    b = Vtxs[fb].Pos;
                    c = Vtxs[fc].Pos;
                }
            break;

            case FMesh::EVertexFormat::Color:
                {
                    const FMesh::FVertexColor *Vtxs = (const FMesh::FVertexColor *)VtxPtr;

                    a = Vtxs[fa].Pos;
                    b = Vtxs[fb].Pos;
                    c = Vtxs[fc].Pos;
                }
            break;

            case FMesh::EVertexFormat::UV:
                {
                    const FMesh::FVertexUV *Vtxs = (const FMesh::FVertexUV *)VtxPtr;

                    a = Vtxs[fa].Pos;
                    b = Vtxs[fb].Pos;
                    c = Vtxs[fc].Pos;
                }
            break;
        }

        for (size_t i = 0; i < FaceInfo.size(); i++)
        {
            FVisibilityFaceInfo &face = FaceInfo[i];
            if (face.a == a && face.b == b && face.c == c)
            {
                face.p += Pixels;
                return;
            }
        }

        FVisibilityFaceInfo face;
        face.a = a;
        face.b = b;
        face.c = c;
        face.p = Pixels;

        FaceInfo.push_back(face);

        return;
    }

    void ReportMesh(FILE *f, uint32 &FacesTotal, uint32 &FacesVisible) const
    {
        if (!VPtr)
            return;

        uint32 VisibleFaces = 0;
        uint32 AllFaces     = uint32(FaceInfo.size());
        uint32 VSize        = 0;

        switch (Mesh->VertexFormat)
        {
            case FMesh::EVertexFormat::Simple: VSize = sizeof(FMesh::FVertex);      break;
            case FMesh::EVertexFormat::Color:  VSize = sizeof(FMesh::FVertexColor); break;
            case FMesh::EVertexFormat::UV:     VSize = sizeof(FMesh::FVertexUV);    break;
        }

        // Save whole mesh for identification (is this enough ?)

        fwrite(&Mesh->VertexFormat, 1, 1, f);
        fwrite(&Mesh->NumVetices,   2, 1, f);
        fwrite(&Mesh->NumFaces,     2, 1, f);

        fwrite(FPtr, Mesh->NumFaces   * 2, 1, f);
        fwrite(VPtr, Mesh->NumVetices * VSize, 1, f);

        // Save faces visibility info

        fwrite(&AllFaces, 4, 1, f);

        for (uint32 i = 0; i < uint32(FaceInfo.size()); i++)
        { 
            FacesTotal++;

            fwrite(&i, 4, 1, f);
            fwrite(&FaceInfo[i].a, 12, 1, f);
            fwrite(&FaceInfo[i].b, 12, 1, f);
            fwrite(&FaceInfo[i].c, 12, 1, f);
            fwrite(&FaceInfo[i].p,  4, 1, f);

            if (FaceInfo[i].p)
            {
                FacesVisible++;
                VisibleFaces++;
            }
        }

        // NOTE::The mesh is only for reference there, it's not exact this mesh, but the set of meshes that handle this vertex table, so we need to handle this accordingly when outputing information for mesh identification in exporter.

        printf("   Mesh 0x%04X - Faces: %i/%i\n", uint32(Mesh->VerticesOffset), uint32(FaceInfo.size()), VisibleFaces);

        return;
    }
};

class FVisibilityChecker
{
public:

    std::string                           SceneName;
    std::map<uint64, FVisibilityMeshInfo> MeshInfo;

    FVisibilityChecker()
    {
        return;
    }

    ~FVisibilityChecker()
    {
        return;
    }

    void ReportFace(const FMesh *Mesh, const void *VtxPtr, const uint16 *FcsPtr, uint16 Face, uint32 Pixels)
    {
        auto [iter, inserted] = MeshInfo.try_emplace(uint64(VtxPtr), FVisibilityMeshInfo(Mesh)); // NOTE::Key is the vertex pointer offset as multiple meshes could share that same vertex table, and we want to handle that case.
        iter->second.ReportFace(VtxPtr, FcsPtr, Face, Pixels);
        return;
    }

    void FlushToDisk()
    {
        printf("\n== SCENE [%s] ==\n", SceneName.c_str());
        FILE *f = fopen((SceneName + ".vis").c_str(), "wb");
        uint32 Magic  = 0x5151B116;
        uint16 MInfos = uint16(MeshInfo.size());
        fwrite(&Magic,  4, 1, f);
        fwrite(&MInfos, 2, 1, f);

        uint32 FacesTotal   = 0;
        uint32 FacesVisible = 0;
        for (const auto &item : MeshInfo)
            item.second.ReportMesh(f, FacesTotal, FacesVisible);

        fclose(f);
        printf("== ENDS (%i/%i) ==\n", FacesTotal, FacesVisible);
        return;
    }
};

static const char *VisCurrentSceneName;
static std::vector<std::pair<std::string, FVisibilityChecker>> VisScenes;
extern bool __PauseVisibilityReporting;
bool __PauseVisibilityReporting;

extern void __ReportSceneContext(const char *SceneName);
extern void __ReportVFace(const FMesh *Mesh, const void *VtxPtr, const uint16 *FcsPtr, uint16 Face, uint32 Pixels);

void __ReportSceneContext(const char *SceneName)
{
    VisCurrentSceneName = SceneName;
    return;
}

void __ReportVFace(const FMesh *Mesh, const void *VtxPtr, const uint16 *FcsPtr, uint16 Face, uint32 Pixels)
{
    if (__PauseVisibilityReporting)
        return;

    if (!VisCurrentSceneName)
        return;

    for (size_t i = 0; i < VisScenes.size(); i++)
    {
        if (VisScenes[i].first == VisCurrentSceneName)
        {
            VisScenes[i].second.ReportFace(Mesh, VtxPtr, FcsPtr, Face, Pixels);
            return;
        }
    }

    VisScenes.push_back(std::pair<std::string, FVisibilityChecker>());
    VisScenes[VisScenes.size()-1].first = VisCurrentSceneName;
    VisScenes[VisScenes.size()-1].second.SceneName = VisCurrentSceneName;
    VisScenes[VisScenes.size()-1].second.ReportFace(Mesh, VtxPtr, FcsPtr, Face, Pixels);
    return;
}

static void __ReportFlushToDisk()
{
    for (size_t i = 0; i < VisScenes.size(); i++)
        VisScenes[i].second.FlushToDisk();
    return;
}
#endif

int main(int argc, char ** argv)
{
    SDL_Init(SDL_INIT_VIDEO);

    SDL_Window   *SDLWindow   = SDL_CreateWindow("SDLToy", SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED, PhysicalScreenSizeX*3, PhysicalScreenSizeY*3, 0);
    SDL_Renderer *SDLRenderer = SDL_CreateRenderer(SDLWindow, -1, 0);
    SDL_Texture  *SDLTexture  = SDL_CreateTexture(SDLRenderer, SDL_PIXELFORMAT_ARGB8888, SDL_TEXTUREACCESS_STATIC, PhysicalScreenSizeX, PhysicalScreenSizeY);
    uint32       *SDLPixels   = new uint32[PhysicalScreenSizeX * PhysicalScreenSizeY];
                                memset(SDLPixels, 0, PhysicalScreenSizeX * PhysicalScreenSizeY * sizeof(uint32));

    InitializeFramebuffer();
    DemoInit(nullptr);
    MusicInitialize();

    #if ((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER)
    uint32 TimeStart  = 0;
    #else
    uint32 TimeStart  = SDL_GetTicks();
    #endif
    uint32 PausedTime = 0;
    uint32 TimeOffset = 0;

    while (true)
    {
        #if ((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER)
        const uint32 DemoTime = TimeStart; TimeStart += 33;
        if (TimeStart > MusicLenInSeconds * 1000)
            break;
        #else
        static MMTIME MMTime =
        {
            TIME_SAMPLES, 0
        };

        waveOutGetPosition(MusicWaveOut, &MMTime, sizeof(MMTIME));
        uint32 DemoTime = DemoIsPaused ? PausedTime : TimeOffset + uint32((double(MMTime.u.sample)/MusicSampleRate)*1000.0);
        #endif

        SDL_Event SDLEvent;

        if (SDL_PollEvent(&SDLEvent))
        {
            if (SDLEvent.type == SDL_QUIT)
                break;

            #if (!((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER))
            #ifdef DEMO_EDITOR
            const uint32 SeekRate = ((SDLEvent.key.keysym.mod & KMOD_SHIFT) ? MusicSeekRate * 5 : MusicSeekRate);

            //
            // Bookmarks
            //
            uint32 BookmarkFrame = DemoPartBSBegin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_0)
                BookmarkFrame = DemoPart00Begin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_1)
                BookmarkFrame = DemoPart01Begin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_2)
                BookmarkFrame = DemoPart02Begin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_3)
                BookmarkFrame = DemoPart03Begin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_4)
                BookmarkFrame = DemoPart04Begin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_5)
                BookmarkFrame = DemoPart05Begin;

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_6)
                BookmarkFrame = DemoPart06Begin;

            if (BookmarkFrame)
            {
                if (DemoIsPaused)
                {
                    DemoTime = PausedTime = BookmarkFrame * 33;
                }
                else
                {
                    TimeOffset = BookmarkFrame * 33;

                    uint32 SamplesOffset = uint32((double(TimeOffset)/1000.0) * MusicSampleRate);

                    waveOutReset(MusicWaveOut);

                    WAVEHDR WaveHDRCpy = WaveHDR;

                    WaveHDRCpy.lpData		  += SamplesOffset * sizeof(sint16) * 2;
                    WaveHDRCpy.dwBufferLength -= SamplesOffset * sizeof(sint16) * 2;

                    waveOutWrite(MusicWaveOut, &WaveHDRCpy, sizeof(WaveHDRCpy));
                }
            }

            //
            // Pause/scrubbing
            //
            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_p)
            {
                if (DemoIsPaused)
                {
                    DemoIsPaused = false;
                    DemoTime     = PausedTime;
                    TimeOffset   = PausedTime;

                    uint32 SamplesOffset = uint32((double(DemoTime)/1000.0) * MusicSampleRate);
                    WAVEHDR WaveHDRCpy   = WaveHDR;

                    WaveHDRCpy.lpData		  += SamplesOffset * sizeof(sint16) * 2;
                    WaveHDRCpy.dwBufferLength -= SamplesOffset * sizeof(sint16) * 2;

                    waveOutWrite(MusicWaveOut, &WaveHDRCpy, sizeof(WaveHDRCpy));
                }
                else
                {
                    DemoIsPaused = true;
                    PausedTime   = DemoTime;
                    waveOutReset(MusicWaveOut);
                }
            }

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_HOME)
            {
                if (DemoIsPaused)
                {
                    DemoTime = PausedTime = TimeOffset = 0;
                }
                else
                {
                    DemoTime = PausedTime = TimeOffset = 0;

                    waveOutReset(MusicWaveOut);
                    waveOutWrite(MusicWaveOut, &WaveHDR, sizeof(WaveHDR));
                }
            }

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_RIGHT && (DemoTime + SeekRate) < (MusicTotalSamples - 256))
            {
                if (DemoIsPaused)
                {
                    DemoTime = PausedTime = DemoTime + SeekRate;
                }
                else
                {
                    TimeOffset = DemoTime + SeekRate;

                    uint32 SamplesOffset = uint32((double(TimeOffset)/1000.0) * MusicSampleRate);

                    waveOutReset(MusicWaveOut);

                    WAVEHDR WaveHDRCpy = WaveHDR;

                    WaveHDRCpy.lpData		  += SamplesOffset * sizeof(sint16) * 2;
                    WaveHDRCpy.dwBufferLength -= SamplesOffset * sizeof(sint16) * 2;

                    waveOutWrite(MusicWaveOut, &WaveHDRCpy, sizeof(WaveHDRCpy));
                }
            }

            if (SDLEvent.type == SDL_KEYUP && SDLEvent.key.keysym.sym == SDLK_LEFT && DemoTime > SeekRate)
            {
                if (DemoIsPaused)
                {
                    DemoTime = PausedTime = DemoTime - SeekRate;
                }
                else
                {
                    TimeOffset = DemoTime - SeekRate;

                    uint32 SamplesOffset = uint32((double(TimeOffset)/1000.0) * MusicSampleRate);

                    waveOutReset(MusicWaveOut);

                    WAVEHDR WaveHDRCpy = WaveHDR;

                    WaveHDRCpy.lpData		  += SamplesOffset * sizeof(sint16) * 2;
                    WaveHDRCpy.dwBufferLength -= SamplesOffset * sizeof(sint16) * 2;

                    waveOutWrite(MusicWaveOut, &WaveHDRCpy, sizeof(WaveHDRCpy));
                }
            }
            #endif
            #endif
        }

        DoImage(SDLPixels, DemoTime);

        SDL_UpdateTexture(SDLTexture, NULL, SDLPixels, PhysicalScreenSizeX * sizeof(uint32));
        SDL_RenderClear(SDLRenderer);
        SDL_RenderCopy(SDLRenderer, SDLTexture, NULL, NULL);
        SDL_RenderPresent(SDLRenderer);
    }

    #if ((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER)
    __ReportFlushToDisk();
    #endif

    MusicDeinitialize();
    DemoDeinit();

    delete [] SDLPixels;

    SDL_DestroyTexture(SDLTexture);
    SDL_DestroyRenderer(SDLRenderer);
    SDL_DestroyWindow(SDLWindow);
    SDL_Quit();

    return 0;
}
