// (c) MX^Add
#pragma once

#include "BaseTypes/BaseTypes.h"

//
// Scalar implementation that uses float (it's more efficient on PICO than actually use fixed point, since no 64bit divide)
//
using Scalar = float;

class FScalar
{
private:

    static const Scalar SinTable [256+1];
    static const Scalar CosTable [256+1];
    static const Scalar ACosTable[256+1];

public:

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    constexpr inline static sint32 toRoundInt(Scalar v) // NOTE::Assumes v is positive (!)
    {
        return sint32(v + 0.5f);
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    constexpr inline static sint32 toCeilInt(Scalar v) // NOTE::Assumes v is positive (!)
    {
        return sint32(v + 0.999f);
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Clamp01(Scalar v)
    {
        return fmaxf(0.0f, fminf(v, 0.999f));
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Abs(Scalar x)
    {
        return Scalar(fabsf(x));
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Min(Scalar a, Scalar b)
    {
        return fminf(a, b);
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Max(Scalar a, Scalar b)
    {
        return fmaxf(a, b);
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    constexpr inline static Scalar Lerp(Scalar a, Scalar b, Scalar alpha)
    {
        return a + (b-a) * alpha;
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar RoundToOne(Scalar x)
    {
        if (fabsf(x-1.0f) < 0.000244f)
            return Scalar(1.0f);
        return x;
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Sqrt(Scalar x)
    {
        return Scalar(sqrtf(x)); // NOTE::This costs ~17 cycles
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    constexpr inline static Scalar InvSqrt(Scalar x)
    {
        // return Scalar(1.0f / sqrtf(x)); // NOTE::This costs ~34 cycles!

        // NOTE::This one is around ~15 cycles!

        Scalar half = Scalar(0.5f) * x;
        sint32 i    = *(sint32*)&x; 
               i    = 0x5f3759df - (i >> 1);
               x    = *(Scalar*)&i;

        return x * (Scalar(1.5f) - half * x * x);
    }

    static Scalar Sin(Scalar f); // Radians
    static Scalar Cos(Scalar f); // Radians

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Sin(uint8 f) // BDeg
    {
        return Scalar(SinTable[f]);
    }

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar Cos(uint8 f) // BDeg
    {
        return Scalar(CosTable[f]);
    }

    static Scalar ACos(Scalar f); // Radians

    #ifdef PI_PICO_TARGET
    __attribute__((always_inline))
    #endif
    inline static Scalar ACosLck(uint8 r) // Return raw table
    {
        return Scalar(ACosTable[r]);
    }
};

constexpr Scalar FixedNrmFactor(0.001953f);
constexpr Scalar FixedEpsilon(0.000244f);
constexpr Scalar FixedSlerpEpsilon(0.949997f);
constexpr Scalar FixedTwo(2.000000f);
constexpr Scalar FixedOne(1.000000f);
constexpr Scalar FixedZero(0.000000f);
constexpr Scalar FixedHalf(0.500000f);
constexpr Scalar FixedPi(3.14159265358979f);
constexpr Scalar FixedTwoPi(6.283185307179586f);
constexpr Scalar FixedHalfPi(1.57079632679f);
constexpr Scalar FixedToBDeg(40.743652f);
constexpr Scalar FixedFromBDeg(0.024536f);

static_assert(sizeof(Scalar) == 4, "Scalar must be 4 bytes !");