//
// PI Pico 2 DemoScene boiler plate (using ST7789 display/VGA for video, and UDA1334A for audio)
// (c) MX^Add
//

#include <string.h>
#include <pico/time.h>

#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)
#include <cstdlib>
#include "pico/stdlib.h"
#else
#include <stdarg.h>
#endif

#if PICO_DO_VGA
#include "pico/stdlib.h"
#include "hardware/spi.h"
#include "hardware/dma.h"
#include "hardware/gpio.h"
#include "hardware/pio.h"
#include "hardware/pwm.h"
#include "hardware/clocks.h"
#include "hardware/vreg.h"
#include "pimoroni_common.hpp"
#include "pimoroni_bus.hpp"
#include "scanvideo.h"
#include "composable_scanline.h"
#else
#include "st7789.hpp"
#include "rgbled.hpp"
#include "button.hpp"
#include "hardware/vreg.h"
#endif

#include "audio_i2s.h"
#include "ModPlayer/ModPlayer.h"
#include "DemoMain.h"
#include "Renderer/PostEfx.h"
#include "Renderer/FrameBuffer.h"
#include "Font/BitmapFont.h"

using namespace pimoroni;

static uint16 FrontBuffer[RasterizerSizeX*RasterizerSizeY];

#if PICO_DO_VGA
// VGA Display

volatile uint8 HWButtonStates[2]; // NOTE::Need two samples to be reliable, when output is pure white ...
volatile uint8 HWButtonXORStt;
constexpr uint VSYNC_PIN = (PICO_SCANVIDEO_COLOR_PIN_BASE + PICO_SCANVIDEO_COLOR_PIN_COUNT + 1);

static inline bool IsVGAButtonPressed(uint32 Idx)
{
	return (((HWButtonStates[0] & (1 << Idx)) && (HWButtonStates[1] & (1 << Idx))));
}

static void __time_critical_func(VGABoardButtonIrqHandler)(uint32 gpio, long unsigned int events) 
{
	constexpr uint BTN_A =  0;
	constexpr uint BTN_B =  6;
	constexpr uint BTN_C = 11;

	if (gpio != VSYNC_PIN)
		return;

	sint32 VSyncCurrentLevel = (events & GPIO_IRQ_EDGE_RISE) ? 1 : 0;

	if (VSyncCurrentLevel != scanvideo_get_mode().default_timing->v_sync_polarity) 
	{
		gpio_pull_down(BTN_A);
		gpio_set_oeover(BTN_A, GPIO_OVERRIDE_LOW);

		gpio_pull_down(BTN_B);
		gpio_set_oeover(BTN_B, GPIO_OVERRIDE_LOW);

		gpio_pull_down(BTN_C);
		gpio_set_oeover(BTN_C, GPIO_OVERRIDE_LOW);
	} 
	else 
	{
		uint32 State = 0;

		State |= gpio_get(BTN_A) << 0;
		gpio_set_oeover(BTN_A, GPIO_OVERRIDE_NORMAL);

		State |= gpio_get(BTN_B) << 1;
		gpio_set_oeover(BTN_B, GPIO_OVERRIDE_NORMAL);

		State |= gpio_get(BTN_C) << 2;
		gpio_set_oeover(BTN_C, GPIO_OVERRIDE_NORMAL);

		HWButtonXORStt				  ^= 1;
		HWButtonStates[HWButtonXORStt] = State;
	}
	
	return ;
}

static void VGABoardInitButtons()
{
	gpio_set_irq_enabled_with_callback(VSYNC_PIN, GPIO_IRQ_EDGE_RISE | GPIO_IRQ_EDGE_FALL, true, VGABoardButtonIrqHandler);
	return;
}

// This one fills every scanline from FrontBuffer

static inline void FillScanlineBuffer(struct scanvideo_scanline_buffer *buffer) 
{
	static uint32_t postamble[] = 
	{
		0x0000u | (COMPOSABLE_EOL_ALIGN << 16)
	};

	buffer->data[0] = 4;
	buffer->data[1] = host_safe_hw_ptr(buffer->data + 8);
	buffer->data[2] = 158; // First four pixels are handled separately

	uint32_t ssnum  = scanvideo_scanline_number(buffer->scanline_id);

	// For first and last 20 lines we repeat first and last line ...

	if (ssnum < 20)
		ssnum = 0;
	else
	if (ssnum >= 220)
		ssnum = RasterizerSizeY-1;
	else
		ssnum-= 20;
	
	uint16_t *pixels = FrontBuffer + ssnum * RasterizerSizeX;

	buffer->data[3]   = host_safe_hw_ptr(pixels + 4);
	buffer->data[4]   = count_of(postamble);
	buffer->data[5]   = host_safe_hw_ptr(postamble);
	buffer->data[6]   = 0;
	buffer->data[7]   = 0;
	buffer->data_used = 8;

	// 3 pixel run followed by main run, consuming the first 4 pixels

	buffer->data[8]  = (pixels[0] << 16u) | COMPOSABLE_RAW_RUN;
	buffer->data[9]  = (pixels[1] << 16u) | 0;
	buffer->data[10] = (COMPOSABLE_RAW_RUN << 16u) | pixels[2];
	buffer->data[11] = ((317 + 1 - 3) << 16u)      | pixels[3]; // NOTE::We add one for the black pixel at the end

	return;
}

// This is alarm calback called every 100us to generate scanlines

static int64_t __not_in_flash_func(VGATimerCallback)(alarm_id_t alarm_id, void *user_data) 
{
	struct scanvideo_scanline_buffer *buffer = scanvideo_begin_scanline_generation(false);

	while (buffer) 
	{
		FillScanlineBuffer(buffer);
		scanvideo_end_scanline_generation(buffer);
		buffer = scanvideo_begin_scanline_generation(false);
	}

	return 100;
}

#else
// ST7789 Display

static ST7789 Display_st7789(get_spi_pins(BG_SPI_FRONT));
static RGBLED Display_led(PicoDisplay2::LED_R, PicoDisplay2::LED_G, PicoDisplay2::LED_B);
static Button Display_button_a(PicoDisplay2::A);
static Button Display_button_b(PicoDisplay2::B);
static Button Display_button_x(PicoDisplay2::X);
static Button Display_button_y(PicoDisplay2::Y);
static bool   FrontBufferSubmited;

#endif

// Audio stuff

static uint32 AudioSamplesDone;
static bool   AudioInitDone;

static void AudioStep(void *Dst, uint size)
{ 
	if (!AudioInitDone)
		return;
	AudioSamplesDone += MODFetchSamples((int16_t *)Dst, size>>2);
	return;
}

static void AudioInit()
{
	static audio_format_t AudioFormat = 
	{
		.sample_freq   = 44100,
		.format 	   = AUDIO_BUFFER_FORMAT_PCM_S16,
		.channel_count = 2,
	};

	static audio_buffer_format ProducerFormat = 
	{
		.format		   = &AudioFormat,
		.sample_stride = 4 // 2 * 2
	}; 

	static audio_i2s_config AudioConfig = 
	{
		#if PICO_DO_VGA
		.data_pin       = DMB_PICO_AUDIO_PACK_I2S_DATA,
		.clock_pin_base = DMB_PICO_AUDIO_PACK_I2S_BCLK, // So it uses PICO_AUDIO_PACK_I2S_BCLK and PICO_AUDIO_PACK_I2S_BCLK+1 for clock
		#else
		.data_pin       = UDA_PICO_AUDIO_PACK_I2S_DATA,
		.clock_pin_base = UDA_PICO_AUDIO_PACK_I2S_BCLK, // So it uses PICO_AUDIO_PACK_I2S_BCLK and PICO_AUDIO_PACK_I2S_BCLK+1 for clock
		#endif
		.dma_channel    = 0xFF,						// Automatic assignment of DMA Channel
		.pio_sm         = 0,
	};  

	audio_i2s_setup(&AudioFormat, &AudioConfig); 
	audio_i2s_connect(AudioStep, 44100);
	audio_i2s_set_enabled(true);

	AudioInitDone = true;
	
	return;
}

// Entry

int main()
{
	InitializeFramebuffer();

	#if PICO_DO_VGA

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	// Config CPU to 300MHz

	vreg_set_voltage(VREG_VOLTAGE_1_30);
	sleep_ms(10);
	set_sys_clock_khz(50000*6, true); // 300Mhz
	sleep_ms(50);
	setup_default_uart();
	
	// Init VGA

	scanvideo_setup(&vga_mode_320x240_60);
	scanvideo_timing_enable(true);
	add_alarm_in_us(100, VGATimerCallback, NULL, true);

	// Init demo stuff (optionally provide callback to function that will be periodically called on 2nd core (rather fast, but non-deterministic interval, usually 0.5 - 1.5ms))

	DemoInit(nullptr);

	MODInitializePlayerData();
	AudioInit();

	VGABoardInitButtons();

	#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)
	char Buff[64] = { 0 };
	#endif

	uint8 ButtonStates = 0;

	#if (defined PICO_DO_TIMINGS)
	uint32 TimeOffset  = 0;
	#endif

	while (true)
	{
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)
		uint64_t DemoTime = time_us_64();
		#endif

		bool PressButtonA = false;
		bool PressButtonB = false;
		bool PressButtonC = false;
		bool PressButtonY = false;

		if (IsVGAButtonPressed(0))
		{
			if ((ButtonStates & 0x01) == 0)
			{
				ButtonStates |= 0x01;
				PressButtonA  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x01);
		}

		if (IsVGAButtonPressed(1))
		{
			if ((ButtonStates & 0x02) == 0)
			{
				ButtonStates |= 0x02;
				PressButtonB  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x02);
		}

		if (IsVGAButtonPressed(2))
		{
			if ((ButtonStates & 0x04) == 0)
			{
				ButtonStates |= 0x04;
				PressButtonC  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x04);
		}

		uint32 MusicTime = millis(); // TODO::Fetch timer from music and use it

		#if (defined PICO_DO_TIMINGS)
		if (PressButtonA)
			TimeOffset += 1000 * 5;

		if (PressButtonB)
			TimeOffset -= 1000 * 5;

		MusicTime += TimeOffset;
		#endif

		const uint32 PostEfx = DemoDo(MusicTime);

		DoPostEfx(PostEfx, FrontBuffer); // NOTE::It will tear if scanline is below update, should be done in V-Sync...

		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)
		DemoTime = time_us_64() - DemoTime;
		sprintf(Buff, "%2.2f", float(DemoTime)/1000.0f);
		PrintString2X(Buff, strlen(Buff), 0, 0, 0xFFFF, FrontBuffer);
		#endif
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	#else

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	// Config CPU to 300MHz, and SPI to 300MHz
	vreg_set_voltage(VREG_VOLTAGE_1_30);
	sleep_ms(10);
	set_sys_clock_khz(50000*6, true); // 300Mhz
	sleep_ms(50);
	clock_configure(clk_peri, 0, CLOCKS_CLK_PERI_CTRL_AUXSRC_VALUE_CLKSRC_PLL_SYS, 300000000, 300000000);
	sleep_ms(50);

	#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)	
	stdio_init_all();
	#endif

	// Init display
	Display_st7789.initialize(75'000'000);
	Display_st7789.set_backlight(0xFF);
	Display_led.set_brightness(0);

	// Clear whole display to black at start (we use the fact that here after RasterizerFramebuffer lies RasterizerDepthbuffer and both are 0 (BSS))
	Display_st7789.update(RasterizerFramebuffer, true, true);

	// Init demo stuff (optionally provide callback to function that will be periodically called on 2nd core (rather fast, but non-deterministic interval, usually 0.5 - 1.5ms))
	DemoInit(nullptr);
	MODInitializePlayerData();
	AudioInit();

	#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)
	char Buff[64] = { 0 };
	#endif

	uint8 ButtonStates = 0;

	#if (defined PICO_DO_TIMINGS)
	uint32 TimeOffset  = 0;
	uint32 TimeFreeze  = 0;

	static const uint32 FreezeTimeTable[10]
	{
		  0,
		  3 * 1000,
		 16 * 1000,
		 20 * 1000,
		 28 * 1000 + 500,
		 34 * 1000,
		 55 * 1000,
		 69 * 1000 + 500,
		 91 * 1000,
		122 * 1000 + 500
	};
	#endif

	while (true)
	{
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS)
		uint64_t DemoTime = time_us_64();
		#endif

		bool PressButtonA = false;
		bool PressButtonB = false;
		bool PressButtonX = false;
		bool PressButtonY = false;

		if (Display_button_a.raw())
		{
			if ((ButtonStates & 0x01) == 0)
			{
				ButtonStates |= 0x01;
				PressButtonA  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x01);
		}

		if (Display_button_b.raw())
		{
			if ((ButtonStates & 0x02) == 0)
			{
				ButtonStates |= 0x02;
				PressButtonB  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x02);
		}

		if (Display_button_x.raw())
		{
			if ((ButtonStates & 0x04) == 0)
			{
				ButtonStates |= 0x04;
				PressButtonX  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x04);
		}

		if (Display_button_y.raw())
		{
			if ((ButtonStates & 0x08) == 0)
			{
				ButtonStates |= 0x08;
				PressButtonY  = true;
			}
		}
		else
		{
			ButtonStates &= ~uint8(0x08);
		}

		// Time jumps
		#if (defined PICO_DO_TIMINGS)
		if (PressButtonA)
			TimeOffset += 1000 * 5;

		if (PressButtonB)
			TimeOffset -= 1000 * 5;

		if (PressButtonX)
			TimeFreeze = (TimeFreeze + 1) % 10;
		#endif

		// Rendering (of graphics and audio)
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		uint64_t RenderTime = time_us_64();
		#endif

		uint32 MusicTime = millis(); // TODO::Fetch timer from music and use it

		#if (defined PICO_DO_TIMINGS)
		if (TimeFreeze)
			MusicTime  = FreezeTimeTable[TimeFreeze];
		else
			MusicTime += TimeOffset;
		#endif

		const uint32 PostEfx = DemoDo(MusicTime); 

		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		RenderTime = time_us_64() - RenderTime;
		#endif

		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		uint64_t DMAFinishTime = time_us_64();
		#endif
		if (FrontBufferSubmited)
		{
			Display_st7789.endupdate();
			FrontBufferSubmited = false;
		}
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		DMAFinishTime = time_us_64() - DMAFinishTime;
		#endif

		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		uint64_t SwapTime = time_us_64();
		#endif
		DoPostEfx(PostEfx, FrontBuffer);
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		PrintString(Buff, strlen(Buff), 0, 0, 0xFFFF, FrontBuffer);
		#endif
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 2)
		PrintString2X(Buff, strlen(Buff), 0, 0, 0xFFFF, FrontBuffer);
		#endif
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		SwapTime = time_us_64() - SwapTime;
		#endif
		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		uint64_t FlipTime = time_us_64();
		#endif

		Display_st7789.update(FrontBuffer, uint16((PhysicalScreenSizeY - RasterizerSizeY) >> 1), uint16(RasterizerSizeY), true, false);
		FrontBufferSubmited = true;

		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 1)
		FlipTime = time_us_64() - FlipTime;
		DemoTime = time_us_64() - DemoTime;
		sprintf(Buff, "R:%2.1f S:%2.1f F:%2.1f D:%2.1f A:%2.1f S:%i", float(RenderTime)/1000.0f, float(SwapTime)/1000.0f, float(FlipTime)/1000.0f, float(DMAFinishTime)/1000.0f, float(DemoTime)/1000.0f, AudioSamplesDone/1024);
		#endif

		#if (defined PICO_DO_TIMINGS && PICO_DO_TIMINGS == 2)
		DemoTime = time_us_64() - DemoTime;
		sprintf(Buff, "%2.2f", float(DemoTime)/1000.0f);
		#endif
	}

	#endif // PICO_DO_VGA

	return 0;
}

//
// LTO Stubs, if -flto is used
//
/*
extern "C" int vprintf(const char *format, va_list ap) 
{
	return 0;
}

extern "C" int puts(const char* s)
{
	return 0;
}

extern "C" int printf(const char* format, ...)
{
	va_list args;
	va_start(args, format);
	va_end(args);
	return 0;
}
*/
