// (c) MX^Add

#include "DemoMain.h"
#include "BaseTypes/BaseTypes.h"
#include "BaseTypes/ScalarType.h"
#include "RendererTypes/Scene.h"
#include "Renderer/FrameBuffer.h"
#include "Renderer/SoftwareRasterizer.h"
#include "Renderer/Rasterizers.h"
#include "Font/BitmapFont.h"
#include "MetaBalls/MetaBalls.h"
#include "BaseTypes/PicoIntrinsics.h"

#ifdef PI_PICO_TARGET
#include <hardware/interp.h> // Interpolators
#include <pico/time.h>
#undef MIN
#undef MAX
#endif

#include "Renderer/PostEfx.h"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Static demo data
// 

// All scenes

#include "Scenes/Scene00.h"
#include "Scenes/Scene01.h"
#include "Scenes/Scene02.h"
#include "Scenes/Scene03.h"
#include "Scenes/Scene04.h"
#include "Scenes/Scene05.h"

// All textures

#include "Textures/Texture_CyberBlue.h"
#include "Textures/Texture_CyberGray.h"
#include "Textures/Texture_CyberViolet.h"

#include "Textures/Texture_WrapClouds.h"
#include "Textures/Texture_WrapAbstractDark.h"
#include "Textures/Texture_WrapTunel.h"
#include "Textures/Texture_WrapCyber.h"
#include "Textures/Texture_MotoMask.h"
#include "Textures/Texture_Logo.h"

#include "Textures/Texture_StartGradient.h"
#include "Textures/Texture_FontGradientSmall.h"
#include "Textures/Texture_FontGradientLarge.h"
#include "Textures/Texture_WrapBrushedMetal.h"
#include "Textures/Texture_SplashImage.h"

// #include "Textures/Texture_CyberGreen.h"
// #include "Textures/Texture_CyberLight.h"
// #include "Textures/Texture_CyberOrange.h"
// #include "Textures/Texture_CyberYelow.h"
// #include "Textures/Texture_WrapCircuitGreen.h"
// #include "Textures/Texture_WrapCircuitViolet.h"
// #include "Textures/Texture_WrapGalaxy.h"
// #include "Textures/Texture_WrapAbstract.h"
// #include "Textures/Texture_WrapNMap.h"
// #include "Textures/Texture_WrapDiffuse.h"

static const uint8 WrapSinTable32[] = { 0x00, 0x01, 0x02, 0x02, 0x03, 0x04, 0x05, 0x05, 0x06, 0x07, 0x08, 0x09, 0x09, 0x0A, 0x0B, 0x0C, 0x0C, 0x0D, 0x0E, 0x0E, 0x0F, 0x10, 0x10, 0x11, 0x12, 0x12, 0x13, 0x14, 0x14, 0x15, 0x15, 0x16, 0x17, 0x17, 0x18, 0x18, 0x19, 0x19, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1C, 0x1C, 0x1D, 0x1D, 0x1D, 0x1E, 0x1E, 0x1E, 0x1E, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1E, 0x1E, 0x1E, 0x1E, 0x1D, 0x1D, 0x1D, 0x1C, 0x1C, 0x1B, 0x1B, 0x1B, 0x1A, 0x1A, 0x19, 0x19, 0x18, 0x18, 0x17, 0x17, 0x16, 0x15, 0x15, 0x14, 0x14, 0x13, 0x12, 0x12, 0x11, 0x10, 0x10, 0x0F, 0x0E, 0x0E, 0x0D, 0x0C, 0x0C, 0x0B, 0x0A, 0x09, 0x09, 0x08, 0x07, 0x06, 0x05, 0x05, 0x04, 0x03, 0x02, 0x02, 0x01, 0x00, 0x00, 0xFF, 0xFF, 0xFE, 0xFD, 0xFC, 0xFC, 0xFB, 0xFA, 0xF9, 0xF8, 0xF8, 0xF7, 0xF6, 0xF5, 0xF5, 0xF4, 0xF3, 0xF3, 0xF2, 0xF1, 0xF1, 0xF0, 0xEF, 0xEF, 0xEE, 0xED, 0xED, 0xEC, 0xEC, 0xEB, 0xEA, 0xEA, 0xE9, 0xE9, 0xE8, 0xE8, 0xE7, 0xE7, 0xE6, 0xE6, 0xE6, 0xE5, 0xE5, 0xE4, 0xE4, 0xE4, 0xE3, 0xE3, 0xE3, 0xE3, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE3, 0xE3, 0xE3, 0xE3, 0xE4, 0xE4, 0xE4, 0xE5, 0xE5, 0xE6, 0xE6, 0xE6, 0xE7, 0xE7, 0xE8, 0xE8, 0xE9, 0xE9, 0xEA, 0xEA, 0xEB, 0xEC, 0xEC, 0xED, 0xED, 0xEE, 0xEF, 0xEF, 0xF0, 0xF1, 0xF1, 0xF2, 0xF3, 0xF3, 0xF4, 0xF5, 0xF5, 0xF6, 0xF7, 0xF8, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFC, 0xFD, 0xFE, 0xFF, 0xFF, 0x00 };

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Demo scenario code
//

static inline uint8 GetEffectFromFrame(uint32 Frame)
{
    uint8 Result = 0;

    if (Frame > DemoPart00Begin) Result++;

    if (Frame > DemoPart01Begin) Result++;

    if (Frame > DemoPart02Begin) Result++;

    if (Frame > DemoPart03Begin) Result++;

    if (Frame > DemoPart04Begin) Result++;

    if (Frame > DemoPart05Begin) Result++;

    if (Frame > DemoPart06Begin) Result++;

    return Result;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DoTunelPart)(uint16 Frame, sint32 YOffset)
{
    constexpr Scalar DeltaX = FixedTwo / Scalar(RasterizerSizeX);
    constexpr Scalar DeltaY = FixedTwo / Scalar(RasterizerSizeY);

    __restrict const uint16 *Image    = GetData_WrapTunel(); 
    __restrict       uint16 *h        = RasterizerFramebuffer + YOffset * RasterizerSizeX;
    __restrict const uint16 *z        = RasterizerDepthbuffer + YOffset * RasterizerSizeX;

    const Scalar Time     = Scalar(sint32(Frame)) * Scalar(1.5f);
    const Scalar TimeFast = Time * 4;
    const Scalar TimeSlow = Time * Scalar(0.015f);

    Scalar  dx, dy;
    uint32 x,  y;

    for (y = 0, dy = FixedOne - (DeltaX * ((RasterizerSizeX - RasterizerSizeY) >> 1)) - DeltaX * YOffset; y < RasterizerSizeY >> 1; y++, dy -= DeltaX)
    {
        Scalar absDy = FScalar::Abs(dy);

        for (x = 0, dx = -FixedOne; x < RasterizerSizeX; x++, dx += DeltaX, h++, z++)
        { 
            if (*z != 0xFFFF)
                continue;

            Scalar absDx = FScalar::Abs(dx);
            Scalar r     = Scalar(0.9604f) * MAX(absDx, absDy) + Scalar(0.3978f) * MIN(absDx, absDy); // Faster than: FScalar::Sqrt(dx*dx + dy*dy)

            if (r > Scalar(0.25f))
            {
                Scalar invr     = (dx*dx + dy*dy);
                Scalar half     = FixedHalf * invr;
                sint32 i        = *(sint32*)&invr;
                       i        = 0x5f3759df - (i >> 1);
                       invr     = *(Scalar*)&i;
                       invr     = invr * (Scalar(1.5f) - half * invr * invr);
                Scalar CosTheta = dy * invr;
                Scalar CosBias  = CosTheta * Scalar(127.5f) + Scalar(127.5f);
                sint32 CosII    = sint32(CosBias);
                Scalar AcosA    = FScalar::ACosLck(uint8(CosII));
                Scalar AcosB    = FScalar::ACosLck(uint8(CosII+1));
                Scalar Theta    = AcosA + (AcosB-AcosA) * (CosBias - CosII);
                       Theta    = x < (RasterizerSizeX>>1) ? -Theta : Theta;
                Scalar Twist    = (TimeSlow + r + r) * Scalar(14);
                sint32 TwN      = sint32(Twist);
                Scalar TwCA     = FScalar::Cos(uint8(TwN));
                Scalar TwCB     = FScalar::Cos(uint8(TwN+1));
                       Twist    = TwCA + (TwCB-TwCA) * (Twist - TwN);
                Scalar Light    = MIN(FixedOne, Scalar(1.25f) + CosTheta);
                       Theta   -= Twist;

                uint8  u = sint32(Theta * Scalar(256));
                uint8  v = sint32(invr * Scalar(128) + TimeFast);

                Scalar Fade  = MIN(FixedOne, (r - Scalar(0.25f)) * FixedTwo);
                       Fade *= Light * Light;
                uint32 Scale = uint32(Fade * 32);

                if (Scale == 32)
                { 
                    *h = Image[(uint32(u)<<8) | uint32(v)];
                }
                else
                {
                    uint32 Color = Image[(uint32(u)<<8) | uint32(v)];

                    *h = uint16(((((Color & (0x1F<<11)) * Scale) >> 5) & (0x1F<<11)) | 
                                ((((Color & (0x3F<< 5)) * Scale) >> 5) & (0x3F<< 5)) | 
                                ((((Color & (0x1F    )) * Scale) >> 5) & (0x1F    )) );
                }
            }
            else
            {
                *h = 0;
            }
        }
    }   

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DoWrapPart)(const uint16 *Image, uint16 *h, const uint8 *SinTable, const uint8 MonoTime, const uint32 XOff, const uint32 YOff, const uint32 StartY, const uint32 SizeY)
{
    for (uint32 y = StartY; y < SizeY; y++)
    {
        uint32 yWarp = uint32(SinTable[(y + MonoTime) & 0xFF]) + YOff;
        for (uint32 x = 0; x < RasterizerSizeX; x++, h++)
            *h = Image[(((y + (uint32(SinTable[(x + MonoTime) & 0xFF]) + XOff)) & 0xFF) << 8) + ((x + yWarp) & 0xFF)];
    }  

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void GradientFill(const FColor16 CTop, const FColor16 CBot)
{
    const Scalar Step  = FixedOne / Scalar(RasterizerSizeY);
          Scalar Alpha = FixedZero;

    __restrict uint32 *g = (uint32*)RasterizerFramebuffer;

    for (uint32 ScanY = 0; ScanY < RasterizerSizeY; ScanY++, Alpha += Step)
    {
        FColor16 ColA; ColA.Lerp(CTop, CBot, (ScanY&1) ? FScalar::Max(0, Alpha - Step * FixedHalfPi) : Alpha);
        FColor16 ColB; ColB.Lerp(CTop, CBot, (ScanY&1) ? Alpha : FScalar::Max(0, Alpha - Step * FixedHalfPi));

        uint32 Color = uint32(ColA.toColorNoClip()) | (uint32(ColB.toColorNoClip()) << 16);

        for (uint32 i = 0; i < RasterizerSizeX>>1; i++, g++)
            *g = Color;
    }

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DrawTwisterY)(sint32 Frame)
{
    struct FStripe
    {
        sint32 Depth;
        sint32 XCenter;
        sint32 XStart, XEnd;
    };

    constexpr sint32 CenterX   = 30;    // Where the twister is
    constexpr Scalar Amplitude = 20.0f; // Cylinder radius in pixels
    constexpr Scalar BaseWidth = 10.0f; // Stripe base half-width
    constexpr sint32 Sides     = 5;     // Number of stripes

    const uint16 *Texture = GetData_WrapTunel();

    const Scalar Beat = FScalar::Abs(FScalar::Sin(1.50f + (Scalar(Frame) / 30.0f) * 5.0f));
    
    FStripe Stripes[Sides];

    __restrict uint16 *g = RasterizerFramebuffer;

    for (sint32 y = 0; y < RasterizerSizeY; y++, g += RasterizerSizeX)
    {
        const Scalar Rotation = Scalar(Frame)  * 0.11f + y * 0.045f + Beat;
        const Scalar Twist    = FScalar::Sin(y * 0.07f + Scalar(Frame) * 0.061f) * 0.5f;

        const uint16 * __restrict t = &Texture[(y & 0xFF) << 8];

        // Fill stripes

        for (sint32 i = 0; i < Sides; i++) 
        {
            Scalar ang = Rotation + Twist + i * (FixedTwoPi / Sides);
            Scalar sx  = FScalar::Sin(ang) * Amplitude;
            Scalar sz  = FScalar::Cos(ang);
            Scalar XCe = CenterX + sx;
            Scalar XWe = BaseWidth * (0.7f + 0.3f * sz);

            Stripes[i].Depth   = sint32((0.5f + sz * 0.5f) * 65535.0f);
            Stripes[i].XCenter = sint32(XCe);
            Stripes[i].XStart  = CLAMP(sint32(XCe - XWe), 0, RasterizerSizeX-1);
            Stripes[i].XEnd    = CLAMP(sint32(XCe + XWe), 0, RasterizerSizeX-1);
        }

        // Sort by depth

        for (sint32 i = 0; i < Sides - 1; i++) 
            for (sint32 j = i + 1; j < Sides; j++) 
                if (Stripes[i].Depth > Stripes[j].Depth) 
                    SWAP(Stripes[i], Stripes[j]);

        // Draw

        for (sint32 i = 0; i < Sides; i++)
        {
            const FStripe &Strip = Stripes[i];

            for (sint32 x = Strip.XStart; x <= Strip.XEnd; x++) 
                g[x] = FColor16(t[((x-Strip.XCenter) & 0xFF)]).ScaledValueFP(Strip.Depth).AddSat(g[x]);
            //  g[x] = FColor16(t[((x-Strip.XCenter) & 0xFF)]).ScaledValueFP(Strip.Depth).toColorNoClip();
        }
    }

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DrawTwisterX)(sint32 Frame, uint8 Factor)
{
    struct FStripe
    {
        sint32 Depth;
        sint32 YCenter;
        sint32 YStart, YEnd;
    };

    constexpr sint32 CenterY   = 100;   // Where the twister is
    constexpr Scalar Amplitude = 32.0f; // Cylinder radius in pixels
    constexpr Scalar BaseWidth = 16.0f; // Stripe base half-width
    constexpr sint32 Sides     = 5;     // Number of stripes

    const uint16* Texture = GetData_WrapBrushedMetal();
    const Scalar SFactor  = Scalar(Factor) / 255.0f;

    FStripe Stripes[Sides];

    __restrict uint16* g = RasterizerFramebuffer;

    for (sint32 x = 0; x < RasterizerSizeX; x++)
    {
        const Scalar Rotation = Scalar(Frame) * 0.11f + x * 0.045f + SFactor;
        const Scalar Twist = FScalar::Sin(x * 0.07f + Scalar(Frame) * 0.061f) * 0.5f;

        const uint16* __restrict t = &Texture[(x & 0xFF) << 8]; // Will be flipped X/Y

        // Fill stripes

        for (sint32 i = 0; i < Sides; i++)
        {
            Scalar ang = Rotation + Twist + i * (FixedTwoPi / Sides);
            Scalar sy = FScalar::Sin(ang) * Amplitude;
            Scalar sz = FScalar::Cos(ang);
            Scalar YCe = CenterY + sy;
            Scalar YWe = BaseWidth * (0.7f + 0.3f * sz);

            Stripes[i].Depth = sint32((0.5f + sz * 0.5f) * 65535.0f);
            Stripes[i].YCenter = sint32(YCe);
            Stripes[i].YStart = CLAMP(sint32(YCe - YWe), 0, RasterizerSizeY - 1);
            Stripes[i].YEnd = CLAMP(sint32(YCe + YWe), 0, RasterizerSizeY - 1);
        }

        // Sort by depth

        for (sint32 i = 0; i < Sides - 1; i++)
            for (sint32 j = i + 1; j < Sides; j++)
                if (Stripes[i].Depth > Stripes[j].Depth)
                    SWAP(Stripes[i], Stripes[j]);

        // Draw

        for (sint32 i = 0; i < Sides; i++)
        {
            const FStripe& Strip = Stripes[i];
            for (sint32 y = Strip.YStart; y <= Strip.YEnd; y++)
                g[x + y * RasterizerSizeX] = FColor16(t[((y - Strip.YCenter) & 0xFF)]).ScaledValueFP(Strip.Depth).toColorNoClip();
        }
    }

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DrawMotoBoardInner)(sint32 StartX, sint32 EndX, sint32 StartY, sint32 EndY, sint32 offx, sint32 offy, sint32 ImageSizeX, sint32 ImageSizeY, Scalar SinN, Scalar CosA, const uint16 *ImageData, uint16 *g)
{
    constexpr uint16 RGBColorKey  = uint16(0x1F) | (uint16(0x1F) << 11);
    constexpr uint16 RGBMirrorKey = uint16(0x3F) << 5;

    const sint32 PivotX = ImageSizeX/2;
    const sint32 PivotY = ImageSizeY/2;

    for (sint32 y = StartY; y < EndY; y++, g+=RasterizerSizeX) 
    {
        const Scalar dys = Scalar(y - offy) * SinN;
        const Scalar dyc = Scalar(y - offy) * CosA;

        for (sint32 x = StartX; x < EndX; x++) 
        {
            const Scalar dx = Scalar(x - offx);

            const sint32 ix = sint32(dx * CosA - dys) + PivotX;
            const sint32 iy = sint32(dx * SinN + dyc) + PivotY;

            if (ix >= 0 && iy >= 0 && ix < ImageSizeX && iy < ImageSizeY) 
            {
                uint16 Color = ImageData[iy * ImageSizeX + ix];
                if (Color != RGBColorKey)
                {
                    if (Color == RGBMirrorKey)
                        g[x] = g[((RasterizerSizeX-1) - x) - (RasterizerSizeX * 80)];
                    else
                        g[x] = Color;
                }
            }
        }
    }

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DrawMotoBoard)(sint32 offx, sint32 offy, Scalar Angle, const uint16 *ImageData, const sint32 ImageSizeX, const sint32 ImageSizeY)
{
    const Scalar SinA = FScalar::Sin(Angle);
    const Scalar SinN = -SinA;
    const Scalar CosA = FScalar::Cos(Angle);
    const sint32 PivotX = ImageSizeX/2;
    const sint32 PivotY = ImageSizeY/2;

    // Image box corners

    sint32 CornersX[4] = { -PivotX,  ImageSizeX - PivotX, -PivotX, ImageSizeX - PivotX };
    sint32 CornersY[4] = { -PivotY,  -PivotY, ImageSizeY  -PivotY, ImageSizeY - PivotY };

    // Bounding box of image

    sint32 MinX = 0xFFFF, MinY = 0xFFFF, MaxX = -0xFFFF, MaxY = -0xFFFF;

    #ifdef PI_PICO_TARGET
    #pragma GCC unroll 4
    #endif
    for (uint32 i = 0; i < 4; i++) 
    {
        sint32 x = sint32(Scalar(CornersX[i]) * CosA - Scalar(CornersY[i] * SinA) + offx);
        sint32 y = sint32(Scalar(CornersX[i]) * SinA + Scalar(CornersY[i] * CosA) + offy);

        MinX = MIN(MinX, x);
        MaxX = MAX(MaxX, x);
        MinY = MIN(MinY, y);
        MaxY = MAX(MaxY, y);
    }

    // Clamp bounding box to framebuffer

    sint32 StartX = MAX(0, MinX-1);
    sint32 StartY = MAX(0, MinY-1);
    sint32 EndX   = MIN(RasterizerSizeX, MaxX+1);
    sint32 EndY   = MIN(RasterizerSizeY, MaxY+1);

    if (StartX >= EndX || StartY >= EndY)
        return;

    __restrict uint16 *g = (uint16*)RasterizerFramebuffer;

    g += StartY * RasterizerSizeX;

    // Draw rotated

    sint32 YSpan = EndY - StartY;

    if (YSpan > 8)
    {
        struct FLocalData
        {
            sint32 StartX;
            sint32 EndX; 
            sint32 StartY; 
            sint32 EndY;
            sint32 offx; 
            sint32 offy;
            sint32 ImageSizeX; 
            sint32 ImageSizeY; 
            Scalar SinN; 
            Scalar CosA; 
            const uint16 *ImageData; 
            uint16 *g;
        };

        FLocalData *Local = (FLocalData *)RasterizerGetInSpace();
        Local->StartX     = StartX;
        Local->EndX       = EndX;
        Local->StartY     = StartY;
        Local->EndY       = StartY + (YSpan >> 1);
        Local->offx       = offx;
        Local->offy       = offy;
        Local->ImageSizeX = ImageSizeX;
        Local->ImageSizeY = ImageSizeY;
        Local->SinN       = SinN;
        Local->CosA       = CosA;
        Local->ImageData  = ImageData;
        Local->g          = g;

        PushSecondCoreWork([](const void* Arg)
        {
            const FLocalData *Local = (FLocalData *)Arg;
            DrawMotoBoardInner(Local->StartX, Local->EndX, Local->StartY, Local->EndY, Local->offx, Local->offy, Local->ImageSizeX, Local->ImageSizeY, Local->SinN, Local->CosA, Local->ImageData, Local->g);
            return;
        });

        DrawMotoBoardInner(StartX, EndX, StartY + (YSpan >> 1), EndY, offx, offy, ImageSizeX, ImageSizeY, SinN, CosA, ImageData, g + (YSpan >> 1) * RasterizerSizeX);
        FlushWorkItems();
    }
    else
    {
        DrawMotoBoardInner(StartX, EndX, StartY, EndY, offx, offy, ImageSizeX, ImageSizeY, SinN, CosA, ImageData, g);
    }

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DrawMotoBoardInnerBi)(sint32 StartX, sint32 EndX, sint32 StartY, sint32 EndY, sint32 offx, sint32 offy, sint32 ImageSizeX, sint32 ImageSizeY, Scalar SinN, Scalar CosA, const uint16 *ImageData, uint16 *g)
{
    constexpr uint16 RGBColorKey  = uint16(0x1F) | (uint16(0x1F) << 11);
    constexpr uint16 RGBMirrorKey = uint16(0x3F) << 5;
    constexpr uint32 RMask        = 0x1F << 11;
    constexpr uint32 GMask        = 0x3F <<  5;
    constexpr uint32 BMask        = 0x1F;

    const sint32 PivotX = ImageSizeX/2;
    const sint32 PivotY = ImageSizeY/2;

    for (sint32 y = StartY; y < EndY; y++, g+=RasterizerSizeX) 
    {
        const Scalar dys = Scalar(y - offy) * SinN;
        const Scalar dyc = Scalar(y - offy) * CosA;

        for (sint32 x = StartX; x < EndX; x++) 
        {
            const Scalar dx = Scalar(x - offx);
                  Scalar fx = dx * CosA - dys;
                  Scalar fy = dx * SinN + dyc;

                  sint32 ix = sint32(fx);
                  sint32 iy = sint32(fy);
                         fx-= fx < 0.001f ? ix-1 : ix;
                         fy-= fy < 0.001f ? iy-1 : iy;
                         ix+= PivotX;
                         iy+= PivotY;

            if (ix >= 0 && iy >= 0 && (ix+1) < ImageSizeX && (iy+1) < ImageSizeY) 
            {
                uint16 c00 = ImageData[iy * ImageSizeX + ix];
                uint16 c10 = ImageData[iy * ImageSizeX + ix + 1];
                uint16 c01 = ImageData[(iy + 1) * ImageSizeX + ix];
                uint16 c11 = ImageData[(iy + 1) * ImageSizeX + ix + 1];

                if (c00 == RGBColorKey && c10 == RGBColorKey && c01 == RGBColorKey && c11 == RGBColorKey)
                    continue;

                uint16 cbg = g[x];
                uint16 cmr = g[((RasterizerSizeX-1) - x) - (RasterizerSizeX * 80)];

                if (c00 == RGBColorKey) c00 = cbg;
                if (c10 == RGBColorKey) c10 = cbg;
                if (c01 == RGBColorKey) c01 = cbg;
                if (c11 == RGBColorKey) c11 = cbg;

                if (c00 == RGBMirrorKey) c00 = cmr;
                if (c10 == RGBMirrorKey) c10 = cmr;
                if (c01 == RGBMirrorKey) c01 = cmr;
                if (c11 == RGBMirrorKey) c11 = cmr;

                sint32 w00 = sint32(((1.0f - fx) * (1.0f - fy)) * 255.0f);
                sint32 w10 = sint32((fx * (1.0f - fy)) * 255.0f);
                sint32 w01 = sint32(((1.0f - fx) * fy) * 255.0f);
                sint32 w11 = sint32((fx * fy) * 255.0f);

                sint32 rr  = ((((c00 & RMask) * w00) + ((c10 & RMask) * w10) + ((c01 & RMask) * w01) + ((c11 & RMask) * w11)) >> 8) & RMask;
                sint32 gg  = ((((c00 & GMask) * w00) + ((c10 & GMask) * w10) + ((c01 & GMask) * w01) + ((c11 & GMask) * w11)) >> 8) & GMask;
                sint32 bb  = ((((c00 & BMask) * w00) + ((c10 & BMask) * w10) + ((c01 & BMask) * w01) + ((c11 & BMask) * w11)) >> 8) & BMask;

                g[x] = uint16(rr | gg | bb);
            }
        }
    }

    return;
}

#ifdef PI_PICO_TARGET
__attribute__((noinline))
#endif
static void __not_in_flash_func(DrawMotoBoardBi)(sint32 offx, sint32 offy, Scalar Angle, const uint16 *ImageData, const sint32 ImageSizeX, const sint32 ImageSizeY)
{
    const Scalar SinA = FScalar::Sin(Angle);
    const Scalar SinN = -SinA;
    const Scalar CosA = FScalar::Cos(Angle);
    const sint32 PivotX = ImageSizeX/2;
    const sint32 PivotY = ImageSizeY/2;

    // Image box corners

    sint32 CornersX[4] = { -PivotX,  ImageSizeX - PivotX, -PivotX, ImageSizeX - PivotX };
    sint32 CornersY[4] = { -PivotY,  -PivotY, ImageSizeY  -PivotY, ImageSizeY - PivotY };

    // Bounding box of image

    sint32 MinX = 0xFFFF, MinY = 0xFFFF, MaxX = -0xFFFF, MaxY = -0xFFFF;

    #ifdef PI_PICO_TARGET
    #pragma GCC unroll 4
    #endif
    for (uint32 i = 0; i < 4; i++) 
    {
        sint32 x = sint32(Scalar(CornersX[i]) * CosA - Scalar(CornersY[i] * SinA) + offx);
        sint32 y = sint32(Scalar(CornersX[i]) * SinA + Scalar(CornersY[i] * CosA) + offy);

        MinX = MIN(MinX, x);
        MaxX = MAX(MaxX, x);
        MinY = MIN(MinY, y);
        MaxY = MAX(MaxY, y);
    }

    // Clamp bounding box to framebuffer

    sint32 StartX = MAX(0, MinX-1);
    sint32 StartY = MAX(0, MinY-1);
    sint32 EndX   = MIN(RasterizerSizeX, MaxX+1);
    sint32 EndY   = MIN(RasterizerSizeY, MaxY+1);

    if (StartX >= EndX || StartY >= EndY)
        return;

    __restrict uint16 *g = (uint16*)RasterizerFramebuffer;

    g += StartY * RasterizerSizeX;

    // Draw rotated

    sint32 YSpan = EndY - StartY;

    if (YSpan > 8)
    {
        struct FLocalData
        {
            sint32 StartX;
            sint32 EndX; 
            sint32 StartY; 
            sint32 EndY;
            sint32 offx; 
            sint32 offy;
            sint32 ImageSizeX; 
            sint32 ImageSizeY; 
            Scalar SinN; 
            Scalar CosA; 
            const uint16 *ImageData; 
            uint16 *g;
        };

        FLocalData *Local = (FLocalData *)RasterizerGetInSpace();
        Local->StartX     = StartX;
        Local->EndX       = EndX;
        Local->StartY     = StartY;
        Local->EndY       = StartY + (YSpan >> 1);
        Local->offx       = offx;
        Local->offy       = offy;
        Local->ImageSizeX = ImageSizeX;
        Local->ImageSizeY = ImageSizeY;
        Local->SinN       = SinN;
        Local->CosA       = CosA;
        Local->ImageData  = ImageData;
        Local->g          = g;

        PushSecondCoreWork([](const void* Arg)
        {
            const FLocalData *Local = (FLocalData *)Arg;
            DrawMotoBoardInnerBi(Local->StartX, Local->EndX, Local->StartY, Local->EndY, Local->offx, Local->offy, Local->ImageSizeX, Local->ImageSizeY, Local->SinN, Local->CosA, Local->ImageData, Local->g);
            return;
        });

        DrawMotoBoardInnerBi(StartX, EndX, StartY + (YSpan >> 1), EndY, offx, offy, ImageSizeX, ImageSizeY, SinN, CosA, ImageData, g + (YSpan >> 1) * RasterizerSizeX);
        FlushWorkItems();
    }
    else
    {
        DrawMotoBoardInnerBi(StartX, EndX, StartY, EndY, offx, offy, ImageSizeX, ImageSizeY, SinN, CosA, ImageData, g);
    }

    return;
}

uint32 DemoDo(uint32 Tick)
{
    //
    // Scene switch
    // 
    DBGM(0, nullptr);
    DBGS(nullptr);

    const uint32 OveralFrame  = Tick / 33;
    const uint8 CurrentEffect = GetEffectFromFrame(OveralFrame);

    uint32               Result     = MakePostEffect();
    FSoftwareRasterizer *Rasterizer = FSoftwareRasterizer::Get();
        
    //
    // Render scene at given frame
    //
    switch (CurrentEffect)
    {
        case 0:
            {
                const uint32 LocalFrame = OveralFrame - DemoPartBSBegin;

                DBGM(LocalFrame, "Cam0");
                DBGS("SceneBS");

                memset(RasterizerFramebuffer, 0, RasterizerSizeX * RasterizerSizeY * sizeof(uint16));

                const uint16 *StartGradient = GetData_StartGradient();
                const uint8 Intensity = 0xFF - (((StartGradient[LocalFrame & 0xFF] >> 5) & 0x3F) << 2);

                DrawTwisterX(LocalFrame, Intensity);

                Result = MakePostEffect(EPostEffect::FadeToBlack, Intensity);
            }
        break;

        case 1:
            {
                const uint32 LocalFrame = OveralFrame - DemoPart00Begin;
                const FScene *Scene = Get_Scene00();

                FVector3D LightVector = LocalFrame > 540 ? FVector3D(1, 2, -2) : FVector3D(1, 1, -1);
                LightVector.Normalize();

                Rasterizer->SetDefaultState();
                Rasterizer->SetState_ClearColor(true, 0, [](uint16 Frame, const FVector3D& CamDir, const FVector3D& CamUp, const FMatrix& CamMat, const FMatrix& ProjMat, const FMatrix& CamProjMat)
                {
                    GradientFill(FColor16(uint8(0x0F), uint8(0x0F), uint8(0x1F)), FColor16(uint8(0x50), uint8(0x50), uint8(0x80)));
                    return;
                });
                Rasterizer->SetState_ClearDepth(true);
                Rasterizer->SetState_AmbientLight(Scalar(0.0625f));
                Rasterizer->SetState_DirectionalLight(true, LightVector);

                uint8 EfxStage = 0;

                if (LocalFrame > 210) EfxStage++;
                if (LocalFrame > 350) EfxStage++;
                if (LocalFrame > 540) EfxStage++;

                switch (EfxStage)
                {
                    case 0:
                        {
                            const uint32 CurrentFrame = LocalFrame;

                            DBGM(CurrentFrame, "Cam0");
                            DBGS("Scene00");
                            Rasterizer->Render(Scene, CurrentFrame, Scene00_Camera_Cam0);

                            if (CurrentFrame < 63)
                                Result = MakePostEffect(EPostEffect::FadeToBlack, (63-CurrentFrame) << 2);
                        }
                    break;

                    case 1:
                        {
                            const uint32 CurrentFrame = LocalFrame - 210;

                            DBGM(CurrentFrame, "Cam1");
                            DBGS("Scene00");
                            Rasterizer->Render(Scene, CurrentFrame, Scene00_Camera_Cam1);
                        }
                    break;

                    case 2:
                        {
                            const uint32 CurrentFrame = (LocalFrame - 350) + 40;

                            DBGM(CurrentFrame, "Cam2");
                            DBGS("Scene00");
                            Rasterizer->Render(Scene, CurrentFrame, Scene00_Camera_Cam2);
                        }
                    break;

                    case 3:
                        {
                            const uint32 CurrentFrame = LocalFrame - 540;

                            DBGM(270 + CurrentFrame, "Cam2");
                            DBGS("Scene00");
                            Rasterizer->Render(Scene, 270 + CurrentFrame, Scene00_Camera_Cam2);

                            if (CurrentFrame > 100)
                            {
                                Result = MakePostEffect(EPostEffect::FadeToBlack, CLAMP((sint32(CurrentFrame)-100) << 3, 0, 0xFF));
                            }
                        }
                    break;
                }
                
                if (LocalFrame > 60 && LocalFrame < 350)
                {
                    uint8 Alpha0 = CLAMP((sint32(LocalFrame) -  60) * 5, 0, 0xFF);
                    uint8 Alpha1 = CLAMP((sint32(LocalFrame) -  90) * 5, 0, 0xFF);
                    uint8 Alpha2 = CLAMP((sint32(LocalFrame) - 120) * 5, 0, 0xFF);

                    constexpr sint32 YPosition = -30;

                    if (LocalFrame > 300)
                    {
                        Alpha0 = Alpha1 = Alpha2 = CLAMP(0xFF - (sint32(LocalFrame)-300) * 5, 0, 0xFF);
                    }

                    if (Alpha0)
                    {
                        // PrintString2XBlend("Addict labs", 11, (160-(5*16+8))+2,  YPosition + 55+2, 0, Alpha0 >> 1);
                        // PrintString2XBlend("Addict labs", 11,  160-(5*16+8),     YPosition + 55,   0xFFFF, Alpha0);

                        PrintString2XBlend   ("Addict labs", 11, (160-(5*16+8))+2,  YPosition + 55+2, 0, Alpha0 >> 1);
                        PrintString2XBlendTex("Addict labs", 11,  160-(5*16+8),     YPosition + 55,   GetData_FontGradientLarge(), Alpha0);
                    }

                    if (Alpha1)
                    {
                        // PrintStringBlend("presents",     8, (160-(4*8))+1,     YPosition + 80+1, 0, Alpha1 >> 1);
                        // PrintStringBlend("presents",     8,  160-(4*8),        YPosition + 80,   0xFFFF, Alpha1);
                         
                        PrintStringBlend   ("presents",     8, (160-(4*8))+1,     YPosition + 80+1, 0, Alpha1 >> 1);
                        PrintStringBlendTex("presents",     8,  160-(4*8),        YPosition + 80,   GetData_FontGradientSmall(), Alpha1);
                    }

                    if (Alpha2)
                    {
                        BlitImage((320 - Img_Logo_Width) / 2, YPosition + 98, Alpha2, GetData_Logo(), Img_Logo_Width, Img_Logo_Height, true);
                    }
                }

                if (LocalFrame > 350 && LocalFrame < 660)
                {
                    const char *CRTexts[] =
                    {
                        { "code & 3d" },
                        { "mx" },
                        { "rts" },
                        { "" },

                        { "gfx" },
                        { "suule" },
                        { "" },

                        { "music" },
                        { "xtd" },
                    };

                    const sint32 CRFrame[] =
                    {
                        350, 440+60,
                        350, 460+60,
                        350, 460+60,
                        350, 460+60,

                        375, 465+60,
                        375, 485+60,
                        375, 485+60,

                        400, 490+60,
                        400, 510+60,
                    };

                    sint32 Offset = 20;

                    for (uint32 i = 0; i < sizeof(CRTexts) / sizeof(CRTexts[0]); i++)
                    {
                        uint32 Len = strlen(CRTexts[i]);

                        if (Len)
                        {
                            sint32 CRS = CRFrame[i*2+0];
                            sint32 CRE = CRFrame[i*2+1];

                            uint8 Alpha = CLAMP((sint32(LocalFrame) - CRS) * 5, 0, 0xFF);

                            if (sint32(LocalFrame) > CRE)
                                Alpha = CLAMP(0xFF - (sint32(LocalFrame)-CRE) * 5, 0, 0xFF);

                            if (Alpha)
                            {
                                PrintString2XBlend   (CRTexts[i], Len, (160-(Len * 8)) + 6, Offset + 2, 0,                           Alpha >> 1);
                                PrintString2XBlendTex(CRTexts[i], Len, (160-(Len * 8)) + 4, Offset,     GetData_FontGradientLarge(), Alpha     );
                            }

                            Offset += 20;
                        }
                        else
                        {
                            Offset += 10;
                        }
                    }
                }
            }
        break;

        case 2:
            {
                const uint32  LocalFrame = OveralFrame - DemoPart01Begin;
                const FScene *Scene = Get_Scene01();

                Result = MakePostEffect(EPostEffect::BlurFade, 0u);

                Rasterizer->SetDefaultState();
                Rasterizer->SetState_ClearColor(true, 0, [](uint16 Frame, const FVector3D& CamDir, const FVector3D& CamUp, const FMatrix& CamMat, const FMatrix& ProjMat, const FMatrix& CamProjMat)
                {
                    GradientFill(FColor16(uint8(0x0F), uint8(0x0F), uint8(0x1F)), FColor16(uint8(0x50), uint8(0x50), uint8(0x80)));
                    return;
                });

                Rasterizer->SetState_ClearDepth(true);
                Rasterizer->SetState_AmbientLight(Scalar(0.0625f));

                uint8 EfxStage = LocalFrame > 360 ? 1 : 0;
                
                switch (EfxStage)
                {
                    case 0:
                        {
                            const uint32 CurrentFrame = LocalFrame;

                            FVector3D LightVector = FVector3D(-1, 1, 1);
                            LightVector.Normalize();
                            Rasterizer->SetState_DirectionalLight(true, LightVector);

                            DBGM(CurrentFrame, "Cam0");
                            DBGS("Scene01");
                            Rasterizer->Render(Scene, CurrentFrame, Scene01_Camera_Cam0);

                            if (CurrentFrame < 30)
                                Result = MakePostEffect(EPostEffect::BlurFade, CLAMP((30-CurrentFrame) << 3, 0u, 0xFFu));

                            if (CurrentFrame > 345)
                                Result = MakePostEffect(EPostEffect::BlurFade, CLAMP((CurrentFrame-345) << 4, 0u, 0xFFu));
                        }
                    break;

                    case 1:
                        {
                            const uint32 CurrentFrame = LocalFrame - 440 + 570 + 80;

                            FVector3D LightVector = FVector3D(1, 2, -2);
                            LightVector.Normalize();
                            Rasterizer->SetState_DirectionalLight(true, LightVector);

                            DBGM(CurrentFrame, "Cam0");
                            DBGS("Scene01");
                            Rasterizer->Render(Scene, CurrentFrame, Scene01_Camera_Cam0);

                            if (LocalFrame < 375)
                                Result = MakePostEffect(EPostEffect::BlurFade, CLAMP((375-LocalFrame) << 4, 0u, 0xFFu));

                            if (LocalFrame > 520)
                                Result = MakePostEffect(EPostEffect::BlurFade, CLAMP((sint32(LocalFrame)-520) << 3, 0, 0xFF));
                        }
                    break;
                }
            }
        break;

        case 3:
            {
                const uint32 LocalFrame = OveralFrame - DemoPart02Begin;
                const FScene *Scene = Get_Scene02();

                Rasterizer->SetDefaultState();
                Rasterizer->SetState_ClearColor(true, 0, [](uint16 Frame, const FVector3D& CamDir, const FVector3D& CamUp, const FMatrix& CamMat, const FMatrix& ProjMat, const FMatrix& CamProjMat)
                {
                    // Offsets from camera direction 

                    const uint32 XOff = uint32(CamDir.x * Scalar(80));
                    const uint32 YOff = uint32(CamDir.z * Scalar(1000));
                    const uint8  MonoTime = uint8(Frame);

                    // Wrap (divided between two cores)

                    struct FLocalData
                    {
                        uint8        MonoTime;
                        uint32       XOff, YOff;
                    };

                    FLocalData *Local = (FLocalData *)RasterizerGetInSpace();
                    Local->MonoTime = MonoTime;
                    Local->XOff     = XOff;
                    Local->YOff     = YOff;

                    PushSecondCoreWork([](const void* Arg)
                    {
                        const FLocalData *Local = (FLocalData *)Arg;
                        DoWrapPart(GetData_WrapClouds(), RasterizerFramebuffer, WrapSinTable32, Local->MonoTime, Local->XOff, Local->YOff, 0, RasterizerSizeY>>1);
                        return;
                    });

                    DoWrapPart(GetData_WrapClouds(), RasterizerFramebuffer + (RasterizerSizeY>>1) * RasterizerSizeX, WrapSinTable32, MonoTime, XOff, YOff, RasterizerSizeY>>1, RasterizerSizeY);
                    FlushWorkItems();
                    return;
                });
                Rasterizer->SetState_ClearDepth(true);
                Rasterizer->SetState_AmbientLight(Scalar(0.125f));
                Rasterizer->SetState_CameraCallback([](uint16 Frame, FVector3D &CamPos, FVector3D &CamDir, FVector3D &CamUp, Scalar &Aspect, uint8 &FOV, uint16 &Near, uint16 &Far)
                {
                    FSoftwareRasterizer *Rasterizer = FSoftwareRasterizer::Get();
                    Rasterizer->SetState_PointLightLight(true, CamPos + CamDir * Scalar(14), Scalar(12));
                });
                Rasterizer->SetState_PostRender(false, [](uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat)
                {
                    DrawMotoBoardBi(24 + Img_MotoMask_Width/2, 180, CLAMP(CamUp.z * 1.33f, -0.3f, 0.3f), GetData_MotoMask(), Img_MotoMask_Width, Img_MotoMask_Height);
                    return;
                });

                uint8 EfxStage = 0;

                if (LocalFrame > 500)
                    EfxStage++;

                if (LocalFrame > 600)
                    EfxStage++;

                switch (EfxStage)
                {
                    case 0:
                        {
                            const uint32 CurrentFrame = LocalFrame;

                            FVector3D LightVector = FVector3D(0, 1, -3);
                            LightVector.Normalize();
                            Rasterizer->SetState_DirectionalLight(true, LightVector);

                            DBGM(CurrentFrame, "Cam2");
                            DBGS("Scene02");
                            Rasterizer->Render(Scene, CurrentFrame, Scene02_Camera_Cam2);

                            if (CurrentFrame < 30)
                                Result = MakePostEffect(EPostEffect::FadeToBlack, CLAMP((30-CurrentFrame) << 3, 0u, 0xFFu));

                            if (CurrentFrame > 484)
                                Result = MakePostEffect(EPostEffect::FadeToColor, CLAMP((CurrentFrame-484) << 4, 0u, 0xFFu), 0xFFFF);
                        }
                    break;

                    case 1:
                        {
                            const uint32 CurrentFrame = LocalFrame - 500;
                            DBGM(CurrentFrame, "Cam8");
                            DBGS("Scene02");

                            memset(RasterizerFramebuffer, 0xFFFFFFFF, RasterizerSizeX*RasterizerSizeY*sizeof(uint16));

                            uint8 Alpha = 0xFF;

                            if (CurrentFrame < 16)
                                Alpha = CLAMP(CurrentFrame << 4, 0u, 0xFFu);

                            if (CurrentFrame > 84)
                                Alpha = CLAMP((100-CurrentFrame) << 4, 0u, 0xFFu);

                            if (CurrentFrame < 32)
                            {
                                __restrict uint16       *dst = RasterizerFramebuffer;
                                __restrict const uint16 *src = GetData_SplashImage();

                                const sint32 WrapAlpha = 0xFF - CLAMP(CurrentFrame << 3, 0u, 0xFFu);
                                const sint32 al        = sint32(Alpha);

                                for (uint32 y = 0; y < RasterizerSizeY; y++)
                                {
                                    sint32 yWarp = sint8(WrapSinTable32[(y + CurrentFrame) & 0xFF]);
                                           yWarp = (yWarp * WrapAlpha) >> 8;

                                    for (uint32 x = 0; x < RasterizerSizeX; x++, dst++)
                                    {
                                        sint32 xWarp = sint8(WrapSinTable32[(x + CurrentFrame) & 0xFF]);
                                               xWarp = (xWarp * WrapAlpha) >> 8;

                                        sint32 sx = CLAMP<sint32>(x + yWarp, 0, RasterizerSizeX);
                                        sint32 sy = CLAMP<sint32>(y + xWarp, 0, RasterizerSizeY);

                                        const uint16 Color = src[sx + sy * RasterizerSizeX];
				                        const uint16 dc    = *dst;

				                        const sint32 cr =  Color >> 11;
				                        const sint32 cg = (Color >> 5) & 0x3F;
				                        const sint32 cb =  Color & 0x1F;
				
				                        const sint32 dr =  dc >> 11;
				                        const sint32 dg = (dc >> 5) & 0x3F;
				                        const sint32 db =  dc & 0x1F;

                                        *dst = ((dr + (((cr - dr) * al) >> 8)) << 11) |
					                           ((dg + (((cg - dg) * al) >> 8)) <<  5) |
					                            (db + (((cb - db) * al) >> 8))        ;
                                    }
                                }  
                            }
                            else
                            {
                                BlitImage(0, 0, Alpha, GetData_SplashImage(), Img_SplashImage_Width, Img_SplashImage_Height, false);
                            }
                        }
                    break;

                    case 2:
                        {
                            const uint32 CurrentFrame = LocalFrame - 600;

                            FVector3D LightVector = FVector3D(0, 1, 3);
                            LightVector.Normalize();
                            Rasterizer->SetState_DirectionalLight(true, LightVector);

                            DBGM(CurrentFrame, "Cam3");
                            DBGS("Scene02");
                            Rasterizer->Render(Scene, CurrentFrame + 60, Scene02_Camera_Cam3);

                            if (CurrentFrame < 16)
                                Result = MakePostEffect(EPostEffect::FadeToColor, CLAMP((16-CurrentFrame) << 5, 0u, 0xFFu), 0xFFFF);

                            if (CurrentFrame > 395)
                                Result = MakePostEffect(EPostEffect::FadeToColor, CLAMP((sint32(CurrentFrame)-395) * 11, 0, 0xFF), 0xFFFF);
                        }
                    break;
                }
            }
        break;

        case 4:
            {
                const uint32 LocalFrame = OveralFrame - DemoPart03Begin;
                const FScene *Scene = Get_Scene03();

                FVector3D LightVector = FVector3D(1, 3, 1);
                LightVector.Normalize();

                Rasterizer->SetDefaultState();
                Rasterizer->SetState_ClearColor(true, 0, [](uint16 Frame, const FVector3D& CamDir, const FVector3D& CamUp, const FMatrix& CamMat, const FMatrix& ProjMat, const FMatrix& CamProjMat)
                {
                    GradientFill(FColor16(uint8(0xA0), uint8(0x80), uint8(0x80)), FColor16(uint8(0x30), uint8(0x30), uint8(0x80)));
                    return;
                });
                Rasterizer->SetState_ClearDepth(true);
                Rasterizer->SetState_AmbientLight(Scalar(0.25f));
                Rasterizer->SetState_DirectionalLight(true, LightVector, FixedHalf);
                Rasterizer->SetState_PointLightLight(true, FVector3D(FixedZero, Scalar(6.6f), FixedZero), Scalar(16), Scalar(1));

                Rasterizer->SetState_SetTextureCallback([](uint16 Frame, uint8 MID, uint16 *Dst)
                {
                    if (MID == Scene03_Material_MatTexCrystaline)
                        return GetData_CyberViolet();

                    if (MID == Scene03_Material_MatTexObelisk)
                        return GetData_CyberBlue();

                    return GetData_CyberGray();
                });

                DBGM(LocalFrame, "Cam0");
                DBGS("Scene03");
                Rasterizer->Render(Scene, LocalFrame + 80, Scene03_Camera_Cam0);

                if (LocalFrame < 30)
                    Result = MakePostEffect(EPostEffect::FadeToColor, CLAMP((30-LocalFrame) << 3, 0u, 0xFFu), 0xFFFF);

                if (LocalFrame > 510)
                    Result = MakePostEffect(EPostEffect::FadeToColor, CLAMP((LocalFrame-510) << 3, 0u, 0xFFu), 0xFFFF);
            }
        break;

        case 5:
            {
                const uint32 LocalFrame = OveralFrame - DemoPart04Begin;
                const FScene *Scene = Get_Scene04();

                FVector3D LightVector = FVector3D(1, 2, -2);
                LightVector.Normalize();

                Rasterizer->SetDefaultState();
                Rasterizer->SetState_ClearColor(false, 0, nullptr);
                Rasterizer->SetState_ClearDepth(true);
                Rasterizer->SetState_AmbientLight(Scalar(0.0625f));
                Rasterizer->SetState_DirectionalLight(true, LightVector);
                Rasterizer->SetState_PostRender(true, [](uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat)
                {
                    struct FLocalData
                    {
                        uint16 Frame;
                    };

                    FLocalData *Local = (FLocalData *)RasterizerGetInSpace();

                    Local->Frame = Frame;

                    PushSecondCoreWork([](const void* Arg)
                    {
                        const FLocalData *Local = (FLocalData *)Arg;
                        DoTunelPart(Local->Frame, 100);
                        return;
                    });

                    DoTunelPart(Frame, 0);
                    return;
                });
             
                DBGM(LocalFrame, "Cam0");
                DBGS("Scene04");
                Rasterizer->Render(Scene, LocalFrame, Scene04_Camera_Cam0);

                if (LocalFrame < 32)
                    Result = MakePostEffect(EPostEffect::FadeToColor, CLAMP((32-LocalFrame) << 3, 0u, 0xFFu), 0xFFFF);

                if (LocalFrame > 570)
                    Result = MakePostEffect(EPostEffect::FadeToBlack, CLAMP((LocalFrame-570) << 4, 0u, 0xFFu), 0);
            }
        break;

        case 6:
            {
                const uint32 LocalFrame = OveralFrame - DemoPart05Begin;
                const FScene *Scene = Get_Scene05();

                // Recalculate metaballs grid

                PrepareMetaGrid(LocalFrame);

                // Render scene (and insert metaball triangles there)

                FVector3D LightVector = FVector3D(1, 1, 1);
                LightVector.Normalize();

                Rasterizer->SetDefaultState();
                Rasterizer->SetState_ClearColor(true, 0, [](uint16 Frame, const FVector3D& CamDir, const FVector3D& CamUp, const FMatrix& CamMat, const FMatrix& ProjMat, const FMatrix& CamProjMat)
                {
                    // Offsets from camera direction 

                    const uint32 XOff = uint32(CamDir.x * Scalar(80));
                    const uint32 YOff = uint32(CamDir.z * Scalar(100));
                    const uint8  MonoTime = uint8((Frame * 3) >> 1);

                    // Wrap (divided between two cores)

                    struct FLocalData
                    {
                        uint8        MonoTime;
                        uint32       XOff, YOff;
                    };

                    FLocalData *Local = (FLocalData *)RasterizerGetInSpace();
                    Local->MonoTime = MonoTime;
                    Local->XOff     = XOff;
                    Local->YOff     = YOff;

                    PushSecondCoreWork([](const void* Arg)
                    {
                        const FLocalData *Local = (FLocalData *)Arg;
                        DoWrapPart(GetData_WrapCyber(), RasterizerFramebuffer, WrapSinTable32, Local->MonoTime, Local->XOff, Local->YOff, 0, RasterizerSizeY>>1);
                        return;
                    });

                    DoWrapPart(GetData_WrapCyber(), RasterizerFramebuffer + (RasterizerSizeY>>1) * RasterizerSizeX, WrapSinTable32, MonoTime, XOff, YOff, RasterizerSizeY>>1, RasterizerSizeY);
                    FlushWorkItems();
                    return;
                });
                Rasterizer->SetState_ClearDepth(true);
                Rasterizer->SetState_AmbientLight(Scalar(0.0625f));
                Rasterizer->SetState_DirectionalLight(true, LightVector);
                Rasterizer->SetState_PostRender(true, [](uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat)
                {
                    FSoftwareRasterizer *Rasterizer = FSoftwareRasterizer::Get();

                    FVector3D LightDir = Rasterizer->GetDirectionalDirection();
                              LightDir = LightDir - CamDir;
                              LightDir.Normalize();

                    ExtractMetaTriangles(Frame, CamProjMat, LightDir, FixedZero);
                    return;
                });

                DBGM(LocalFrame, "Cam0");
                DBGS("Scene05");
                Rasterizer->Render(Scene, LocalFrame, Scene05_Camera_Cam0);

                if (LocalFrame < 30)
                    Result = MakePostEffect(EPostEffect::FadeToBlack, CLAMP((30-LocalFrame) << 3, 0u, 0xFFu), 0);

                if (LocalFrame > 570)
                    Result = MakePostEffect(EPostEffect::FadeToBlack, CLAMP((LocalFrame-570) << 3, 0u, 0xFFu), 0);

                DrawTwisterY(LocalFrame);
            }
        break;

        case 7:
            {
                const uint32 LocalFrame = OveralFrame - DemoPart06Begin;
                const uint8  MonoTime   = uint8(LocalFrame);

                DBGM(LocalFrame, "Cam0");
                DBGS("Scene06");

                // Wrap 

                TriggerColorWriteBarrier();
                DoWrapPart(GetData_WrapAbstractDark(), RasterizerFramebuffer, WrapSinTable32, MonoTime, 0, 0, 0, RasterizerSizeY);

                // Scroller

                const char *GRTexts[] =
                {
                    { "Greetingz" },
                    { "" },
                    { "ABERRATION CREATIONS" },
                    { "ALTAIR" },
                    { "ANADUNE" },
                    { "ANDROMEDA" },
                    { "CNCD" },
                    { "COCOON" },
                    { "CONSPIRACY" },
                    { "ELUDE" },
                    { "ELYSIUM" },
                    { "FAIRLIGHT" },
                    { "FAITH DESIGN" },
                    { "FARBRAUSCH" },
                    { "FLOPPY" },
                    { "FUTURIS" },
                    { "GHOSTOWN" },
                    { "HAUJOBB" },
                    { "LAMERS" },
                    { "MADWIZARDS" },
                    { "MERCURY" },
                    { "MYSTIC BYTES" },
                    { "NAH KOLOR" },
                    { "THE BLACK LOTUS" },
                    { "THELO0P" },
                    { "WANTED TEAM" },
                    { "" },
                    { "" },
                    { "" },
                    { "" },
                    { "" },
                    { "" },
                    { "" },
                    { "" },
                    { "ALL XENIUM PARTISANS" },
                    { "AND YOU!" },
                };

                sint32 Offset = 250 - MIN(LocalFrame, 425u) * 2;
                uint16 Color  = 0xFFFF;
                
                for (uint32 i = 0; i < sizeof(GRTexts) / sizeof(GRTexts[0]); i++)
                {
                    uint32 Len = strlen(GRTexts[i]);

                    if (Len)
                    {
                        sint32 FinY = Offset + i * 20;
                        uint8 Alpha = 0xFF;

                        if (FinY < 32)
                            Alpha = CLAMP(MAX(FinY, 0)<<3, 0, 0xFF);

                        if (FinY > (200 - 32))
                            Alpha = CLAMP(MAX(200 - FinY, 0)<<3, 0, 0xFF);

                        PrintString2XBlend   (GRTexts[i], Len, (160-(Len * 8)) + 6, FinY + 2, 0,                           Alpha >> 1);
                        PrintString2XBlendTex(GRTexts[i], Len, (160-(Len * 8)) + 4, FinY,     GetData_FontGradientLarge(), Alpha     );
                    }
                }

                Result = MakePostEffect(EPostEffect::Blur, 0xFF, 0u);

                if (LocalFrame < 30)
                    Result = MakePostEffect(EPostEffect::BlurFade, CLAMP((30-LocalFrame) << 3, 0u, 0xFFu), 0);

                if (LocalFrame > 500)
                    Result = MakePostEffect(EPostEffect::BlurFade, CLAMP((LocalFrame-500) << 2, 0u, 0xFFu));
            }
        break;
    }

    return Result;
}

void DemoInit(void (*SecondCoreFunction)(void))
{
    RasterizerSpawnSecondCoreFunction(SecondCoreFunction);
    FSoftwareRasterizer::Get()->SetDefaultState();
    return;
}

void DemoDeinit()
{
    return;
}
