//
// ST7789 Display driver
// Modified implementation, based on pico SDK
//

#if !PICO_DO_VGA 

#include "st7789.hpp"

namespace pimoroni 
{
	static uint8_t  madctl;
	static uint16_t caset[2] = { 0, 0 };
	static uint16_t raset[2] = { 0, 0 };

	enum MADCTL : uint8_t 
	{
		ROW_ORDER = 0b10000000,
		COL_ORDER = 0b01000000,
		SWAP_XY = 0b00100000,  // AKA "MV"
		SCAN_ORDER = 0b00010000,
		RGB_BGR = 0b00001000,
		HORIZ_ORDER = 0b00000100
	};

	enum reg 
	{
		SWRESET = 0x01,
		TEOFF = 0x34,
		TEON = 0x35,
		MADCTL = 0x36,
		COLMOD = 0x3A,
		RAMCTRL = 0xB0,
		GCTRL = 0xB7,
		VCOMS = 0xBB,
		LCMCTRL = 0xC0,
		VDVVRHEN = 0xC2,
		VRHS = 0xC3,
		VDVS = 0xC4,
		FRCTRL2 = 0xC6,
		PWCTRL1 = 0xD0,
		PORCTRL = 0xB2,
		GMCTRP1 = 0xE0,
		GMCTRN1 = 0xE1,
		INVOFF = 0x20,
		SLPOUT = 0x11,
		DISPON = 0x29,
		GAMSET = 0x26,
		DISPOFF = 0x28,
		RAMWR = 0x2C,
		INVON = 0x21,
		CASET = 0x2A,
		RASET = 0x2B,
		PWMFRSEL = 0xCC
	};

	void ST7789::common_init() 
	{
		gpio_set_function(dc, GPIO_FUNC_SIO);
		gpio_set_dir(dc, GPIO_OUT);

		gpio_set_function(cs, GPIO_FUNC_SIO);
		gpio_set_dir(cs, GPIO_OUT);

		// if a backlight pin is provided then set it up for
		// pwm control
		if (bl != PIN_UNUSED) 
		{
			pwm_config cfg = pwm_get_default_config();
			pwm_set_wrap(pwm_gpio_to_slice_num(bl), 65535);
			pwm_init(pwm_gpio_to_slice_num(bl), &cfg, true);
			gpio_set_function(bl, GPIO_FUNC_PWM);
			set_backlight(0); // Turn backlight off initially to avoid nasty surprises
		}

		command(reg::SWRESET);

		sleep_ms(150);

		// Common init
		command(reg::TEON);  // enable frame sync signal if used
		command(reg::COLMOD, 1, "\x05");  // 16 bits per pixel

		command(reg::PORCTRL, 5, "\x0c\x0c\x00\x33\x33");
		command(reg::LCMCTRL, 1, "\x2c");
		command(reg::VDVVRHEN, 1, "\x01");
		command(reg::VRHS, 1, "\x12");
		command(reg::VDVS, 1, "\x20");
		command(reg::PWCTRL1, 2, "\xa4\xa1");
		command(reg::FRCTRL2, 1, "\x0f");

		// As noted in https://github.com/pimoroni/pimoroni-pico/issues/1040
		// this is required to avoid a weird light grey banding issue with low brightness green.
		// The banding is not visible without tweaking gamma settings (GMCTRP1 & GMCTRN1) but
		// it makes sense to fix it anyway.
		command(reg::RAMCTRL, 2, "\x00\xc0");
		command(reg::GCTRL, 1, "\x35");
		command(reg::VCOMS, 1, "\x1f");
		command(reg::GMCTRP1, 14, "\xD0\x08\x11\x08\x0C\x15\x39\x33\x50\x36\x13\x14\x29\x2D");
		command(reg::GMCTRN1, 14, "\xD0\x08\x10\x08\x06\x06\x39\x44\x51\x0B\x16\x14\x2F\x31");
		command(reg::INVON);   // set inversion mode
		command(reg::SLPOUT);  // leave sleep mode
		command(reg::DISPON);  // turn display on

		sleep_ms(100);

		configure_display();

		if (bl != PIN_UNUSED) 
		{
			sleep_ms(50); // Wait for the update to apply
			set_backlight(255); // Turn backlight on now surprises have passed
		}

		return;
	}

	void ST7789::cleanup() 
	{
		if (dma_channel_is_claimed(st_dma)) 
		{
			dma_channel_abort(st_dma);
			dma_channel_unclaim(st_dma);
		}

		return;
	}

	void ST7789::configure_display() 
	{
		caset[0] = 0;
		caset[1] = 319;
		raset[0] = 0;
		raset[1] = 239;
		madctl = MADCTL::COL_ORDER;
		madctl |= MADCTL::SWAP_XY | MADCTL::SCAN_ORDER;

		// Byte swap the 16bit rows/cols values
		caset[0] = __builtin_bswap16(caset[0]);
		caset[1] = __builtin_bswap16(caset[1]);
		raset[0] = __builtin_bswap16(raset[0]);
		raset[1] = __builtin_bswap16(raset[1]);

		command(reg::CASET, 4, (char*)caset);
		command(reg::RASET, 4, (char*)raset);
		command(reg::MADCTL, 1, (char*)&madctl);

		return;
	}

	void ST7789::command(uint8_t command, size_t len, const char* data) 
	{
		gpio_put(dc, 0); // command mode
		gpio_put(cs, 0);

		spi_write_blocking(spi, &command, 1);

		if (data) 
		{
			gpio_put(dc, 1); // data mode
			spi_write_blocking(spi, (const uint8_t*)data, len);
		}

		gpio_put(cs, 1);
		return;
	}

	uint32_t ST7789::initialize(uint32_t SpiSpeed) 
	{
		if (SpiSpeed == 0)
			SpiSpeed = SPI_BAUD;
		spi_init(spi, SpiSpeed);
		SpiSpeed = spi_set_baudrate(spi, SpiSpeed);

		gpio_set_function(wr_sck, GPIO_FUNC_SPI);
		gpio_set_function(d0, GPIO_FUNC_SPI);

		st_dma = dma_claim_unused_channel(true);
		dma_channel_config config = dma_channel_get_default_config(st_dma);
		channel_config_set_transfer_data_size(&config, DMA_SIZE_8);
		channel_config_set_bswap(&config, false);
		channel_config_set_read_increment(&config, true);
		channel_config_set_write_increment(&config, false);
		channel_config_set_dreq(&config, spi_get_dreq(spi, true));
		dma_channel_configure(st_dma, &config, &spi_get_hw(spi)->dr, nullptr, width * height * sizeof(uint16_t), false);

		common_init();
		return SpiSpeed;
	}

	void ST7789::setWindow(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1) 
	{
		if (caset[0] == __builtin_bswap16(x0) &&
			caset[1] == __builtin_bswap16(x1) &&
			raset[0] == __builtin_bswap16(y0) &&
			raset[1] == __builtin_bswap16(y1)  )
			return;

		caset[0] = x0;
		caset[1] = x1;
		raset[0] = y0;
		raset[1] = y1;

		caset[0] = __builtin_bswap16(caset[0]);
		caset[1] = __builtin_bswap16(caset[1]);
		raset[0] = __builtin_bswap16(raset[0]);
		raset[1] = __builtin_bswap16(raset[1]);

		command(reg::CASET, 4, (char*)caset);
		command(reg::RASET, 4, (char*)raset);

		while (dma_channel_is_busy(st_dma));

		dma_channel_config config = dma_channel_get_default_config(st_dma);
		channel_config_set_transfer_data_size(&config, DMA_SIZE_8);
		channel_config_set_bswap(&config, false);
		channel_config_set_read_increment(&config, true);
		channel_config_set_write_increment(&config, false);
		channel_config_set_dreq(&config, spi_get_dreq(spi, true));
		dma_channel_configure(st_dma, &config, &spi_get_hw(spi)->dr, nullptr, uint32_t((x1-x0)+1) * uint32_t((y1-y0)+1) * sizeof(uint16_t), false);

		return;
	}

	void ST7789::update(const uint16_t* data, bool DMA, bool Finish) 
	{
		uint8_t cmd = reg::RAMWR;

		setWindow();

		if (DMA)
		{
			gpio_put(dc, 0); // command mode
			gpio_put(cs, 0);
			spi_write_blocking(spi, &cmd, 1);
			gpio_put(dc, 1); // data mode

			while (dma_channel_is_busy(st_dma));
			dma_channel_set_read_addr(st_dma, (const uint8_t*)data, true);

			if (Finish)
			{
				dma_channel_wait_for_finish_blocking(st_dma);
				gpio_put(cs, 1);
			}
		}
		else
		{
			command(cmd, width * height * sizeof(uint16_t), (const char*)data);
		}

		return;
	}

	void ST7789::update(const uint16_t* data, uint16_t StartScanline, uint16_t NumScanlines, bool DMA, bool Finish)
	{
		uint8_t cmd = reg::RAMWR;

		setWindow(0, StartScanline, width-1, StartScanline + NumScanlines-1);

		if (DMA)
		{
			gpio_put(dc, 0); // command mode
			gpio_put(cs, 0);
			spi_write_blocking(spi, &cmd, 1);
			gpio_put(dc, 1); // data mode

			while (dma_channel_is_busy(st_dma));
			dma_channel_set_read_addr(st_dma, (const uint8_t*)data, true);

			if (Finish)
			{
				dma_channel_wait_for_finish_blocking(st_dma);
				gpio_put(cs, 1);
			}
		}
		else
		{
			command(cmd, width * NumScanlines * sizeof(uint16_t), (const char*)data);
		}

		return;
	}

	void ST7789::endupdate()
	{
		dma_channel_wait_for_finish_blocking(st_dma);
		gpio_put(cs, 1);
		return;
	}

	void ST7789::set_backlight(uint8_t brightness) 
	{
		uint16_t value = uint16_t(brightness) << 8; // NOTE::Display gamma is 2.8, so this value should be gamma correctd to take this into account...
		pwm_set_gpio_level(bl, value);
		return;
	}
}

#endif // PICO_DO_VGA 