//
// ST7789 Display driver
// Modified implementation, based on pico SDK
//

#pragma once

#include "hardware/spi.h"
#include "hardware/dma.h"
#include "hardware/gpio.h"
#include "hardware/pio.h"
#include "hardware/pwm.h"
#include "hardware/clocks.h"
#include "pimoroni_common.hpp"
#include "pimoroni_bus.hpp"

namespace pimoroni 
{
	class PicoDisplay2
	{
	public:
		static const int WIDTH = 320;
		static const int HEIGHT = 240;
		static const uint8_t A = 12;
		static const uint8_t B = 13;
		static const uint8_t X = 14;
		static const uint8_t Y = 15;
		static const uint8_t LED_R = 6;
		static const uint8_t LED_G = 7;
		static const uint8_t LED_B = 8;
	};

	class ST7789
	{
		uint16_t width;
		uint16_t height;
		spi_inst_t* spi = PIMORONI_SPI_DEFAULT_INSTANCE;
		uint cs;
		uint dc;
		uint wr_sck;
		uint rd_sck = PIN_UNUSED;
		uint d0;
		uint bl;
		uint vsync = PIN_UNUSED; // only available on some products
		uint parallel_sm;
		PIO parallel_pio;
		uint parallel_offset;
		uint st_dma;

	public:

		// The ST7789 requires 16 ns between SPI rising edges.
		// 16 ns = 62,500,000 Hz
		// 2350 doesn't support 62,500,000 so use 75,000,000 seems to work.
#if !PICO_RP2350
		static const uint32_t SPI_BAUD = 62'500'000;
#else
		static const uint32_t SPI_BAUD = 75'000'000;
#endif

		// Serial init
		ST7789(SPIPins pins) : width(320), height(240), spi(pins.spi), cs(pins.cs), dc(pins.dc), wr_sck(pins.sck), d0(pins.mosi), bl(pins.bl) {}

		// Init
		uint32_t initialize(uint32_t SpiSpeed = 0);

		// Deinit
		void cleanup();

		// Set transfer window, it then expects 16bit RGB565 (BigEndian) of data with pitch == (((y1-y0)+1) * 2) bytes
		void setWindow(uint16_t x0 = 0, uint16_t y0 = 0, uint16_t x1 = 319, uint16_t y1 = 239);

		// Update whole display, pass Finish == true to actually synchronously wait for DMA to complete (if using DMA)
		// Format is RGB565 BigEndian
		void update(const uint16_t* data, bool DMA, bool Finish);

		// Update display starting at specific scanline, and for number of scanlines (it adjusts window), 
		// pass Finish == true to actually synchronously wait for DMA to complete (if using DMA)
		// Format is RGB565 BigEndian
		void update(const uint16_t* data, uint16_t StartScanline, uint16_t NumScanlines, bool DMA, bool Finish);

		// In case of Finish == false, synchronously wait for DMA to finish (so we could call it before we modify the framebuffer for next time)
		void endupdate();

		// Set backlight brightness (it's in gamma 2.8 space)
		void set_backlight(uint8_t brightness);

	private:
		void common_init();
		void configure_display();
		void command(uint8_t command, size_t len = 0, const char* data = nullptr);
	};
}
