// (c) MX^Add

#include "Exporter.h"
#include "BaseTypes/BaseTypes.h"
#include "SceneExporter.h"

#define STB_IMAGE_IMPLEMENTATION
#include "STB/stb_image.h"

#ifdef  IOS_REF
#undef  IOS_REF
#define IOS_REF (*(pManager->GetIOSettings()))
#endif

#pragma comment (lib, "libfbxsdk-md.lib") 
#pragma comment (lib, "libxml2-md.lib")
#pragma comment (lib, "zlib-md.lib")

static bool EndsWith(const std::string &filename, const std::string suffix) 
{
    if (filename.size() < suffix.size()) 
        return false;

    return std::equal(suffix.rbegin(), suffix.rend(), filename.rbegin(), [](char a, char b) 
        {
            return std::tolower(a) == std::tolower(b);
        });
}

static bool InitializeSdkObjects(FbxManager *&pManager, FbxScene *&pScene)
{
    pManager = FbxManager::Create();

    if( !pManager )
    {
        printf("Error: Unable to create FBX Manager!\n");
		return false;
    }
	
	printf("FBX SDK version %s\n", pManager->GetVersion());

	FbxIOSettings* ios = FbxIOSettings::Create(pManager, IOSROOT);
	pManager->SetIOSettings(ios);

	FbxString lPath = FbxGetApplicationDirectory();
	pManager->LoadPluginsDirectory(lPath.Buffer());

    pScene = FbxScene::Create(pManager, "Scene");

	if (!pScene)
    {
        printf("Error: Unable to create FBX scene!\n");
        return false;
    }

	return true;
}

static void DestroySdkObjects(FbxManager* pManager)
{
    if (pManager) 
		pManager->Destroy();
	return;
}

static bool LoadScene(FbxManager* pManager, FbxDocument* pScene, const char* pFilename)
{
    int  lFileMajor, lFileMinor, lFileRevision;
    int  lSDKMajor,  lSDKMinor,  lSDKRevision;
    int  lAnimStackCount;
    bool lStatus;

    // Get the file version number generate by the FBX SDK.
    FbxManager::GetFileFormatVersion(lSDKMajor, lSDKMinor, lSDKRevision);

    // Create an importer.
    FbxImporter* lImporter = FbxImporter::Create(pManager, "");

    // Initialize the importer by providing a filename.
    const bool lImportStatus = lImporter->Initialize(pFilename, -1, pManager->GetIOSettings());
                               lImporter->GetFileVersion(lFileMajor, lFileMinor, lFileRevision);

    if( !lImportStatus )
    {
        FbxString error = lImporter->GetStatus().GetErrorString();
        printf("Call to FbxImporter::Initialize() failed.\n");
        printf("Error returned: %s\n\n", error.Buffer());

        if (lImporter->GetStatus().GetCode() == FbxStatus::eInvalidFileVersion)
        {
            printf("FBX file format version for this FBX SDK is %d.%d.%d\n", lSDKMajor, lSDKMinor, lSDKRevision);
            printf("FBX file format version for file '%s' is %d.%d.%d\n\n", pFilename, lFileMajor, lFileMinor, lFileRevision);
        }

        return false;
    }

    printf("FBX file format version for this FBX SDK is %d.%d.%d\n", lSDKMajor, lSDKMinor, lSDKRevision);

    if (lImporter->IsFBX())
    {
        printf("FBX file format version for file '%s' is %d.%d.%d\n\n", pFilename, lFileMajor, lFileMinor, lFileRevision);

        // From this point, it is possible to access animation stack information without
        // the expense of loading the entire file.

        printf("Animation Stack Information\n");

        lAnimStackCount = lImporter->GetAnimStackCount();

        printf("    Number of Animation Stacks: %d\n", lAnimStackCount);
        printf("    Current Animation Stack: \"%s\"\n", lImporter->GetActiveAnimStackName().Buffer());
        printf("\n");

        for(int i = 0; i < lAnimStackCount; i++)
        {
            FbxTakeInfo* lTakeInfo = lImporter->GetTakeInfo(i);

            printf("    Animation Stack %d\n", i);
            printf("         Name: \"%s\"\n", lTakeInfo->mName.Buffer());
            printf("         Description: \"%s\"\n", lTakeInfo->mDescription.Buffer());
            printf("         Import Name: \"%s\"\n", lTakeInfo->mImportName.Buffer());
            printf("         Import State: %s\n", lTakeInfo->mSelect ? "true" : "false");
            printf("\n");
        }

        // Set the import states.
        IOS_REF.SetBoolProp(IMP_FBX_MATERIAL,        true);
        IOS_REF.SetBoolProp(IMP_FBX_TEXTURE,         true);
        IOS_REF.SetBoolProp(IMP_FBX_LINK,            true);
        IOS_REF.SetBoolProp(IMP_FBX_SHAPE,           true);
        IOS_REF.SetBoolProp(IMP_FBX_GOBO,            true);
        IOS_REF.SetBoolProp(IMP_FBX_ANIMATION,       true);
        IOS_REF.SetBoolProp(IMP_FBX_GLOBAL_SETTINGS, true);
    }

    // Import the scene.
    lStatus = lImporter->Import(pScene);
	if (!lStatus || (lImporter->GetStatus() != FbxStatus::eSuccess))
	{
		printf("********************************************************************************\n");
		if (lStatus)
		{
			printf("WARNING:\n");
			printf("   The importer was able to read the file but with errors.\n");
			printf("   Loaded scene may be incomplete.\n\n");
		}
		else
		{
			printf("Importer failed to load the file!\n\n");
		}

		if (lImporter->GetStatus() != FbxStatus::eSuccess)
			printf("   Last error message: %s\n", lImporter->GetStatus().GetErrorString());

		FbxArray<FbxString*> history;
		lImporter->GetStatus().GetErrorStringHistory(history);
		if (history.GetCount() > 1)
		{
			printf("   Error history stack:\n");
			for (int i = 0; i < history.GetCount(); i++)
			{
				printf("      %s\n", history[i]->Buffer());
			}
		}
		FbxArrayDelete<FbxString*>(history);
		printf("********************************************************************************\n");
	}
    else
    {
        printf("\nScene loaded.\n");
        printf("********************************************************************************\n");
    }

    // Destroy the importer.
    lImporter->Destroy();

    return lStatus;
}

static int SceneExporter(const char* SceneName, const char* OutputName, const char* VarsName, const char* VisName)
{
    FbxManager *SdkManager = nullptr;
    FbxScene   *Scene	   = nullptr;

    if (!InitializeSdkObjects(SdkManager, Scene))
    {
        DestroySdkObjects(SdkManager);
        return 0;
    }

    if (!LoadScene(SdkManager, Scene, SceneName))
    {
        DestroySdkObjects(SdkManager);
        return 0;
    }

    //
    // Triangulate scene and replace original meshes
    //
    FbxGeometryConverter *Converter = new FbxGeometryConverter(SdkManager);
    Converter->Triangulate(Scene, true);
    delete Converter;

    //
    // Export scene
    //
    FSceneExporter *Exporter = new FSceneExporter(Scene, VisName);
    Exporter->Save(OutputName, VarsName);
    delete Exporter;

    printf("********************************************************************************\n");
    printf("All done!\n");
    DestroySdkObjects(SdkManager);
    return 1;
}

static int ImageExporter(const char* ImageName, const char* OutputName, const char* VarsName, const char* Compression)
{
    sint32 width = 0, height = 0, channels = 0;
    const uint8 *data = stbi_load(ImageName, &width, &height, &channels, 3);

    if (!data || width <= 0 || height <= 0)
    {
        printf("\033[31mERROR::Could not open [%s] as image !\033[0m\n", ImageName);
        return 0;
    }

    printf("Image [%s] [%ix%ix%i]\n", ImageName, width, height, channels);

    std::string SourceFilename = OutputName;
    std::string HeaderFilename = OutputName;
    {
        size_t p = HeaderFilename.rfind(".cpp");

        if (p != std::string::npos && p + 4 == HeaderFilename.size())
            HeaderFilename.replace(p, 4, ".h");
        else
            HeaderFilename += ".h";
    }

    FILE *SourceFile = fopen(SourceFilename.c_str(), "wt");
    FILE *HeaderFile = fopen(HeaderFilename.c_str(), "wt");

    if (!SourceFile || !HeaderFile)
    {
        if (SourceFile) fclose(SourceFile);
        if (HeaderFile) fclose(HeaderFile);

        printf("\033[31mERROR::Could not open target files for write !\033[0m\n");
        return 0;
    }

    fprintf(HeaderFile, "// Image export for [%s]\n\n", VarsName);
    fprintf(HeaderFile, "extern const uint16 *GetData_%s();\n\n", MakeSafeName(VarsName).c_str());
    fprintf(HeaderFile, "constexpr uint16 Img_%s_Width = %i;\n", MakeSafeName(std::string(VarsName)).c_str(), width);
    fprintf(HeaderFile, "constexpr uint16 Img_%s_Height = %i;\n", MakeSafeName(std::string(VarsName)).c_str(), height);

    fclose(HeaderFile);
    HeaderFile = nullptr;

    fprintf(SourceFile, "// Image export for [%s]\n\n#include \"BaseTypes/BaseTypes.h\"\n\n", VarsName);

    fprintf(SourceFile, "alignas(2) static const uint16 ImageData[%i] = { ", uint32(width*height));

    for (sint32 i = 0; i < width*height; i++)
    {
        const uint8 *rgb = &data[i*3];

        uint16 r = rgb[0]>>3;
        uint16 g = rgb[1]>>2;
        uint16 b = rgb[2]>>3;

        uint16 pixel = (r<<11) | (g<<5) | b;

        fprintf(SourceFile, "0x%04X, ", uint32(pixel));
    }
    fprintf(SourceFile, " };\n\n");

    fprintf(SourceFile, "const uint16 *GetData_%s()\n", MakeSafeName(VarsName).c_str());
    fprintf(SourceFile, "{\n");
    fprintf(SourceFile, "\treturn &ImageData[0];\n");
    fprintf(SourceFile, "}\n\n");

    fclose(SourceFile);
    SourceFile = nullptr;
    stbi_image_free((void *)data);
    printf("Saved image to [%s]\n\n", OutputName);
    return 1;
}

int main(int argc, char ** argv)
{
	if (argc < 4)
	{
		printf("%s <scenne.fbx> <outputname> <scenename> [<visname>]\n", argv[0]);
        printf("%s <image.png> <outputname> <imagename> [<compression=type (none, rle)>]\n", argv[0]);
		return 0;
	}

    if (EndsWith(argv[1], ".fbx"))
    {
	    const char *SceneName  = argv[1];
	    const char *OutputName = argv[2];
        const char *VarsName   = argv[3];
        const char *VisName    = argc > 4 ? argv[4] : nullptr;

        return SceneExporter(SceneName, OutputName, VarsName, VisName);
    }

    if (EndsWith(argv[1], ".png"))
    {
        const char *ImageName   = argv[1];
        const char *OutputName  = argv[2];
        const char *VarsName    = argv[3];
        const char *Compression = argc > 4 ? argv[4] : nullptr;

        return ImageExporter(ImageName, OutputName, VarsName, Compression);
    }

	return 0;
}
