// (c) MX^Add

#include "SceneExporter.h"
#include "SceneVisibilityData.h"

FSceneExporter::FSceneExporter(class FbxScene *FBXScene, const char *VisDataName)
{
	VisInfo = new FSceneVisibilityInfo(VisDataName);

	Cameras.reserve(32);
	Materials.reserve(64);
	MaterialsWithUVs.reserve(64);
	MaterialsWithADDs.reserve(64);
	MaterialsWithUNLITs.reserve(64);
	UniqueMeshes.reserve(128);

	FbxArray<FbxString*> TakeNames;
	FBXScene->FillAnimStackNameArray(TakeNames);
	FbxTakeInfo *TakeInfo = nullptr;
	std::string	FBXTakeName;

	if (!(TakeNames.GetCount() == 0 || TakeNames[0] == nullptr))
	{
		for (sint32 i = 0; i < TakeNames.GetCount(); i++)
		{
			const char *CTake = TakeNames[i]->Buffer();
			if ((FBXTakeName.length() == 0 && TakeInfo == nullptr) || strcmp(CTake, FBXTakeName.c_str()) == 0)
			{
				TakeInfo    = FBXScene->GetTakeInfo(*TakeNames[i]);
				FBXTakeName = CTake;
				break;
			}
		}

		TakeNames.Clear();
	}

	/*
	 * NOTE::We need this ?!
	 * 
	if (TakeInfo)
	{
		if (!FBXScene->SetTakeInfo(*TakeInfo))
			return;

		for (sint32 i = 0; i < FBXScene->GetSrcObjectCount<FbxAnimStack>(); i++)
		{
			FbxAnimStack *AnimStack = FBXScene->GetSrcObject<FbxAnimStack>(i);
			if (strcmp(AnimStack->GetName(), FBXTakeName.c_str()) == 0)
			{
				FBXScene->SetCurrentAnimationStack(AnimStack);
				break;
			}
		}
	}
	*/

	//
	// Fill frames info
	//
	if (TakeInfo)
	{
		StartTime    = float(TakeInfo->mLocalTimeSpan.GetStart().GetSecondDouble());
		EndTime	     = float(TakeInfo->mLocalTimeSpan.GetStop().GetSecondDouble());
	}
	else
	{
		StartTime    = 0.0f;
		EndTime	     = 1.0f;
	}

	printf("StartTime: %3.4f\n", StartTime);
	printf("EndTime  : %3.4f\n", EndTime);

	printf("Scene exporting...\n\n");

	Root = CreateObject(FBXScene->GetRootNode(), nullptr);

	printf("\nScene exported...\n\n");

	return;
}

FSceneExporter::~FSceneExporter()
{
	delete Root;
	delete VisInfo;
	return;
}

void FSceneExporter::FillMaterialFlags(const FbxSurfaceMaterial *Material) const
{ 
	const char *MaterialName = Material->GetName();

	if (Material)
	{
		FbxProperty prop = Material->GetFirstProperty();

		while (prop.IsValid()) 
		{
			if (prop.GetFlag(FbxPropertyFlags::eUserDefined)) 
			{
				FbxString propName = prop.GetName();

				//
				// Look for custom property called 'SRBlendModeAdd', if it's value is 1, then blend mode is additive
				//
				if (propName == "SRBlendModeAdd") 
				{
					sint32 BlendMode = prop.Get<FbxInt>();

					if (BlendMode == 1) 
					{
						AddAdditiveUnique(MaterialName);
					}
				}

				//
				// Look for custom property called 'SRUnlitMode', if it's value is 1, then material is unlit
				//
				if (propName == "SRUnlitMode") 
				{
					sint32 BlendMode = prop.Get<FbxInt>();

					if (BlendMode == 1) 
					{
						AddUnlitUnique(MaterialName);
					}
				}
			}

			prop = Material->GetNextProperty(prop);
		}
	}
	
	return;
}

void FSceneExporter::AddAdditiveUnique(const char* MaterialName) const
{
	for (size_t i = 0; i < MaterialsWithADDs.size(); i++)
	{
		if (MaterialsWithADDs[i] == MaterialName)
			return;
	}

	MaterialsWithADDs.push_back(MaterialName);
	return;
}

void FSceneExporter::AddUnlitUnique(const char* MaterialName) const
{
	for (size_t i = 0; i < MaterialsWithUNLITs.size(); i++)
	{
		if (MaterialsWithUNLITs[i] == MaterialName)
			return;
	}

	MaterialsWithUNLITs.push_back(MaterialName);
	return;
}

bool FSceneExporter::MaterialNeedsUV(const FbxSurfaceMaterial *Material) const
{
	const char *MaterialName = Material->GetName();

	for (size_t i = 0; i < MaterialsWithUVs.size(); i++)
	{
		if (MaterialsWithUVs[i] == MaterialName)
			return true;
	}

	if (Material && Material->GetClassId().Is(FbxSurfacePhong::ClassId))
	{
		//
		// We found a Phong material.
		//
		FbxSurfacePhong *Phong = (FbxSurfacePhong *)Material;

		FbxTexture *DiffTex = Phong->Diffuse.GetSrcObject<FbxTexture>();
		FbxFileTexture *FTex = FbxCast<FbxFileTexture>(DiffTex);
		if (FTex)
		{
			const char *RelName = FTex->GetRelativeFileName();
			if (RelName && RelName[0])
			{ 
				MaterialsWithUVs.push_back(MaterialName);
				return true;
			}
		}
	}
	else
	if (Material && Material->GetClassId().Is(FbxSurfaceLambert::ClassId))
	{
		//
		// We found a Lambert material.
		//
		FbxSurfaceLambert *Lambert = (FbxSurfaceLambert *)Material;
		
		FbxTexture *DiffTex = Lambert->Diffuse.GetSrcObject<FbxTexture>();
		FbxFileTexture *FTex = FbxCast<FbxFileTexture>(DiffTex);
		if (FTex)
		{
			const char *RelName = FTex->GetRelativeFileName();
			if (RelName && RelName[0])
			{ 
				MaterialsWithUVs.push_back(MaterialName);
				return true;
			}
		}
	}

	return false;
}

bool FSceneExporter::MaterialNeedsUV(const char* MaterialName) const
{
	for (size_t i = 0; i < MaterialsWithUVs.size(); i++)
	{
		if (MaterialsWithUVs[i] == MaterialName)
			return true;
	}

	return false;
}

bool FSceneExporter::MaterialIsAdditive(const char* MaterialName) const
{
	for (size_t i = 0; i < MaterialsWithADDs.size(); i++)
	{
		if (MaterialsWithADDs[i] == MaterialName)
			return true;
	}

	return false;
}

bool FSceneExporter::MaterialIsUnlit(const char* MaterialName) const
{
	for (size_t i = 0; i < MaterialsWithUNLITs.size(); i++)
	{
		if (MaterialsWithUNLITs[i] == MaterialName)
			return true;
	}

	return false;
}

bool FSceneExporter::MaterialNeedsUV(uint8 ID) const
{
	return MaterialNeedsUV(Materials[ID].c_str());
}

bool FSceneExporter::MaterialIsAdditive(uint8 ID) const
{
	return MaterialIsAdditive(Materials[ID].c_str());
}

bool FSceneExporter::MaterialIsUnlit(uint8 ID) const
{
	return MaterialIsUnlit(Materials[ID].c_str());
}

FGizmo::ENodeType FSceneExporter::IdentifyObject(class FbxNode* Node)
{
	if (Node->GetCamera())
		return FGizmo::ENodeType::Camera;

	if (Node->GetMesh())
		return FGizmo::ENodeType::Mesh;

	return FGizmo::ENodeType::Gizmo;
}

FExporterGizmo* FSceneExporter::CreateObject(class FbxNode* Node, FExporterGizmo *Parent)
{
	switch (IdentifyObject(Node))
	{ 
		case FGizmo::ENodeType::Gizmo:
			return new FExporterGizmo(this, Parent, Node, true);

		case FGizmo::ENodeType::Camera:
			return new FExporterCamera(this, Parent, Node);

		case FGizmo::ENodeType::Mesh:
		{
			if (Node->GetMaterialCount() <= 1)
			{
				FExporterMesh *Mesh = new FExporterMesh(this, Parent, Node, 0, true);

				if (Mesh->Vertices.size() == 0)
				{
					delete Mesh;
					return new FExporterGizmo(this, Parent, Node, true);
				}

				return Mesh;
			}
			else
			{
				FExporterGizmo *DummyParent = new FExporterGizmo(this, Parent, Node, true);

				for (sint32 i = 0; i < Node->GetMaterialCount(); i++)
				{
					FExporterMesh *Mesh = new FExporterMesh(this, DummyParent, Node, i, false);
					if (Mesh->Vertices.size() == 0)
					{
						delete Mesh;
					}
					else
					{
						DummyParent->Childs.push_back(Mesh);
					}
				}

				return DummyParent;
			}
		}
	}

	return nullptr;
}

uint8 FSceneExporter::AddUniqueMaterial(const std::string& Name)
{
	for (size_t i = 0; i < Materials.size(); i++)
	{
		if (Materials[i] == Name)
			return uint8(i);
	}

	Materials.push_back(Name);

	if (Materials.size() > 0x3F)
	{
		printf("\033[31mERROR::Exceeded allowed number of materials (63) - exported scene will be broken (material indices will overlap material flags) !\033[0m\n");
		assert(false);
	}

	return uint8(Materials.size()-1);
}

std::string FSceneExporter::GetMaterialName(uint8 ID) const
{ 
	if (ID > 0x3F || ID >= Materials.size())
		return "<Out of range !>";
	return Materials[ID];
}

void FSceneExporter::RegisterCamera(uint16 ID, std::string& Name)
{
	Cameras.push_back(std::pair<uint16, std::string>(ID, Name));
	return;
}

uint16 FSceneExporter::GetOffset(bool AlignPointer)
{
	size_t Size = Data.size() - sizeof(FScene);

	if (AlignPointer && (Size % 8))
	{
		while (Size % 8) 
		{ 
			Data.push_back(0);
			Size = Data.size() - sizeof(FScene);
		}
	}
	
	Size = Size >> 3;

	if (Size > 0xFFFF)
	{
		printf("\033[31mERROR::To much data while generating block !\033[0m\n");
		assert(false);
	}

	return uint16(Size);
}

void *FSceneExporter::SaveData(const void* DataPtr, size_t Size)
{
	const uint8 *Dpt = (const uint8 *)DataPtr;
	for (size_t i = 0; i < Size; i++)
		Data.push_back(Dpt ? Dpt[i] : 0);
	return &Data[Data.size()-Size];
}

static inline bool NEQ(const FMesh::FVertex &A, const FMesh::FVertex &B)
{ 
	FVector3D v = (A.Pos - B.Pos).Abs();
	if (v.x > 0.001f || v.y > 0.001f || v.z > 0.001f)
		return false;

	return A.Nrm == B.Nrm;
}
static inline bool NEQ(const FMesh::FVertexColor &A, const FMesh::FVertexColor &B)
{ 
	FVector3D v = (A.Pos - B.Pos).Abs();
	if (v.x > 0.001f || v.y > 0.001f || v.z > 0.001f)
		return false;

	return A.Nrm == B.Nrm && A.Clr == B.Clr;
}
static inline bool NEQ(const FMesh::FVertexUV &A, const FMesh::FVertexUV &B)
{ 
	FVector3D v = (A.Pos - B.Pos).Abs();
	if (v.x > 0.001f || v.y > 0.001f || v.z > 0.001f)
		return false;

	return A.Nrm == B.Nrm && A.UV == B.UV;
}

template < class T >
static bool AreMeshesEqual(const std::vector<uint16>& FacesA, const std::vector<uint16>& FacesB, const T* VerticesA, const T* VerticesB)
{
	std::vector<uint16> FC = FacesA;

	for (size_t i = 0; i < FC.size(); i += 3)
	{
		uint16 fa = FC[i+0];
		uint16 fb = FC[i+1];
		uint16 fc = FC[i+2];

		const T &Va = VerticesA[fa];
		const T &Vb = VerticesA[fb];
		const T &Vc = VerticesA[fc];

		bool VertexFound = false;

		for (size_t j = 0; j < FacesB.size(); j += 3)
		{
			uint16 ta = FacesB[j+0];
			uint16 tb = FacesB[j+1];
			uint16 tc = FacesB[j+2];

			const T &Qa = VerticesB[ta];
			const T &Qb = VerticesB[tb];
			const T &Qc = VerticesB[tc];

			if (NEQ(Va, Qa) && NEQ(Vb, Qb) && NEQ(Vc, Qc))
			{
				VertexFound = true;
				break;
			}
		}

		if (VertexFound)
		{
			FC.erase(FC.begin()+i+0, FC.begin()+i+2+1);
			i -= 3;
		}
	}

	return FC.empty();
}

const FSceneExporter::FUniqueMesh *FSceneExporter::FindUniqueMesh(FMesh::EVertexFormat Format, const std::vector<uint16>& Faces, const std::vector<uint8>& VertexesData) const
{
	if (VertexesData.empty())
		return nullptr;

	for (size_t MeshIndex = 0; MeshIndex < UniqueMeshes.size(); MeshIndex++)
	{
		const FUniqueMesh *UM = &UniqueMeshes[MeshIndex];

		if (UM->Format == Format && UM->Faces.size() == Faces.size())
		{
			switch (Format)
			{
				case FMesh::EVertexFormat::Simple:
					{
						const FMesh::FVertex *MeshAV = (FMesh::FVertex *)&UM->PayloadVertexes[0];
						const FMesh::FVertex *MeshBV = (FMesh::FVertex *)&VertexesData[0];

						if (AreMeshesEqual(UM->Faces, Faces, MeshAV, MeshBV))
							return UM;
					}
				break;

				case FMesh::EVertexFormat::Color:
					{
						const FMesh::FVertexColor *MeshAV = (FMesh::FVertexColor *)&UM->PayloadVertexes[0];
						const FMesh::FVertexColor *MeshBV = (FMesh::FVertexColor *)&VertexesData[0];

						if (AreMeshesEqual(UM->Faces, Faces, MeshAV, MeshBV))
							return UM;
					}
				break;

				case FMesh::EVertexFormat::UV:
					{
						const FMesh::FVertexUV *MeshAV = (FMesh::FVertexUV *)&UM->PayloadVertexes[0];
						const FMesh::FVertexUV *MeshBV = (FMesh::FVertexUV *)&VertexesData[0];

						if (AreMeshesEqual(UM->Faces, Faces, MeshAV, MeshBV))
							return UM;
					}
				break;
			}
		}
	}

	return nullptr;
}

void FSceneExporter::AddUniqueMesh(FMesh::EVertexFormat Format, const std::vector<uint16>& Faces, const std::vector<uint8>& VertexesData, uint16 VerticesOffset, uint16	FacesOffset)
{
	if (VertexesData.empty() || Faces.empty())
		return;

	FUniqueMesh UM;
	UM.Format = Format;
	UM.Faces  = Faces;
	UM.PayloadVertexes = VertexesData;
	UM.VerticesOffset = VerticesOffset;
	UM.FacesOffset = FacesOffset;

	UniqueMeshes.push_back(UM);
	return;
}

void FSceneExporter::Save(const char* Filename, const char *Scenename)
{
	if (Root)
	{ 
		/*
		printf("Scene structure:\n\n");
		Root->Print(1);
		printf("\n");
		*/

		for (sint32 i = 0; i < 8; i++)
		{
			Root->ReduceChilds(nullptr);
			Root->PostReduceCleanup();
			Root->SortChilds();
		}

		printf("Scene structure:\n\n");
		Root->Print(1);
		printf("\n");
	}

	printf("Saving scene to [%s]\n\n", Filename);

	Data.clear();
	Data.reserve(1024*1024);

	// Push scene
	{
		FScene Scene;
		SaveData(&Scene, sizeof(FScene));
	}

	FScene *ScenePtr = (FScene *)&Data[0];

	ScenePtr->NumFrames		  = uint16(GetAnimationFrames());
	ScenePtr->SceneRootOffset = GetOffset();

	Root->SaveData(this);

	printf("\n\033[32mTotal size of exported data = %i bytes (%i faces eliminated by vis).\033[0m\n", uint32(Data.size()), VisRemoved);

	std::string SourceFilename = Filename;
	std::string HeaderFilename = Filename;
	{
		size_t p = HeaderFilename.rfind(".cpp");

		if (p != std::string::npos && p + 4 == HeaderFilename.size())
			HeaderFilename.replace(p, 4, ".h");
		else
			HeaderFilename += ".h";
	}

	FILE *SourceFile = fopen(SourceFilename.c_str(), "wt");
	FILE *HeaderFile = fopen(HeaderFilename.c_str(), "wt");

	if (!SourceFile || !HeaderFile)
	{
		if (SourceFile) fclose(SourceFile);
		if (HeaderFile) fclose(HeaderFile);

		printf("\033[31mERROR::Could not open target files for write !\033[0m\n");
		return;
	}

	fprintf(HeaderFile, "// Scene export for [%s]\n\n", Scenename);
	fprintf(HeaderFile, "extern const FScene *Get_%s();\n\n", MakeSafeName(Scenename).c_str());

	for (size_t i = 0; i < Cameras.size(); i++)
		fprintf(HeaderFile, "constexpr uint16 %s = 0x%04X;\n", MakeSafeName(std::string(Scenename) + "_Camera_" + Cameras[i].second).c_str(), uint32(Cameras[i].first));

	if (Cameras.size())
		fprintf(HeaderFile, "\n");

	for (size_t i = 0; i < Materials.size(); i++)
		fprintf(HeaderFile, "constexpr uint8 %s = 0x%02X;\n", MakeSafeName(std::string(Scenename) + "_Material_" + Materials[i]).c_str(), uint32(i));

	fclose(HeaderFile);
	HeaderFile = nullptr;
	
	fprintf(SourceFile, "// Scene export for [%s]\n\n#include \"RendererTypes/Scene.h\"\n\n", Scenename);

	fprintf(SourceFile, "alignas(4) static const uint8 SceneData[%i] = { ", uint32(Data.size()));

	for (size_t i = 0; i < Data.size(); i++)
		fprintf(SourceFile, "0x%02X, ", uint32(Data[i]));
	fprintf(SourceFile, " };\n\n");

	fprintf(SourceFile, "const FScene *Get_%s()\n", MakeSafeName(Scenename).c_str());
	fprintf(SourceFile, "{\n");
	fprintf(SourceFile, "\treturn (const FScene *)&SceneData[0];\n");
	fprintf(SourceFile, "}\n\n");

	fclose(SourceFile);
	SourceFile = nullptr;
	printf("Saved scene to [%s]\n\n", Filename);
	return;
}
