// (c) MX^Add
#pragma once

#include "Exporter.h"
#include "SceneExporterGizmo.h"
#include "SceneExporterCamera.h"
#include "SceneExporterMesh.h"

class FSceneVisibilityInfo;

class FSceneExporter
{
protected:

	struct FUniqueMesh
	{
		FMesh::EVertexFormat Format;
		std::vector<uint16>  Faces;
		std::vector<uint8>   PayloadVertexes;
		uint16				 VerticesOffset;
		uint16				 FacesOffset;
	};

private:
	uint16										ID        = 1;
	FExporterGizmo							   *Root      = nullptr;
	float										StartTime = 0.0f;
	float										EndTime   = 0.0f;
	float										FPS       = 30.0f;
	std::vector<std::string>				    Materials;
	std::vector<std::pair<uint16, std::string>> Cameras;
	std::vector<uint8>							Data;
	std::vector<FUniqueMesh>					UniqueMeshes;
	FSceneVisibilityInfo					   *VisInfo   = nullptr;
	uint32										VisRemoved= 0;

	mutable std::vector<std::string>			MaterialsWithUVs;
	mutable std::vector<std::string>			MaterialsWithADDs;
	mutable std::vector<std::string>			MaterialsWithUNLITs;

protected:

	friend class FExporterGizmo;
	friend class FExporterCamera;
	friend class FExporterMesh;

	uint16 AllocateID()
	{
		return ID++;
	}

	FGizmo::ENodeType IdentifyObject(class FbxNode *Node);

	FExporterGizmo *CreateObject(class FbxNode *Node, FExporterGizmo *Parent);

	uint8 AddUniqueMaterial(const std::string &Name);

	std::string GetMaterialName(uint8 ID) const;

	void RegisterCamera(uint16 ID, std::string &Name);

	const FSceneVisibilityInfo *GetVis() const
	{
		return VisInfo;
	}

	void AddVisRemoved(uint32 v)
	{
		VisRemoved += v;
		return;
	}

	float GetStartTime() const
	{
		return StartTime;
	}

	float GetEndTime() const
	{
		return EndTime;
	}

	float GetFPS() const
	{
		return FPS;
	}

	uint32 GetAnimationFrames() const
	{
		return std::min(1024u, uint32(::ceilf((EndTime-StartTime)*FPS)+0.5f));
	}

	uint32 GetDataSize() const
	{
		return uint32(Data.size());
	}

	uint16 GetOffset(bool AlignPointer = true);

	void *SaveData(const void* DataPtr, size_t Size);

	//
	// Material UV's
	//
	bool MaterialNeedsUV(const FbxSurfaceMaterial *Mat) const;

	bool MaterialNeedsUV(uint8 ID) const;

	bool MaterialNeedsUV(const char *MaterialName) const;


	//
	// Material FLAGS
	//
	void FillMaterialFlags(const FbxSurfaceMaterial *Material) const;

	void AddAdditiveUnique(const char* MaterialName) const;

	void AddUnlitUnique(const char* MaterialName) const;

	bool MaterialIsAdditive(uint8 ID) const;

	bool MaterialIsAdditive(const char* MaterialName) const;

	bool MaterialIsUnlit(uint8 ID) const;

	bool MaterialIsUnlit(const char* MaterialName) const;

	//
	// Unique meshes
	//
	const FUniqueMesh *FindUniqueMesh(FMesh::EVertexFormat Format, const std::vector<uint16> &Faces, const std::vector<uint8> &VertexesData) const;

	void AddUniqueMesh(FMesh::EVertexFormat Format, const std::vector<uint16> &Faces, const std::vector<uint8> &VertexesData, uint16 VerticesOffset, uint16	FacesOffset);

public:

	FSceneExporter(class FbxScene *FBXScene, const char *VisDataName);
	~FSceneExporter();

	void Save(const char *Filename, const char *Scenename);
};
