// (c) MX^Add

#include "SceneExporterGizmo.h"
#include "SceneExporter.h"

FExporterGizmo::FExporterGizmo(FSceneExporter* Exporter, FExporterGizmo* ParentPtr, class FbxNode* _Node, bool Recurse)
{
	Type   = FGizmo::ENodeType::Gizmo;
	Node   = _Node;
	ID     = Exporter->AllocateID();
	Name   = Node->GetName();
	Parent = ParentPtr;

	HaveAnyAnimation = false;

	if (Recurse)
	{
		const uint32 Frames = Exporter->GetAnimationFrames();
		const float  Start  = Exporter->GetStartTime();
		const float  FPS    = Exporter->GetFPS();

		for (uint32 i = 0; i <= Frames; i++)
		{
			FbxTime Time; Time.SetSecondDouble(double(Start) + double(i) / double(FPS));
			FbxAMatrix Local = Node->EvaluateLocalTransform(Time);

			Locals.push_back(ConvertFBXMatrixXM(Local));

			if (i > 0 && !HaveAnyAnimation)
			{
				XMFLOAT4X4 FirstFrame = Locals[0];
				XMFLOAT4X4 ThisFrame  = Locals[i];

				for (uint32 r = 0; r < 4; r++)
				{
					for (uint32 c = 0; c < 4; c++)
					{
						if (fabsf(FirstFrame.m[r][c] - ThisFrame.m[r][c]) > 0.001f)
						{
							HaveAnyAnimation = true;
						}
					}
				}
			}
		}

		Childs.reserve(Node->GetChildCount());
		for (sint32 i = 0; i < Node->GetChildCount(); i++)
			Childs.push_back(Exporter->CreateObject(Node->GetChild(i), this));
	}
	else
	{
		XMFLOAT4X4 Identity; 
		XMStoreFloat4x4(&Identity, XMMatrixIdentity());
		Locals.push_back(Identity);
	}

	return;
}

FExporterGizmo::~FExporterGizmo()
{
	for (size_t i = 0; i < Childs.size(); i++)
		delete Childs[i];
	return;
}

void FExporterGizmo::ReduceChilds(FExporterGizmo *Parent)
{
	if (Parent == nullptr || HaveAnimation() || Type != FGizmo::ENodeType::Gizmo)
	{
		size_t ChildsNum = Childs.size();
		for (size_t i = 0; i < ChildsNum; i++)
			Childs[i]->ReduceChilds(this);

		return;
	}

	bool CouldReduceChilds = true;
	
	for (size_t i = 0; i < Childs.size(); i++)
	{
		if (!Childs[i]->Childs.empty())
		{
			CouldReduceChilds = false;
			break;
		}

		if (Childs[i]->HaveAnimation())
		{
			CouldReduceChilds = false;
			break;
		}
	}

	if (CouldReduceChilds)
	{
		// We are Gizmo, we don't have animation and our childs don't have childs.
		// We can remove self and add our childs to our parent childs

		for (size_t i = 0; i < Parent->Childs.size(); i++)
		{
			if (Parent->Childs[i] == this)
			{
				Parent->Childs[i] = nullptr;
				break;
			}
		}

		for (size_t i = 0; i < Childs.size(); i++)
		{
			Parent->Childs.push_back(Childs[i]);
			Childs[i]->Parent = Parent;
			Childs[i]->ReduceSelf(Locals[0]);
		}

		Childs.clear();
		delete this;
	}
	
	return;
}

void FExporterGizmo::PostReduceCleanup()
{
	for (size_t i = 0; i < Childs.size(); i++)
	{
		if (!Childs[i])
		{
			Childs.erase(Childs.begin()+i);
			i--;
		}
	}

	for (size_t i = 0; i < Childs.size(); i++)
		Childs[i]->PostReduceCleanup();

	return;
}

void FExporterGizmo::ReduceSelf(const XMFLOAT4X4 &Parent)
{
	XMFLOAT4X4 Self = Locals[0];
	Locals.clear();
	XMStoreFloat4x4(&Self, XMMatrixMultiply(XMLoadFloat4x4(&Self), XMLoadFloat4x4(&Parent)));
	Locals.push_back(Self);
	HaveAnyAnimation = false;
	return;
}

uint32 FExporterGizmo::GetSortKey() const
{
	uint32 Result = 0;

	if (Type == FGizmo::ENodeType::Camera)
		Result |= 0x80000000;

	if (Type == FGizmo::ENodeType::Mesh)
	{
		FExporterMesh *Mesh = (FExporterMesh *)this;

		Result |= 0x40000000;
		Result |= uint32(Mesh->MaterialID);
	}

	return Result;
}

void FExporterGizmo::SortChilds()
{
	std::sort(Childs.begin(), Childs.end(), [](const FExporterGizmo *a, const FExporterGizmo *b) 
	{
		return a->GetSortKey() > b->GetSortKey();
	});

	for (size_t i = 0; i < Childs.size(); i++)
		Childs[i]->SortChilds();

	return;
}

void FExporterGizmo::Print(sint32 Tabs) const
{
	static char Buffer[1024];
	memset(Buffer, 0, 1024);
	for (sint32 i = 0; i < MIN(1023, Tabs); i++)
		Buffer[i] = ' ';

	const char *TypeName = "Gizmo";
	switch (Type)
	{
		case FGizmo::ENodeType::Camera:
			TypeName = "Camera";
			break;

		case FGizmo::ENodeType::Mesh:
			TypeName = "Mesh";
			break;
	}

	printf("%s |->[%s][%s]%s", Buffer, Name.c_str(), TypeName, HaveAnyAnimation ? " *" : "  ");
	if (Type == FGizmo::ENodeType::Mesh)
	{
		FExporterMesh *Msh = (FExporterMesh *)this;
		printf(" [F:%i V:%i]\n", uint32(Msh->Faces.size())/3, uint32(Msh->Vertices.size()));
	}
	else
	{
		printf("\n");
	}

	for (size_t i = 0; i < Childs.size(); i++)
		Childs[i]->Print(Tabs+3);

	return;
}

size_t FExporterGizmo::FillInternals(FSceneExporter *Exporter, FGizmo* Ptr)
{
	Ptr->ID     = ID;
	Ptr->Type   = Type;
	Ptr->Childs = 0;

	return sizeof(FGizmo);
}

bool FExporterGizmo::OptimizeLinear(std::vector<FCurveFixed::FKeys>& Keys, Scalar Epsilon, const char *CurveName) const
{
	const size_t OriginalKeysNum = Keys.size();

	if (Keys.size() > 2)
	{
		for (uint32 i = 0; i < Keys.size();)
		{
			uint32 PushBest = 0;
			uint32 PushCnt  = 2;
			Scalar  KeyFirst = Keys[i].v;
			Scalar  KeyNext;

			while ((i+PushCnt) < Keys.size())
			{
				KeyNext = Keys[i+PushCnt].v;

				//
				// Check all keys along the push line
				//
				Scalar MaxError = 0;

				for (uint32 j = 1; j < PushCnt; j++)
				{
					Scalar Alpha    = Scalar(sint32(j)) / Scalar(sint32(PushCnt));

					Scalar KeyInt   = FScalar::Lerp(KeyFirst, KeyNext, Alpha);
					Scalar KeyNow   = Keys[i+j].v; 
					Scalar Error    = FScalar::Abs(KeyInt - KeyNow);
						  MaxError = FScalar::Max(MaxError, Error);
				}

				if (MaxError < Epsilon)
				{
					PushBest = PushCnt;
					PushCnt++;
				}
				else
				{
					break;
				}
			}

			if (PushBest == 0)
			{
				i++;
			}
			else
			{
				//
				// Remove all keys that we can lerp out
				//
				Keys.erase(Keys.begin()+i+1, Keys.begin()+i+1+(PushBest-1));
				i++;
			}
		}
	}

	if (Keys.size() == 2)
	{
		if (FScalar::Abs(Keys[0].v - Keys[1].v) < Epsilon)
			Keys.erase(Keys.begin()+1);
	}

	if (Keys.size() > 1)
		Keys.push_back(Keys[Keys.size()-1]); // Duplicate last key for search acceleration ...

	printf("Keys reduced (%s) %i ==> %i\n", CurveName, uint32(OriginalKeysNum), uint32(Keys.size()));

	return Keys.size() > 1;
}

bool FExporterGizmo::OptimizeQat(std::vector<FCurveQuat::FKeys>& Keys, Scalar Epsilon) const
{ 
	const size_t OriginalKeysNum = Keys.size();

	if (Keys.size() > 2)
	{
		for (uint32 i = 0; i < Keys.size();)
		{
			uint32    PushBest = 0;
			uint32    PushCnt  = 2;
			FVector4D KeyFirst = Keys[i].v;
			FVector4D KeyNext;

			while ((i+PushCnt) < Keys.size())
			{
				KeyNext = Keys[i+PushCnt].v;

				//
				// Check all keys along the push line
				//
				FVector4D MaxError(0, 0, 0, 0);

				for (uint32 j = 1; j < PushCnt; j++)
				{
					Scalar Alpha = Scalar(sint32(j)) / Scalar(sint32(PushCnt));

					FVector4D KeyInt; KeyInt.Slerp(KeyFirst, KeyNext, Alpha);
					FVector4D KeyNow   = Keys[i+j].v; 
					FVector4D Error    = (KeyInt - KeyNow).Abs();
							  MaxError = FVector4D::Max(MaxError, Error);
				}

				if (MaxError.x < Epsilon && MaxError.y < Epsilon && MaxError.z < Epsilon && MaxError.w < Epsilon)
				{
					PushBest = PushCnt;
					PushCnt++;
				}
				else
				{
					break;
				}
			}

			if (PushBest == 0)
			{
				i++;
			}
			else
			{
				//
				// Remove all keys that we can lerp out
				//
				Keys.erase(Keys.begin()+i+1, Keys.begin()+i+1+(PushBest-1));
				i++;
			}
		}
	}

	if (Keys.size() == 2)
	{
		if (FScalar::Abs(Keys[0].v.x - Keys[1].v.x) < Epsilon && FScalar::Abs(Keys[0].v.y - Keys[1].v.y) < Epsilon && FScalar::Abs(Keys[0].v.z - Keys[1].v.z) < Epsilon && FScalar::Abs(Keys[0].v.w - Keys[1].v.w) < Epsilon)
			Keys.erase(Keys.begin()+1);
	}

	if (Keys.size() > 1)
		Keys.push_back(Keys[Keys.size()-1]); // Duplicate last key for search acceleration ...

	printf("Keys reduced (ROT) %i ==> %i\n", uint32(OriginalKeysNum), uint32(Keys.size()));

	return Keys.size() > 1;
}

bool FExporterGizmo::Test(Scalar ConstTrnsX, const std::vector<FCurveFixed::FKeys> &TrnsX,
						  Scalar ConstTrnsY, const std::vector<FCurveFixed::FKeys> &TrnsY,
						  Scalar ConstTrnsZ, const std::vector<FCurveFixed::FKeys> &TrnsZ,
						  Scalar ConstScleX, const std::vector<FCurveFixed::FKeys> &ScleX,
						  Scalar ConstScleY, const std::vector<FCurveFixed::FKeys> &ScleY,
						  Scalar ConstScleZ, const std::vector<FCurveFixed::FKeys> &ScleZ,
						  const FVector4D &ConstRot, const std::vector<FCurveQuat::FKeys> &Rotation,
						  uint16 FrameStart, uint16 FrameEnd) const
{ 
	float MaxScaleError     = 0.0f;
	float MaxTranslateError = 0.0f;
	float MaxRotateError    = 0.0f;
	float MaxMatrixError    = 0.0f;

	for (uint16 Frame = FrameStart; Frame < FrameEnd; Frame++)
	{
		FVector4D Rot = Rotation.empty() ? ConstRot : FCurveQuat::Eval(&Rotation[0], uint16(Rotation.size()), Frame);

		FVector3D Scl = FVector3D(ScleX.empty() ? ConstScleX : FCurveFixed::Eval(&ScleX[0], uint16(ScleX.size()), Frame),
								  ScleY.empty() ? ConstScleY : FCurveFixed::Eval(&ScleY[0], uint16(ScleY.size()), Frame),
								  ScleZ.empty() ? ConstScleZ : FCurveFixed::Eval(&ScleZ[0], uint16(ScleZ.size()), Frame));

		FVector3D Trn = FVector3D(TrnsX.empty() ? ConstTrnsX : FCurveFixed::Eval(&TrnsX[0], uint16(TrnsX.size()), Frame),
								  TrnsY.empty() ? ConstTrnsY : FCurveFixed::Eval(&TrnsY[0], uint16(TrnsY.size()), Frame),
								  TrnsZ.empty() ? ConstTrnsZ : FCurveFixed::Eval(&TrnsZ[0], uint16(TrnsZ.size()), Frame));

		FMatrix Final = Rot.FromQuaterion();

		if (Scl.x != FixedOne || Scl.y != FixedOne || Scl.z != FixedOne)
		{
			FMatrix s; s.Identity();

			s.m[0][0] = Scl.x;
			s.m[1][1] = Scl.y;
			s.m[2][2] = Scl.z;

			Final = s * Final;
		}

		Final.m[3][0] = Trn.x;
		Final.m[3][1] = Trn.y;
		Final.m[3][2] = Trn.z;

		XMFLOAT4X4 SFinal = Locals[Frame];
		XMMATRIX FltFinal = XMLoadFloat4x4(&Locals[Frame]);
		XMVECTOR Rotate, Translate, Scale;
		XMFLOAT4 SRot, STrans, SScl;
		XMMatrixDecompose(&Scale, &Rotate, &Translate, FltFinal);
		XMStoreFloat4(&SRot, Rotate);
		XMStoreFloat4(&STrans, Translate);
		XMStoreFloat4(&SScl, Scale);

		//
		// Calculate relative errors...
		//
		MaxScaleError = std::max(MaxScaleError, fabsf(SScl.x - Scl.x));
		MaxScaleError = std::max(MaxScaleError, fabsf(SScl.y - Scl.y));
		MaxScaleError = std::max(MaxScaleError, fabsf(SScl.z - Scl.z));

		MaxTranslateError = std::max(MaxTranslateError, fabsf(STrans.x - Trn.x));
		MaxTranslateError = std::max(MaxTranslateError, fabsf(STrans.y - Trn.y));
		MaxTranslateError = std::max(MaxTranslateError, fabsf(STrans.z - Trn.z));

		MaxRotateError = std::max(MaxRotateError, fabsf(SRot.x - Rot.x));
		MaxRotateError = std::max(MaxRotateError, fabsf(SRot.y - Rot.y));
		MaxRotateError = std::max(MaxRotateError, fabsf(SRot.z - Rot.z));
		MaxRotateError = std::max(MaxRotateError, fabsf(SRot.w - Rot.w));

		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[0][0] - Final.m[0][0]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[0][1] - Final.m[0][1]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[0][2] - Final.m[0][2]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[0][3] - Final.m[0][3]));

		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[1][0] - Final.m[1][0]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[1][1] - Final.m[1][1]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[1][2] - Final.m[1][2]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[1][3] - Final.m[1][3]));

		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[2][0] - Final.m[2][0]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[2][1] - Final.m[2][1]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[2][2] - Final.m[2][2]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[2][3] - Final.m[2][3]));

		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[3][0] - Final.m[3][0]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[3][1] - Final.m[3][1]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[3][2] - Final.m[3][2]));
		MaxMatrixError = std::max(MaxMatrixError, fabsf(SFinal.m[3][3] - Final.m[3][3]));
	}

	if ((MaxScaleError > 0.01f || MaxRotateError > 0.01f || MaxTranslateError > 0.01f || MaxMatrixError > 0.01f))
	{
		printf("Testing animation for [%s] [%i - %i]...", Name.c_str(), uint32(FrameStart), uint32(FrameEnd));
		printf(" [Scale:%f][Rotate:%f][Translate:%f][Matrix:%f]\n", MaxScaleError, MaxRotateError, MaxTranslateError, MaxMatrixError);
		return false;
	}

	return true;
}

void FExporterGizmo::SavePayload(FSceneExporter* Exporter, FGizmo* Ptr)
{
	constexpr Scalar FixedEpsilonTrns(0.0050f);
	constexpr Scalar FixedEpsilonScl (0.0050f);
	constexpr Scalar FixedEpsilonRot (0.0025f);

	XMMATRIX FirstFrame = XMLoadFloat4x4(&Locals[0]);
	XMVECTOR Rotate, Translate, Scale;
	XMFLOAT4 SRot, STrans, SScl;
	XMMatrixDecompose(&Scale, &Rotate, &Translate, FirstFrame);
	XMStoreFloat4(&SRot, Rotate);
	XMStoreFloat4(&STrans, Translate);
	XMStoreFloat4(&SScl, Scale);

	FVector3D FixedTrns = FVector3D(STrans.x, STrans.y, STrans.z);
	FVector3D FixedScl  = FVector3D(FScalar::RoundToOne(SScl.x), FScalar::RoundToOne(SScl.y), FScalar::RoundToOne(SScl.z));
	FVector4D FixedRot  = FVector4D(SRot.x, SRot.y, SRot.z, SRot.w);

	bool HaveAnimS = false;
	bool HaveAnimR = false;
	bool HaveAnimT = false;

	std::vector<FCurveFixed::FKeys> TKeysX; TKeysX.resize(Locals.size());
	std::vector<FCurveFixed::FKeys> TKeysY; TKeysY.resize(Locals.size());
	std::vector<FCurveFixed::FKeys> TKeysZ; TKeysZ.resize(Locals.size());
	std::vector<FCurveFixed::FKeys> SKeysX; SKeysX.resize(Locals.size());
	std::vector<FCurveFixed::FKeys> SKeysY; SKeysY.resize(Locals.size());
	std::vector<FCurveFixed::FKeys> SKeysZ; SKeysZ.resize(Locals.size());
	std::vector<FCurveQuat::FKeys > RKeys;  RKeys .resize(Locals.size());

	for (size_t i = 0; i < Locals.size(); i++)
	{
		XMMATRIX Frame = XMLoadFloat4x4(&Locals[i]);
		XMVECTOR LRotate, LTranslate, LScale;
		XMFLOAT4 LRot, LTrans, LScl;
		XMMatrixDecompose(&LScale, &LRotate, &LTranslate, Frame);
		XMStoreFloat4(&LRot, LRotate);
		XMStoreFloat4(&LTrans, LTranslate);
		XMStoreFloat4(&LScl, LScale);

		TKeysX[i].f = uint16(i); TKeysX[i].v = LTrans.x;
		TKeysY[i].f = uint16(i); TKeysY[i].v = LTrans.y;
		TKeysZ[i].f = uint16(i); TKeysZ[i].v = LTrans.z;

		SKeysX[i].f = uint16(i); SKeysX[i].v = FScalar::RoundToOne(LScl.x);
		SKeysY[i].f = uint16(i); SKeysY[i].v = FScalar::RoundToOne(LScl.y);
		SKeysZ[i].f = uint16(i); SKeysZ[i].v = FScalar::RoundToOne(LScl.z);

		RKeys[i].f  = uint16(i); RKeys[i].v  = FVector4D(LRot.x, LRot.y, LRot.z, LRot.w);

		if (FScalar::Abs(Scalar(LTrans.x)-FixedTrns.x) > FixedEpsilonTrns)
			HaveAnimT = true;

		if (FScalar::Abs(Scalar(LTrans.y)-FixedTrns.y) > FixedEpsilonTrns)
			HaveAnimT = true;

		if (FScalar::Abs(Scalar(LTrans.z)-FixedTrns.z) > FixedEpsilonTrns)
			HaveAnimT = true;

		if (FScalar::Abs(Scalar(LScl.x)-FixedScl.x) > FixedEpsilonScl)
			HaveAnimS = true;

		if (FScalar::Abs(Scalar(LScl.y)-FixedScl.y) > FixedEpsilonScl)
			HaveAnimS = true;

		if (FScalar::Abs(Scalar(LScl.z)-FixedScl.z) > FixedEpsilonScl)
			HaveAnimS = true;

		if (FScalar::Abs(Scalar(LRot.x)-FixedRot.x) > FixedEpsilonRot)
			HaveAnimR = true;

		if (FScalar::Abs(Scalar(LRot.y)-FixedRot.y) > FixedEpsilonRot)
			HaveAnimR = true;

		if (FScalar::Abs(Scalar(LRot.z)-FixedRot.z) > FixedEpsilonRot)
			HaveAnimR = true;

		if (FScalar::Abs(Scalar(LRot.w)-FixedRot.w) > FixedEpsilonRot)
			HaveAnimR = true;
	}

	Ptr->Transform.Position[0].Value = FixedTrns.x;
	Ptr->Transform.Position[1].Value = FixedTrns.y;
	Ptr->Transform.Position[2].Value = FixedTrns.z;

	Ptr->Transform.Rotation.Value = FixedRot;

	Ptr->Transform.Scale[0].Value = FixedScl.x;
	Ptr->Transform.Scale[1].Value = FixedScl.y;
	Ptr->Transform.Scale[2].Value = FixedScl.z;

	if (HaveAnimT)
	{
		if (OptimizeLinear(TKeysX, FixedEpsilonTrns, "TRX"))
		{
			Ptr->Transform.Position[0].Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(TKeysX.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&TKeysX[0], TKeysX.size()*sizeof(FCurveFixed::FKeys));
		}
		else
		{
			TKeysX.clear();
		}

		if (OptimizeLinear(TKeysY, FixedEpsilonTrns, "TRY"))
		{
			Ptr->Transform.Position[1].Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(TKeysY.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&TKeysY[0], TKeysY.size()*sizeof(FCurveFixed::FKeys));
		}
		else
		{
			TKeysY.clear();
		}

		if (OptimizeLinear(TKeysZ, FixedEpsilonTrns, "TRZ"))
		{
			Ptr->Transform.Position[2].Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(TKeysZ.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&TKeysZ[0], TKeysZ.size()*sizeof(FCurveFixed::FKeys));
		}
		else
		{
			TKeysZ.clear();
		}
	}

	if (HaveAnimS)
	{
		if (OptimizeLinear(SKeysX, FixedEpsilonScl, "SCX"))
		{
			Ptr->Transform.Scale[0].Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(SKeysX.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&SKeysX[0], SKeysX.size()*sizeof(FCurveFixed::FKeys));
		}
		else
		{
			SKeysX.clear();
		}

		if (OptimizeLinear(SKeysY, FixedEpsilonScl, "SCY"))
		{
			Ptr->Transform.Scale[1].Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(SKeysY.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&SKeysY[0], SKeysY.size()*sizeof(FCurveFixed::FKeys));
		}
		else
		{
			SKeysY.clear();
		}

		if (OptimizeLinear(SKeysZ, FixedEpsilonScl, "SCZ"))
		{
			Ptr->Transform.Scale[2].Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(SKeysZ.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&SKeysZ[0], SKeysZ.size()*sizeof(FCurveFixed::FKeys));
		}
		else
		{
			SKeysZ.clear();
		}
	}

	if (HaveAnimR)
	{
		if (OptimizeQat(RKeys, FixedEpsilonRot))
		{
			Ptr->Transform.Rotation.Keys = Exporter->GetOffset();
			uint16 NumKeys = uint16(RKeys.size());
			Exporter->SaveData(&NumKeys, sizeof(uint16));
			Exporter->SaveData(nullptr, sizeof(uint16)); // Alignment!
			Exporter->SaveData(&RKeys[0], RKeys.size()*sizeof(FCurveQuat::FKeys));
		}
		else
		{
			RKeys.clear();
		}
	}

	Ptr->Transform.Position[0].Value = FixedTrns.x;
	Ptr->Transform.Position[1].Value = FixedTrns.y;
	Ptr->Transform.Position[2].Value = FixedTrns.z;

	Ptr->Transform.Rotation.Value = FixedRot;

	Ptr->Transform.Scale[0].Value = FixedScl.x;
	Ptr->Transform.Scale[1].Value = FixedScl.y;
	Ptr->Transform.Scale[2].Value = FixedScl.z;

	bool TestResult = Test(FixedTrns.x, TKeysX, FixedTrns.y, TKeysY, FixedTrns.z, TKeysZ,
						   FixedScl.x,  SKeysX, FixedScl.y,  SKeysY, FixedScl.z,  SKeysZ,
						   FixedRot, RKeys, 0, uint16(Locals.size()));
	if (!TestResult)
		printf("\033[33mWARNING::Failed transformation test for [%s] !\033[0m\n", Name.c_str());

	return;
}

void FExporterGizmo::SaveData(FSceneExporter* Exporter)
{
	uint32 BegOffset = Exporter->GetDataSize();

	FGizmo *ExPtr = nullptr;
	size_t  ExSsz = 0;
	{
		switch (Type)
		{
			case FGizmo::ENodeType::Gizmo:
				ExPtr = new FGizmo();
				ExSsz = sizeof(FGizmo);
			break;

			case FGizmo::ENodeType::Camera:
				ExPtr = new FCamera();
				ExSsz = sizeof(FCamera);
			break;

			case FGizmo::ENodeType::Mesh:
				ExPtr = new FMesh();
				ExSsz = sizeof(FMesh);
			break;
		}

		FillInternals(Exporter, ExPtr);
		void *Ptr = Exporter->SaveData(ExPtr, ExSsz);
		delete ExPtr;
		ExPtr = (FGizmo *)Ptr;
	}

	SavePayload(Exporter, ExPtr);

	BegOffset = (Exporter->GetDataSize() - BegOffset) + (Childs.size() ? ((uint32(Childs.size())+1)*sizeof(uint16)) : 0);
	printf("Total data for [%s] == %i bytes (%i bytes total now)\n\n", Name.c_str(), BegOffset, Exporter->GetDataSize());

	//
	// Save childs
	//
	if (Childs.size())
	{
		ExPtr->Childs = Exporter->GetOffset();
		uint16 *ChildsTable = (uint16 *)Exporter->SaveData(nullptr, (Childs.size()+1)*sizeof(uint16));
		ChildsTable[0] = uint16(Childs.size());

		for (size_t i = 0; i < Childs.size(); i++)
		{
			ChildsTable[i+1] = Exporter->GetOffset();
			Childs[i]->SaveData(Exporter);
		}
	}

	return;
}
