// (c) MX^Add

#include "SceneExporterMesh.h"
#include "SceneExporter.h"
#include "SoftwareRasterizer.h"
#include "SceneVisibilityData.h"

static inline uint16 ConvertColor(const XMFLOAT4& Clr)
{
	uint16 r = uint16(std::clamp(Clr.x * 31.0f, 0.0f, 31.0f));
	uint16 g = uint16(std::clamp(Clr.y * 63.0f, 0.0f, 63.0f));
	uint16 b = uint16(std::clamp(Clr.z * 31.0f, 0.0f, 31.0f));

	return b | (g << 5) | (r << 11);
}

static inline uint32 ConvertNormal(const XMFLOAT3& Clr)
{ 
	uint32 x = uint32(std::clamp(Clr.x * 0.5f + 0.5f, 0.0f, 1.0f) * 1023.0f);
	uint32 y = uint32(std::clamp(Clr.y * 0.5f + 0.5f, 0.0f, 1.0f) * 1023.0f);
	uint32 z = uint32(std::clamp(Clr.z * 0.5f + 0.5f, 0.0f, 1.0f) * 1023.0f);

	return x | (y << 10) | (z << 20);
}

static inline uint16 ConvertUV(const XMFLOAT2& UV)
{
	uint16 u = uint8((1.0f + UV.x) * float(FSoftwareRasterizer::TextureCacheSize-1));
	uint16 v = uint8((1.0f + UV.y) * float(FSoftwareRasterizer::TextureCacheSize-1));

	return u | (v << 8);
}

static inline XMVECTOR CalcNormal(XMVECTOR a, XMVECTOR b, XMVECTOR c)
{
	XMVECTOR Edge0 = XMVectorSubtract(b, a);
	XMVECTOR Edge1 = XMVectorSubtract(c, b);

	return XMVector3Normalize(XMVector3Cross(Edge0, Edge1));
}

FExporterMesh::FExporterMesh(FSceneExporter* Exporter, FExporterGizmo* ParentPtr, class FbxNode* Node, sint32 SectionID, bool Recurse) : FExporterGizmo(Exporter, ParentPtr, Node, Recurse)
{
	Type = FGizmo::ENodeType::Mesh;
	VertexFormat = FMesh::EVertexFormat::Simple;
	MaterialColor = 0;
	BoundingSphere = FVector4D(0, 0, 0, 0);
	MaterialID = 0;

	FbxMesh *Mesh = Node->GetMesh();
	if (!Mesh)
		return;

	XMFLOAT4 DefaultColor(1.0f, 1.0f, 1.0f, 1.0f);
	FbxProperty ColorProp = Mesh->FindProperty("Color");

	if (ColorProp.IsValid()) 
	{
		FbxDouble3 Color = ColorProp.Get<FbxDouble3>();
		DefaultColor.x = float(Color[0]);
		DefaultColor.y = float(Color[1]);
		DefaultColor.z = float(Color[2]);
	}

	bool HasNormals = false;
	bool MatAdded   = false;
	sint32 NeedsUV  = -1;
	uint32 Polygons = Mesh->GetPolygonCount();

	FVertex VtxData;

	FbxSurfaceMaterial *Mat = nullptr;

	if (SectionID < sint32(Node->GetMaterialCount()))
		Mat = Node->GetMaterial(SectionID);

	for (uint32 Poly = 0; Poly < Polygons; Poly++)
	{
		uint32 PolySize = Mesh->GetPolygonSize(Poly);

		if (PolySize != 3)
		{
			ErrorClear();
			return;
		}

		for (uint32 Vert = 0; Vert < PolySize; Vert++)
		{
			uint32 VertexIndex  = Mesh->GetPolygonVertex(Poly, Vert);
			uint32 PositionIndex= Mesh->GetPolygonVertexIndex(Poly) + Vert;					
			FbxVector4 Vertex   = Mesh->GetControlPointAt(VertexIndex);

			VtxData.Vtx = XMFLOAT3((float)Vertex[0], (float)Vertex[1], (float)Vertex[2]);

			bool AddThisVertex = false;

			//
			// Fetch all requests
			//
			for (sint32 l = 0; l < Mesh->GetLayerCount(); l++)
			{
				FbxArray<FbxLayerElementUV const*> UVS    = Mesh->GetLayer(l)->GetUVSets();

				const FbxLayerElementNormal      *Normals = Mesh->GetLayer(l)->GetNormals();
				const FbxLayerElementVertexColor *VColor  = Mesh->GetLayer(l)->GetVertexColors();
				const FbxLayerElementMaterial    *Mats    = Mesh->GetLayer(l)->GetMaterials();

				if (NeedsUV == -1)
					NeedsUV = Exporter->MaterialNeedsUV(Mat) ? 1 : 0;

				if (!MatAdded && !Vertices.empty())
				{
					MatAdded = true;
					std::string MatName = Mat->GetName();
					if (MatName.empty() || MatName == " ")
						MatName = "Default";
					MaterialID = Exporter->AddUniqueMaterial(MatName);
					Exporter->FillMaterialFlags(Mat);
				}

				if (Mats && !AddThisVertex)
				{
					switch (Mats->GetMappingMode()) 
					{							
						case FbxLayerElement::eByPolygon:
						{
							if (Mats->GetIndexArray().GetAt(Poly) == SectionID)
								AddThisVertex = true;
						}
						break;

						case FbxLayerElement::eAllSame:
						{
							if (Mats->GetIndexArray().GetAt(0) == SectionID)
								AddThisVertex = true;
						}
						break;

						default:
							ErrorClear();
							return;
					}
				}

				//
				// Fetch vertex color ?
				//
				if (VColor)
				{
					VtxData.Clr = DefaultColor;

					const auto &ColorArray = VColor->GetDirectArray();

					if (ColorArray.GetCount())
					{
						if (VColor->GetMappingMode() == FbxLayerElement::eByPolygonVertex) 
						{								
							switch (VColor->GetReferenceMode()) 
							{								
								case FbxLayerElement::eDirect:									
								{
									VtxData.Clr = XMFLOAT4((float)ColorArray.GetAt(PositionIndex).mRed, 
														   (float)ColorArray.GetAt(PositionIndex).mGreen, 
														   (float)ColorArray.GetAt(PositionIndex).mBlue,
														   (float)ColorArray.GetAt(PositionIndex).mAlpha);
								}
								break;								

								case FbxLayerElement::eIndexToDirect:									
								{										
									uint32 Idx = VColor->GetIndexArray().GetAt(PositionIndex);	

									VtxData.Clr = XMFLOAT4((float)ColorArray.GetAt(Idx).mRed, 
														   (float)ColorArray.GetAt(Idx).mGreen, 
														   (float)ColorArray.GetAt(Idx).mBlue,
														   (float)ColorArray.GetAt(Idx).mAlpha);
								}									
								break;

								default:
									ErrorClear();
									return;
							}
						}
						else
						if (VColor->GetMappingMode() == FbxLayerElement::eByControlPoint && VColor->GetReferenceMode() == FbxGeometryElement::eDirect)
						{
							VtxData.Clr = XMFLOAT4((float)ColorArray.GetAt(VertexIndex).mRed, 
												   (float)ColorArray.GetAt(VertexIndex).mGreen, 
												   (float)ColorArray.GetAt(VertexIndex).mBlue,
												   (float)ColorArray.GetAt(VertexIndex).mAlpha);
						}
					}
				}
				else
				{
					VtxData.Clr = DefaultColor;
				}

				//
				// Fetch normals ?
				//
				if (Normals)
				{
					if (Normals->GetMappingMode() == FbxLayerElement::eByPolygonVertex) 
					{								
						switch (Normals->GetReferenceMode()) 
						{								
							case FbxLayerElement::eDirect:									
							{
								VtxData.Nrm = XMFLOAT3((float)Normals->GetDirectArray().GetAt(PositionIndex)[0], 
													   (float)Normals->GetDirectArray().GetAt(PositionIndex)[1], 
													   (float)Normals->GetDirectArray().GetAt(PositionIndex)[2]);

								HasNormals = true;
							}
							break;								

							case FbxLayerElement::eIndexToDirect:									
							{										
								uint32 Idx = Normals->GetIndexArray().GetAt(PositionIndex);	

								VtxData.Nrm = XMFLOAT3((float)Normals->GetDirectArray().GetAt(Idx)[0], 
													   (float)Normals->GetDirectArray().GetAt(Idx)[1], 
													   (float)Normals->GetDirectArray().GetAt(Idx)[2]);

								HasNormals = true;
							}									
							break;

							default:
								ErrorClear();
								return;
						}
					}
					else
					if (Normals->GetMappingMode() == FbxLayerElement::eByControlPoint && Normals->GetReferenceMode() == FbxGeometryElement::eDirect)
					{
						VtxData.Nrm = XMFLOAT3((float)Normals->GetDirectArray().GetAt(VertexIndex)[0], 
								               (float)Normals->GetDirectArray().GetAt(VertexIndex)[1], 
								               (float)Normals->GetDirectArray().GetAt(VertexIndex)[2]);

						HasNormals = true;
					}
				}

				// UV0
				if (NeedsUV == 1)
				{
					const FbxLayerElementUV *UV = UVS[0];

					if (UV == NULL)
						continue;

					//
					// Fetch UV
					//
					switch (UV->GetMappingMode()) 
					{							
						case FbxLayerElement::eByControlPoint:	
						
							switch (UV->GetReferenceMode()) 
							{								
								case FbxLayerElement::eDirect:
								{
									VtxData.UV = XMFLOAT2((float)UV->GetDirectArray().GetAt(PositionIndex)[0], 1.0f - (float)UV->GetDirectArray().GetAt(PositionIndex)[1]);
								}
								break;								

								case FbxLayerElement::eIndexToDirect:									
								{										
									uint32 Idx = UV->GetIndexArray().GetAt(PositionIndex);

									VtxData.UV = XMFLOAT2((float)UV->GetDirectArray().GetAt(Idx)[0], 1.0f - (float)UV->GetDirectArray().GetAt(Idx)[1]);
								}
								break;

								default:
									printf("\033[33mWARNING::Unsupported UV mapping mode while processing [%s] !\033[0m\n", Name.c_str());
									return;
							}

						break;

						case FbxLayerElement::eByPolygonVertex:								
						{									
							uint32 Idx = Mesh->GetTextureUVIndex(Poly, Vert);							

							switch (UV->GetReferenceMode()) 
							{									
								case FbxLayerElement::eDirect:									
								case FbxLayerElement::eIndexToDirect:										
								{
									VtxData.UV  = XMFLOAT2((float)UV->GetDirectArray().GetAt(Idx)[0], 1.0f - (float)UV->GetDirectArray().GetAt(Idx)[1]);
								}
								break;

								default:
									printf("\033[33mWARNING::Unsupported UV mapping mode while processing [%s] !\033[0m\n", Name.c_str());
									return;
							}								
						}								
					}						
				}						
			}

			if (AddThisVertex)
			{
				AddVertex(VtxData);
			}
		}
	}

	if (!HasNormals)
	{
		// Calculate soft mesh normals

		for (size_t i = 0; i < Vertices.size(); i++)
			XMStoreFloat3(&Vertices[i].Nrm, XMVectorZero());

		for (size_t i = 0; i < Faces.size(); i += 3)
		{
			XMVECTOR v0  = XMLoadFloat3(&Vertices[Faces[i+0]].Vtx);
			XMVECTOR v1  = XMLoadFloat3(&Vertices[Faces[i+1]].Vtx);
			XMVECTOR v2  = XMLoadFloat3(&Vertices[Faces[i+2]].Vtx);

			XMStoreFloat3(&Vertices[Faces[i+0]].Nrm, XMVectorAdd(XMLoadFloat3(&Vertices[Faces[i+0]].Nrm), CalcNormal(v0, v1, v2)));
			XMStoreFloat3(&Vertices[Faces[i+1]].Nrm, XMVectorAdd(XMLoadFloat3(&Vertices[Faces[i+1]].Nrm), CalcNormal(v0, v1, v2)));
			XMStoreFloat3(&Vertices[Faces[i+2]].Nrm, XMVectorAdd(XMLoadFloat3(&Vertices[Faces[i+2]].Nrm), CalcNormal(v0, v1, v2)));
		}

		for (size_t i = 0; i < Vertices.size(); i++)
			XMStoreFloat3(&Vertices[i].Nrm, XMVector3Normalize(XMLoadFloat3(&Vertices[i].Nrm)));
	}

	OptimizeForVertexCache();
	PreparePayload(Exporter);
	return;
}

FExporterMesh::~FExporterMesh()
{
	return;
}

void FExporterMesh::AddVertex(const FExporterMesh::FVertex& v)
{
	for (size_t i = 0; i < Vertices.size(); i++)
	{
		if (Vertices[i] == v)
		{
			Faces.push_back(uint16(i));
			return;
		}
	}

	Faces.push_back(uint16(Vertices.size()));
	Vertices.push_back(v);

	if (Vertices.size() > 0xFFFF)
	{
		printf("\033[31mERROR::Error while exporting mesh [%s] (more than 64k vertices) !\033[0m\n", Name.c_str());	
		assert(false);
	}

	return;
}

void FExporterMesh::ErrorClear()
{
	printf("\033[31mERROR::Error while exporting mesh [%s] !\033[0m\n", Name.c_str());
	assert(false);
	Vertices.clear();
	Faces.clear();
	return;
}

void FExporterMesh::PreparePayload(FSceneExporter *Exporter, bool FromVisPass)
{
	size_t VertexSize = sizeof(FMesh::FVertex);
	VertexFormat = FMesh::EVertexFormat::Simple;

	uint16 FirstColor = ConvertColor(Vertices[0].Clr);
	bool   HasColors  = false;
	bool   HasUV      = false;
	
	XMVECTOR AvgColor = XMVectorZero();
	XMVECTOR VtxMin   = XMVectorReplicate( 1000000.0f);
	XMVECTOR VtxMax   = XMVectorReplicate(-1000000.0f);

	for (size_t i = 0; i < Vertices.size(); i++)
	{
		if (ConvertColor(Vertices[i].Clr) != FirstColor)
			HasColors = true;
		AvgColor = XMVectorAdd(AvgColor, XMLoadFloat4(&Vertices[i].Clr));

		XMVECTOR Vtx = XMLoadFloat3(&Vertices[i].Vtx);

		VtxMin = XMVectorMin(VtxMin, Vtx);
		VtxMax = XMVectorMax(VtxMax, Vtx);
	}

	XMFLOAT4 AvgClrStore; XMStoreFloat4(&AvgClrStore, XMVectorDivide(AvgColor, XMVectorReplicate(float(Vertices.size()))));
	MaterialColor = ConvertColor(AvgClrStore);

	if (HasColors)
	{
		VertexFormat = FMesh::EVertexFormat::Color;
		VertexSize   = sizeof(FMesh::FVertexColor);
	}

	if (Exporter->MaterialNeedsUV(MaterialID))
	{
		VertexFormat = FMesh::EVertexFormat::UV;
		VertexSize   = sizeof(FMesh::FVertexUV);
		HasUV		 = true;
	}

	XMFLOAT4 StrMin;  XMStoreFloat4(&StrMin, VtxMin);
	XMFLOAT4 StrMax;  XMStoreFloat4(&StrMax, VtxMax);
	XMVECTOR Center = XMVectorMultiply(XMVectorAdd(VtxMin, VtxMax), XMVectorReplicate(0.5f));
	XMVECTOR SphW   = XMVector3Length(XMVectorSubtract(VtxMax, Center));

	BoundingSphere.x = (StrMin.x + StrMax.x) / 2.0f;
	BoundingSphere.y = (StrMin.y + StrMax.y) / 2.0f;
	BoundingSphere.z = (StrMin.z + StrMax.z) / 2.0f;
	BoundingSphere.w = _EXTRACT_X(SphW);

	PayloadVertexes.resize(VertexSize * Vertices.size());

 	switch (VertexFormat)
	{
		case FMesh::EVertexFormat::Simple:
		{
			FMesh::FVertex *Tgt = (FMesh::FVertex *)&PayloadVertexes[0];
			for (size_t i = 0; i < Vertices.size(); i++, Tgt++)
			{
				const FVertex *Vtx = &Vertices[i];

				Tgt->Pos.x = Vtx->Vtx.x;
				Tgt->Pos.y = Vtx->Vtx.y;
				Tgt->Pos.z = Vtx->Vtx.z;

				Tgt->Nrm   = ConvertNormal(Vtx->Nrm);
			}
		}
		break;

		case FMesh::EVertexFormat::Color:
		{
			FMesh::FVertexColor *Tgt = (FMesh::FVertexColor *)&PayloadVertexes[0];
			for (size_t i = 0; i < Vertices.size(); i++, Tgt++)
			{
				const FVertex *Vtx = &Vertices[i];

				Tgt->Pos.x = Vtx->Vtx.x;
				Tgt->Pos.y = Vtx->Vtx.y;
				Tgt->Pos.z = Vtx->Vtx.z;

				Tgt->Nrm   = ConvertNormal(Vtx->Nrm);
				Tgt->Clr   = ConvertColor(Vtx->Clr);
			}
		}
		break;

		case FMesh::EVertexFormat::UV:
		{
			FMesh::FVertexUV *Tgt = (FMesh::FVertexUV *)&PayloadVertexes[0];
			for (size_t i = 0; i < Vertices.size(); i++, Tgt++)
			{
				const FVertex *Vtx = &Vertices[i];

				Tgt->Pos.x = Vtx->Vtx.x;
				Tgt->Pos.y = Vtx->Vtx.y;
				Tgt->Pos.z = Vtx->Vtx.z;

				Tgt->Nrm   = ConvertNormal(Vtx->Nrm);
				Tgt->UV    = ConvertUV(Vtx->UV);
			}
		}
		break;
	}

	if (FromVisPass)
		return;

	VisPass(Exporter);

	printf("MeshData: [0x%04X => %s:%s] - Vertices:%i Triangles:%i HasColors:%i HasUV:%i Bounds:[%3.3f %3.3f %3.3f - %3.3f]\n", ID, Name.c_str(), Exporter->GetMaterialName(MaterialID).c_str(), uint32(Vertices.size()), uint32(Faces.size()/3), HasColors ? 1 : 0, HasUV ? 1 : 0, (StrMin.x + StrMax.x) / 2.0f, (StrMin.y + StrMax.y) / 2.0f, (StrMin.z + StrMax.z) / 2.0f, _EXTRACT_X(SphW));
	
	return;
}

size_t FExporterMesh::FillInternals(FSceneExporter *Exporter, FGizmo *Ptr)
{
	Super::FillInternals(Exporter, Ptr);

	FMesh *MPtr = (FMesh *)Ptr;

	size_t VertexSize    = sizeof(FMesh::FVertex);

	MPtr->VerticesOffset = 0;
	MPtr->FacesOffset    = 0;
	MPtr->NumVetices     = uint16(Vertices.size());
	MPtr->NumFaces       = uint16(Faces.size());
	MPtr->MaterialID     = MaterialID | (Exporter->MaterialIsAdditive(MaterialID) ? FSoftwareRasterizer::MaterialFlagAdd : 0) | (Exporter->MaterialIsUnlit(MaterialID) ? FSoftwareRasterizer::MaterialFlagUnlit : 0);
	MPtr->VertexFormat   = VertexFormat;
	MPtr->MaterialColor  = MaterialColor;
	MPtr->BoundingSphere = BoundingSphere;

	return sizeof(FMesh);
}

void FExporterMesh::SavePayload(FSceneExporter* Exporter, FGizmo* Ptr)
{
	Super::SavePayload(Exporter, Ptr);

	FMesh *MPtr = (FMesh *)Ptr;

	const FSceneExporter::FUniqueMesh *UniqueMesh = Exporter->FindUniqueMesh(MPtr->VertexFormat, Faces, PayloadVertexes);

	if (UniqueMesh)
	{
		MPtr->VerticesOffset = UniqueMesh->VerticesOffset;
		MPtr->FacesOffset    = UniqueMesh->FacesOffset;
	}
	else
	{
		MPtr->VerticesOffset = Exporter->GetOffset();
		Exporter->SaveData(&PayloadVertexes[0], PayloadVertexes.size());

		MPtr->FacesOffset = Exporter->GetOffset();
		Exporter->SaveData(&Faces[0], Faces.size() * sizeof(uint16));

		Exporter->AddUniqueMesh(MPtr->VertexFormat, Faces, PayloadVertexes, MPtr->VerticesOffset, MPtr->FacesOffset);
	}

	return;
}

void FExporterMesh::OptimizeForVertexCache()
{
	//
	// Remaps Faces and Vertices tables to maximize cache efficiency
	//
	const sint32 NumTriangles = sint32(Faces.size() / 3);
	const sint32 NumVertices  = sint32(Vertices.size());
	const sint32 CacheSize    = FSoftwareRasterizer::VertexCacheSize;

	std::vector<std::vector<sint32>> VertexToTriangles(NumVertices);

	for (sint32 tri = 0; tri < NumTriangles; ++tri)
	{
		for (sint32 i = 0; i < 3; ++i)
		{
			uint16 v = Faces[tri * 3 + i];
			VertexToTriangles[v].push_back(tri);
		}
	}

	std::vector<bool> TriangleUsed(NumTriangles, false);
	std::vector<sint32> CacheTag(NumVertices, -CacheSize);
	sint32 CurrentTimestamp = 0;

	std::vector<uint16> NewFaces;
	NewFaces.reserve(Faces.size());

	std::vector<sint32> VertexLiveTriangleCount(NumVertices);
	for (sint32 v = 0; v < NumVertices; ++v)
		VertexLiveTriangleCount[v] = sint32(VertexToTriangles[v].size());

	for (sint32 tri = 0; tri < NumTriangles; ++tri)
	{
		sint32 BestTri = -1;
		sint32 BestScore = -1;

		for (sint32 v = 0; v < NumVertices; ++v)
		{
			if (VertexLiveTriangleCount[v] == 0) 
				continue;

			for (sint32 triIdx : VertexToTriangles[v])
			{
				if (TriangleUsed[triIdx]) 
					continue;

				sint32 CacheHits = 0;

				for (sint32 k = 0; k < 3; ++k)
				{
					uint16 vi = Faces[triIdx * 3 + k];

					if (CurrentTimestamp - CacheTag[vi] < CacheSize)
						++CacheHits;
				}

				if (CacheHits > BestScore)
				{
					BestScore = CacheHits;
					BestTri   = triIdx;

					if (BestScore == 3) 
						break;
				}
			}

			if (BestScore == 3) 
				break;
		}

		if (BestTri == -1)
		{
			for (sint32 t = 0; t < NumTriangles; ++t)
			{
				if (!TriangleUsed[t])
				{
					BestTri = t;
					break;
				}
			}
		}

		if (BestTri == -1) 
			break;

		TriangleUsed[BestTri] = true;

		for (sint32 k = 0; k < 3; ++k)
		{
			uint16 vi = Faces[BestTri * 3 + k];
			NewFaces.push_back(vi);
			CacheTag[vi] = CurrentTimestamp++;
			VertexLiveTriangleCount[vi]--;
		}
	}

	std::vector<sint32>  Remap(NumVertices, -1);
	std::vector<FVertex> NewVertexes;
	NewVertexes.reserve(NumVertices);

	sint32 NewIndex = 0;

	for (uint16 &idx : NewFaces)
	{
		if (Remap[idx] == -1)
		{
			Remap[idx] = NewIndex++;
			NewVertexes.push_back(Vertices[idx]);
		}

		idx = Remap[idx];
	}

	Faces.swap(NewFaces);
	Vertices.swap(NewVertexes);

	return;
}

void FExporterMesh::VisPass(FSceneExporter *Exporter)
{
	const FSceneVisibilityInfo *Vis = Exporter->GetVis();
	if (!Vis || Vis->Meshes.empty())
		return;

	const FMeshVisibility *FoundMesh = nullptr;

	for (size_t mid = 0; mid < Vis->Meshes.size(); mid++)
	{
		const FMeshVisibility *Mesh = &Vis->Meshes[mid];
		if (Mesh->Faces.size() == Faces.size() && Mesh->Verts.size() == PayloadVertexes.size() && Mesh->VFormat == VertexFormat)
		{
			if (memcmp(&Mesh->Faces[0], &Faces[0], Faces.size() * 2) == 0 && memcmp(&Mesh->Verts[0], &PayloadVertexes[0], PayloadVertexes.size()) == 0)
			{
				FoundMesh = Mesh;
				break;
			}
		}
	}

	if (FoundMesh)
	{
		printf("Found visibility info for [%s] - reducing mesh ...\n", Name.c_str());

		std::vector<bool> Vis = FoundMesh->Visis; // We are doing copy of the vis data for simplicity of the loop below.

		uint32 FacesRemoved = 0;

		for (size_t i = 0; i < Vis.size(); i++)
		{
			if (!Vis[i])
			{
				Faces.erase(Faces.begin()+i*3, Faces.begin()+i*3+3);
				Vis.erase(Vis.begin()+i);
				i--;
				FacesRemoved++;
			}
		}

		if (Faces.size() == 0)
		{
			Vertices.clear();
			PayloadVertexes.clear();
		}
		else
		if (FacesRemoved)
		{
			//
			// This will take care of unused indices for us
			//

			OptimizeForVertexCache();
			PreparePayload(Exporter, true);
		}

		Exporter->AddVisRemoved(FacesRemoved);
		printf("Removed %i faces.\n", FacesRemoved);
	}
	else
	{
		printf("\033[33mVisibility info for [%s] not found - mesh differs ?\033[0m\n", Name.c_str());
	}

	return;
}