// (c) MX^Add
#pragma once

#include "BaseTypes/BaseTypes.h"
#include "BaseTypes/ScalarType.h"
#include "RendererTypes/Vector.h"
#include "RendererTypes/Color.h"

#if ((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER)
extern uint32 __VisPixelsRendered;
#define DBGCLRP() __VisPixelsRendered = 0
#define DBGINCP() __VisPixelsRendered+= 1
#else
#define DBGCLRP()
#define DBGINCP()
#endif

//
// Post-Transform vertexes used by rasterizer
//
struct FVertexXYXW
{
    FVector4D v;

    void Lerp(const FVertexXYXW& a, FVertexXYXW& b, Scalar t)
    {
        v.Lerp(a.v, b.v, t);
        return;
    }
};

static_assert(sizeof(FVertexXYXW) == 16, "FVertexXYXW must be 16 bytes !");

struct FVertexXYXWF
{
    FVector4D v;
    Scalar    f;

    void Lerp(const FVertexXYXWF& a, FVertexXYXWF& b, Scalar t)
    {
        v.Lerp(a.v, b.v, t);
        f = a.f + (b.f - a.f) * t;
        return;
    }
};

static_assert(sizeof(FVertexXYXWF) == 20, "FVertexXYXWF must be 20 bytes !");

struct FVertexXYXWC
{
    FVector4D v;
    FColor16  c;
    uint8 __alg; // Alignment!

    void Lerp(const FVertexXYXWC& a, FVertexXYXWC& b, Scalar t)
    {
        v.Lerp(a.v, b.v, t);
        c.Lerp(a.c, b.c, t);
        return;
    }
};

static_assert(sizeof(FVertexXYXWC) == 20, "FVertexXYXWC must be 20 bytes !");

struct FVertexXYXWUV
{
    FVector4D v;
    FVector2D uv;

    void Lerp(const FVertexXYXWUV& a, FVertexXYXWUV& b, Scalar t)
    {
        v.Lerp(a.v, b.v, t);
        uv.Lerp(a.uv, b.uv, t);
        return;
    }
};

static_assert(sizeof(FVertexXYXWUV) == 24, "FVertexXYXWUV must be 24 bytes !");

struct FVertexXYXWUVF
{
    FVector4D v;
    FVector2D uv;
    Scalar    f;

    void Lerp(const FVertexXYXWUVF& a, FVertexXYXWUVF& b, Scalar t)
    {
        v.Lerp(a.v, b.v, t);
        uv.Lerp(a.uv, b.uv, t);
        f = a.f + (b.f - a.f) * t;
        return;
    }
};

static_assert(sizeof(FVertexXYXWUVF) == 28, "FVertexXYXWUVF must be 28 bytes !");

//
// Spawn second core function
// 
extern void RasterizerSpawnSecondCoreFunction(void (*SecondCoreFunctionCall)(void));

//
// Flush work
//
extern void FlushWorkItems();

//
// Assign work as callback on the second core, the argument passed is the copy of array returned by RasterizerGetInSpace(), that can be used to pass any arguments
//
extern void PushSecondCoreWork(void (*Fnc)(const void *Arg));

//
// Rasterize to framebuffer, coords are in projection space (before perspective division), runtime does the clipping
//
extern void RasterizeLine3D(const FVector4D &a, const FVector4D &b, uint16 Color, bool AlwaysTop = false);

//
// Rasterize to framebuffer, coords are in projection space (before perspective division), runtime does the clipping and backface culling
// This returns space for 84 bytes of data
//
extern void *RasterizerGetInSpace();

extern void RasterizeTriangle3DWireframeConstColor(uint16 Color);

extern void RasterizeTriangle3DWireframeVertexColors();

extern void RasterizeTriangle3DConstColor(uint16 Color);

extern void RasterizeTriangle3DVertexFactors(const uint16 *Lockup);

extern void RasterizeTriangle3DVertexColors(bool AdditiveBlend);

extern void RasterizeTriangle3DTexturedUnlit(const uint16 *Texture);

extern void RasterizeTriangle3DTexturedLit(const uint16 *Texture);
