// (c) MX^Add
#pragma once

#include "BaseTypes/BaseTypes.h"
#include "BaseTypes/ScalarType.h"
#include "RendererTypes/Matrix.h"
#include "Renderer/Rasterizers.h"

class FScene;
class FGizmo;
class FCamera;
class FMesh;

#if ((defined VISIBILITY_CHECKER) && VISIBILITY_CHECKER)
extern void __ReportVFace(const FMesh *Mesh, const void *VtxPtr, const uint16 *FcsPtr, uint16 Face, uint32 Pixels);
#define DBGF(a, b, c, d, e) __ReportVFace(a, b, c, d, e)
#else
#define DBGF(a, b, c, d, e)
#endif

#define SW_DEBUG_DRAW_BOUNDS 0 // NOTE::For debugging only !

class FSoftwareRasterizer
{
public:

	static constexpr sint32 TextureCacheSize   = 64;   // Must be Pow2 !
	static constexpr uint8  MaterialFlagAdd    = 0x80;
	static constexpr uint8  MaterialFlagUnlit  = 0x40;
	static constexpr uint8  MaterialUIDMask    = 0x3F;
	static constexpr uint32 AdditiveBlendSlots = 16;
	static constexpr uint32 VertexCacheSize    = 16;   // Must be Pow2 !

private:

	FSoftwareRasterizer();
	~FSoftwareRasterizer();

	//
	// Internal stuff
	//
	uint16          CurrentFrame;
	uint16			CameraID;
	FMatrix			CameraWorldMatrix;
	FMatrix			CameraMatrix;
	FMatrix			ProjectionMatrix;
	FMatrix			CameraProjectionMatrix;
	FVector3D		CameraPosition;
	FVector3D		CameraDirection;
	FVector3D		CameraUp;
	FVector4D       FrustumPlanes[6];
	const FCamera  *CameraPtr;
	const FGizmo   *CameraBck;
	const FScene   *ScenePtr;

	//
	// State values
	//
	bool			ClearDepth = true;
	bool            PostBeforeTranslucent = true;
	void		  (*ColorClearCallback)(uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat) = nullptr;
	void		  (*CameraUpdateCallback)(uint16 Frame, FVector3D &CamPos, FVector3D &CamDir, FVector3D &CamUp, Scalar &Aspect, uint8 &FOV, uint16 &Near, uint16 &Far) = nullptr;
	void		  (*CameraUpdatePlanesCallback)(uint16 Frame, FVector4D *Planes, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat) = nullptr;
	void		  (*PostDrawCallback)(uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat) = nullptr;
	const uint16 *(*TextureCacheCallback)(uint16 Frame, uint8 MID, uint16 *Dst) = nullptr;
	void          (*BlendSlotRenderCallback)(uint16 Frame, FVector3D &CamPos, FVector3D &CamDir, FVector3D &CamUp, const uint8 *Base, const FMatrix &LocalToWorld, const FMesh *Mesh) = nullptr;
	bool			ClearColor = true;
	uint16			ClearColorVal = 0;

	bool		    Wireframe = false;
	bool            UnlitMode = false;

	bool			DirectionalLightEnabled = true;
	FVector3D       DirectionalLightVector = FVector3D(FixedZero, FixedOne, FixedZero);
	Scalar			DirectionalLightIntensity = FixedOne;

	bool			PointLightEnabled = false;
	FVector3D       PointLightPosition = FVector3D(FixedZero, FixedZero, FixedZero);
	Scalar          PointLightRange = FixedOne;
	Scalar          PointLightInvRange = FixedOne;
	Scalar		    PointLightIntensity = FixedOne;

	Scalar          LightAmbientFactor = Scalar(0.125f);

	//
	// Mutable states
	//
	mutable            uint8	ColorClearDone;
	mutable            uint32	LockupColor;	
	alignas(4) mutable uint16	TextureCache[TextureCacheSize*TextureCacheSize]; // NOTE::This cache is used both for TextureLockup _AND_ ColorLockupCache, so the size of it must be >= 64
	mutable      const uint16  *TextureCachePtr = nullptr;
	alignas(4) mutable FMatrix  AdditiveBlendTransforms[AdditiveBlendSlots];
    mutable      const FMesh   *AdditiveBlendMeshes    [AdditiveBlendSlots];
    mutable            uint32   NumOfAdditiveMeshes;
	alignas(4) mutable uint8    VertexCacheSpace[VertexCacheSize * MAX(sizeof(FVertexXYXW), sizeof(FVertexXYXWF), MAX(sizeof(FVertexXYXWC), sizeof(FVertexXYXWUV), sizeof(FVertexXYXWUVF)))];
	mutable			   uint16   VertexCacheInx  [VertexCacheSize];

	inline void InvalidateVertexCache() const
	{
		memset(VertexCacheInx, -1, VertexCacheSize * sizeof(uint16));
		return;
	}

	bool ItetateSceneForCamera(const FGizmo *Parent, const FGizmo *Node);

	void ItetateSceneForRendering(const FGizmo *Parent, FMatrix ToWorld, const FGizmo *Node);

	bool EvaluateCamera();

	void DoColorClear() const;

	const uint16 *EmptyTextureCallback(uint8 MaterialID, uint16 *TextureCache) const;

	#if SW_DEBUG_DRAW_BOUNDS
	void DrawDebugSphere(const FVector3D &Center, Scalar Radius, uint16 Color) const;
	#endif

public:

	static FSoftwareRasterizer *Get();

	void SetDefaultState();

	void SetState_CameraCallback(void (*CamCall)(uint16 Frame, FVector3D &CamPos, FVector3D &CamDir, FVector3D &CamUp, Scalar &Aspect, uint8 &FOV, uint16 &Near, uint16 &Far));
	void SetState_CameraPlanesCallback(void (*CamPCall)(uint16 Frame, FVector4D *Planes, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat));
	void SetState_SetTextureCallback(const uint16 * (*TextureCallback)(uint16 Frame, uint8 MID, uint16 *Dst));
	void SetState_SetBlendSlotRenderCallback(void (*BsCallback)(uint16 Frame, FVector3D &CamPos, FVector3D &CamDir, FVector3D &CamUp, const uint8 *Base, const FMatrix &LocalToWorld, const FMesh *Mesh));
	void SetState_ClearColor(bool ClearColor, uint16 ClearColorValue, void (*ClearCallback)(uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat));
	void SetState_PostRender(bool BeforeTrans, void (*PostCallback)(uint16 Frame, const FVector3D &CamDir, const FVector3D &CamUp, const FMatrix &CamMat, const FMatrix &ProjMat, const FMatrix &CamProjMat));
	void SetState_ClearDepth(bool ClearDepth);
	void SetState_RenderMode(bool WireFrameMode, bool UnlitMode);
	void SetState_AmbientLight(Scalar Factor);
	void SetState_DirectionalLight(bool Enabled, const FVector3D &LightDirection, Scalar Intensity = FixedOne);
	void SetState_PointLightLight(bool Enabled, const FVector3D &LightPosition, Scalar Range, Scalar Intensity = FixedOne);

	void Render(const FScene *Scene, uint16 Frame, uint16 CameraID);

	//
	// Exposed internals to be used in callbacks
	//
	void PrepareLockup(uint16 Color, bool Flush = true) const;
	void PrepareTexture(uint8 UID, bool Flush = true) const;
	void RenderMeshAdditive(const uint8 *Base, const FMatrix &LocalToWorld, const FMesh *Mesh) const;
	void RenderMesh(const uint8 *Base, const FMatrix &LocalToWorld, const FMesh *Mesh) const;

	inline const uint16 *GetLockupTable() const
	{
		return TextureCache;
	}

	inline Scalar GetAmbientFactor() const
	{
		return LightAmbientFactor;
	}

	inline bool GetDirectionalIsEnabled() const
	{
		return DirectionalLightEnabled;
	}

	inline const FVector3D& GetDirectionalDirection() const
	{
		return DirectionalLightVector;
	}

	inline Scalar GetDirectionalIntensity() const
	{
		return DirectionalLightIntensity;
	}

	inline bool GetPointlightIsEnabled() const
	{
		return PointLightEnabled;
	}

	inline const FVector3D& GetPointlightPosition() const
	{
		return PointLightPosition;
	}

	inline Scalar GetPointlightRange() const
	{
		return PointLightRange;
	}

	inline Scalar GetPointlightInvRange() const
	{
		return PointLightInvRange;
	}

	inline Scalar GetPointlightIntensity() const
	{
		return PointLightIntensity;
	}
};
