// (c) MX^Add
#pragma once

#include "BaseTypes/BaseTypes.h"
#include "BaseTypes/ScalarType.h"
#include "RendererTypes/Vector.h"

class FColor16
{
public:
	uint8 r, g, b;

			  FColor16() {}
	constexpr FColor16(uint16 c)					    : r(c>>11), g((c>>5)&0x3F), b(c&0x1F)				   {}
	constexpr FColor16(uint8 _r, uint8 _g, uint8 _b)    : r(_r>>3), g(_g>>2),		b(_b>>3)				   {}
	constexpr FColor16(Scalar _r, Scalar _g, Scalar _b) : r(sint32(_r)>>3), g(sint32(_g)>>2), b(sint32(_b)>>3) {}

	constexpr uint16 toColorNoClip() const
	{
		return (uint16(r) << 11) | uint16(g << 5) | b;
	}

    inline void Lerp(const FColor16& ca, const FColor16& cb, Scalar t) // NOTE::Assumes (cr < 32, cg < 64, cb < 32, t >= 0 && <= 1) !
    {
		const sint32 ft = sint32(t * 65536.0f);

		r = sint32(ca.r) + (((sint32(cb.r) - sint32(ca.r)) * ft) >> 16);
		g = sint32(ca.g) + (((sint32(cb.g) - sint32(ca.g)) * ft) >> 16);
		b = sint32(ca.b) + (((sint32(cb.b) - sint32(ca.b)) * ft) >> 16);

        return;
    }

	constexpr inline uint16 toBrighter() const
	{
		uint8 ra = MIN(r+1, 0x1F);
		uint8 ga = MIN(g+1, 0x3F);
		uint8 ba = MIN(b+1, 0x1F);

		return (uint16(ra) << 11) | uint16(ga << 5) | ba;
	}

	inline static uint16 BarycentricInterpolation(const FColor16& ca, Scalar w0, const FColor16& cb, Scalar w1, const FColor16& cc, Scalar w2)
	{
		const sint32 fw0 = sint32(w0 * 65536.0f);
		const sint32 fw1 = sint32(w1 * 65536.0f);
		const sint32 fw2 = sint32(w2 * 65536.0f);

		uint16 r = ((sint32(ca.r) * fw0) + (sint32(cb.r) * fw1) + (sint32(cc.r) * fw2)) >> 16;
		uint16 g = ((sint32(ca.g) * fw0) + (sint32(cb.g) * fw1) + (sint32(cc.g) * fw2)) >> 16;
		uint16 b = ((sint32(ca.b) * fw0) + (sint32(cb.b) * fw1) + (sint32(cc.b) * fw2)) >> 16;

		return (r << 11) | (g << 5) | b;
	}

	inline static uint16 BarycentricInterpolationDither(const FColor16& ca, Scalar w0, const FColor16& cb, Scalar w1, const FColor16& cc, Scalar w2, uint16 jitter)
	{
		const sint32 fw0 = sint32(w0 * 65536.0f);
		const sint32 fw1 = sint32(w1 * 65536.0f);
		const sint32 fw2 = sint32(w2 * 65536.0f);

		uint16 r = MAX(0, (((sint32(ca.r) * fw0) + (sint32(cb.r) * fw1) + (sint32(cc.r) * fw2)) >> 16) - jitter);
		uint16 g = MAX(0, (((sint32(ca.g) * fw0) + (sint32(cb.g) * fw1) + (sint32(cc.g) * fw2)) >> 16) - jitter);
		uint16 b = MAX(0, (((sint32(ca.b) * fw0) + (sint32(cb.b) * fw1) + (sint32(cc.b) * fw2)) >> 16) - jitter);

		return (r << 11) | (g << 5) | b;
	}

	inline static uint16 BarycentricInterpolation(const FColor16& ca, Scalar w0, const FColor16& cb, Scalar w1, const FColor16& cc, Scalar w2, uint16 ColorToAdd)
	{
		const sint32 fw0 = sint32(w0 * 65536.0f);
		const sint32 fw1 = sint32(w1 * 65536.0f);
		const sint32 fw2 = sint32(w2 * 65536.0f);

		uint16 r = ((sint32(ca.r) * fw0) + (sint32(cb.r) * fw1) + (sint32(cc.r) * fw2)) >> 16;
		uint16 g = ((sint32(ca.g) * fw0) + (sint32(cb.g) * fw1) + (sint32(cc.g) * fw2)) >> 16;
		uint16 b = ((sint32(ca.b) * fw0) + (sint32(cb.b) * fw1) + (sint32(cc.b) * fw2)) >> 16;

		if (ColorToAdd)
		{
			r = MIN(0x1F, r + (ColorToAdd  >> 11));
			g = MIN(0x3F, g + ((ColorToAdd >> 5) & 0x3F));
			b = MIN(0x1F, b + (ColorToAdd & 0x1F));
		}

		return (r << 11) | (g << 5) | b;
	}

	inline static uint16 BarycentricInterpolationDither(const FColor16& ca, Scalar w0, const FColor16& cb, Scalar w1, const FColor16& cc, Scalar w2, uint16 jitter, uint16 ColorToAdd)
	{
		const sint32 fw0 = sint32(w0 * 65536.0f);
		const sint32 fw1 = sint32(w1 * 65536.0f);
		const sint32 fw2 = sint32(w2 * 65536.0f);

		uint16 r = (((sint32(ca.r) * fw0) + (sint32(cb.r) * fw1) + (sint32(cc.r) * fw2)) >> 16) + jitter;
		uint16 g = (((sint32(ca.g) * fw0) + (sint32(cb.g) * fw1) + (sint32(cc.g) * fw2)) >> 16) + jitter;
		uint16 b = (((sint32(ca.b) * fw0) + (sint32(cb.b) * fw1) + (sint32(cc.b) * fw2)) >> 16) + jitter;

			   r = MIN(0x1F, r + (ColorToAdd  >> 11));
			   g = MIN(0x3F, g + ((ColorToAdd >> 5) & 0x3F));
			   b = MIN(0x1F, b + (ColorToAdd & 0x1F));

		return (r << 11) | (g << 5) | b;
	}

	FColor16 ScaledValue(Scalar t) const // NOTE::Assumes (r < 32, g < 64, b < 32, t >= 0 && <= 1) !
	{
		FColor16 res;

		const sint32 ft = sint32(t * 65536.0f);

		res.r = (sint32(r) * ft) >> 16;
		res.g = (sint32(g) * ft) >> 16;
		res.b = (sint32(b) * ft) >> 16;

		return res;
	}

	FColor16 ScaledValueFP(sint32 ft) const // NOTE::Assumes (r < 32, g < 64, b < 32, t >= 0 && <= 1) !
	{
		FColor16 res;

		res.r = (sint32(r) * ft) >> 16;
		res.g = (sint32(g) * ft) >> 16;
		res.b = (sint32(b) * ft) >> 16;

		return res;
	}

	FColor16 ScaledValueToWhite(sint32 t) const // NOTE::Assumes (r < 32, g < 64, b < 32, t >= 0 && <= 0xFFFF), works in similar way to FSoftwareRasterizer::PrepareLockup !
	{
		FColor16 res;

		res.r = MIN(0x1F, ((sint32(r) * t) >> 16) + (t >> 13));
		res.g = MIN(0x3F, ((sint32(g) * t) >> 16) + (t >> 12));
		res.b = MIN(0x1F, ((sint32(b) * t) >> 16) + (t >> 13));

		return res;
	}

	uint16 AddSat(uint16 ColorToAdd) const
	{
		return (MIN(0x1F, r + (ColorToAdd >> 11)) << 11) | (MIN(0x3F, g + ((ColorToAdd >> 5) & 0x3F)) << 5) | MIN(0x1F, b + (ColorToAdd & 0x1F));
	}
};

static_assert(sizeof(FColor16) ==  3, "FColor16 must be 3 bytes !");
