// (c) MX^Add

#include "RendererTypes/Matrix.h"

FMatrix FMatrix::Multiply(const FMatrix& M1, const FMatrix& M2)
{
	FMatrix Result;

    // Cache the invariants in registers
    Scalar x = M1.m[0][0];
    Scalar y = M1.m[0][1];
    Scalar z = M1.m[0][2];
    Scalar w = M1.m[0][3];

    // Perform the operation on the first row
    Result.m[0][0] = (M2.m[0][0] * x) + (M2.m[1][0] * y) + (M2.m[2][0] * z) + (M2.m[3][0] * w);
    Result.m[0][1] = (M2.m[0][1] * x) + (M2.m[1][1] * y) + (M2.m[2][1] * z) + (M2.m[3][1] * w);
    Result.m[0][2] = (M2.m[0][2] * x) + (M2.m[1][2] * y) + (M2.m[2][2] * z) + (M2.m[3][2] * w);
    Result.m[0][3] = (M2.m[0][3] * x) + (M2.m[1][3] * y) + (M2.m[2][3] * z) + (M2.m[3][3] * w);

    // Repeat for all the other rows
    x = M1.m[1][0];
    y = M1.m[1][1];
    z = M1.m[1][2];
    w = M1.m[1][3];
    Result.m[1][0] = (M2.m[0][0] * x) + (M2.m[1][0] * y) + (M2.m[2][0] * z) + (M2.m[3][0] * w);
    Result.m[1][1] = (M2.m[0][1] * x) + (M2.m[1][1] * y) + (M2.m[2][1] * z) + (M2.m[3][1] * w);
    Result.m[1][2] = (M2.m[0][2] * x) + (M2.m[1][2] * y) + (M2.m[2][2] * z) + (M2.m[3][2] * w);
    Result.m[1][3] = (M2.m[0][3] * x) + (M2.m[1][3] * y) + (M2.m[2][3] * z) + (M2.m[3][3] * w);

    x = M1.m[2][0];
    y = M1.m[2][1];
    z = M1.m[2][2];
    w = M1.m[2][3];
    Result.m[2][0] = (M2.m[0][0] * x) + (M2.m[1][0] * y) + (M2.m[2][0] * z) + (M2.m[3][0] * w);
    Result.m[2][1] = (M2.m[0][1] * x) + (M2.m[1][1] * y) + (M2.m[2][1] * z) + (M2.m[3][1] * w);
    Result.m[2][2] = (M2.m[0][2] * x) + (M2.m[1][2] * y) + (M2.m[2][2] * z) + (M2.m[3][2] * w);
    Result.m[2][3] = (M2.m[0][3] * x) + (M2.m[1][3] * y) + (M2.m[2][3] * z) + (M2.m[3][3] * w);

    x = M1.m[3][0];
    y = M1.m[3][1];
    z = M1.m[3][2];
    w = M1.m[3][3];
    Result.m[3][0] = (M2.m[0][0] * x) + (M2.m[1][0] * y) + (M2.m[2][0] * z) + (M2.m[3][0] * w);
    Result.m[3][1] = (M2.m[0][1] * x) + (M2.m[1][1] * y) + (M2.m[2][1] * z) + (M2.m[3][1] * w);
    Result.m[3][2] = (M2.m[0][2] * x) + (M2.m[1][2] * y) + (M2.m[2][2] * z) + (M2.m[3][2] * w);
    Result.m[3][3] = (M2.m[0][3] * x) + (M2.m[1][3] * y) + (M2.m[2][3] * z) + (M2.m[3][3] * w);

    return Result;
}

FMatrix FMatrix::MakeLookAt(const FVector3D &EyePosition, const FVector3D &FocusPosition, const FVector3D &UpDirection)
{ 
    FVector3D EyeDirection = EyePosition - FocusPosition;
    
    FVector3D R2 = EyeDirection;          R2.Normalize();
    FVector3D R0 = UpDirection.Cross(R2); R0.Normalize();
    FVector3D R1 = R2.Cross(R0);

    FVector3D NegEyePosition = -EyePosition;

    Scalar D0 = R0.Dot(NegEyePosition);
    Scalar D1 = R1.Dot(NegEyePosition);
    Scalar D2 = R2.Dot(NegEyePosition);

    return FMatrix(R0.x, R1.x, R2.x, 0,
                   R0.y, R1.y, R2.y, 0,
                   R0.z, R1.z, R2.z, 0,
                   D0,   D1,   D2,   1);
}

FMatrix FMatrix::MakePerspectiveProjection(uint8 FOV, uint16 Near, uint16 Far, Scalar Aspect)
{
    Scalar SinFov = FScalar::Sin(uint8(FOV/2));
    Scalar CosFov = FScalar::Cos(uint8(FOV/2));

    Scalar Height = CosFov / SinFov;
    Scalar Width  = Height / Aspect;
    Scalar Range  = Scalar(sint32(Far)) / Scalar(sint32(Near) - sint32(Far));
    Scalar Zange  = Range * Scalar(sint32(Near));

    return FMatrix(Width,   0,      0,      0,
                   0,       Height, 0,      0,
                   0,       0,      Range, -1,
                   0,       0,      Zange,  0);
}

FVector3D FMatrix::TransformNormal(const FVector3D& Nrm) const
{
    return FVector3D(m[0][0], m[0][1], m[0][2]) * Nrm.x +
           FVector3D(m[1][0], m[1][1], m[1][2]) * Nrm.y +
           FVector3D(m[2][0], m[2][1], m[2][2]) * Nrm.z ;
}

FVector3D FMatrix::TransformPosition(const FVector3D &Pos) const
{
    return FVector3D(m[0][0], m[0][1], m[0][2]) * Pos.x +
           FVector3D(m[1][0], m[1][1], m[1][2]) * Pos.y +
           FVector3D(m[2][0], m[2][1], m[2][2]) * Pos.z +
           FVector3D(m[3][0], m[3][1], m[3][2])         ;
}

FVector4D FMatrix::TransformPositionFull(const FVector3D &Pos) const
{
    return FVector4D(m[0][0], m[0][1], m[0][2], m[0][3]) * Pos.x +
           FVector4D(m[1][0], m[1][1], m[1][2], m[1][3]) * Pos.y +
           FVector4D(m[2][0], m[2][1], m[2][2], m[2][3]) * Pos.z +
           FVector4D(m[3][0], m[3][1], m[3][2], m[3][3])         ;
}

FVector4D FMatrix::TransformPosition(const FVector4D &Pos) const
{
    return FVector4D(m[0][0], m[0][1], m[0][2], m[0][3]) * Pos.x +
           FVector4D(m[1][0], m[1][1], m[1][2], m[1][3]) * Pos.y +
           FVector4D(m[2][0], m[2][1], m[2][2], m[2][3]) * Pos.z +
           FVector4D(m[3][0], m[3][1], m[3][2], m[3][3]) * Pos.w ;
}

Scalar FMatrix::GetScaleX() const
{
    return FVector3D(m[0][0], m[0][1], m[0][2]).Length();
}

Scalar FMatrix::GetScaleY() const
{
    return FVector3D(m[1][0], m[1][1], m[1][2]).Length();
}

Scalar FMatrix::GetScaleZ() const
{
    return FVector3D(m[2][0], m[2][1], m[2][2]).Length();
}
