// (c) MX^Add
#pragma once

#include "BaseTypes/BaseTypes.h"
#include "BaseTypes/ScalarType.h"

class FMatrix;

class FVector2D
{
public:
	Scalar x, y;

	          FVector2D() {}
    constexpr FVector2D(Scalar _x, Scalar _y) : x(_x), y(_y) {}

    inline Scalar Length() const
    {
        return FScalar::Sqrt(x*x + y*y);
    }

    inline void Normalize()
    {
        Scalar il = FScalar::InvSqrt(x*x + y*y);

        x *= il;
        y *= il;

        return;
    }

    inline Scalar Dot(const FVector2D& other) const
    {
        return x * other.x + y * other.y;
    }

    FVector2D operator - () const
    {
        return FVector2D(-x, -y);
    }

	FVector2D operator + (const FVector2D &other) const 
	{
        return FVector2D(x + other.x, y + other.y);
    }

    FVector2D operator - (const FVector2D &other) const 
	{
        return FVector2D(x - other.x, y - other.y);
    }

    FVector2D operator * (const FVector2D &other) const 
    {
        return FVector2D(x * other.x, y * other.y);
    }

    FVector2D operator / (const FVector2D &other) const 
    {
        return FVector2D(x / other.x, y / other.y);
    }

    FVector2D& operator += (const FVector2D &other) 
    {
        x += other.x;
        y += other.y;
        return *this;
    }

    FVector2D& operator -= (const FVector2D &other) 
    {
        x -= other.x;
        y -= other.y;
        return *this;
    }

    FVector2D& operator *= (const FVector2D& other) 
    {
        x *= other.x;
        y *= other.y;
        return *this;
    }

    FVector2D& operator /= (const FVector2D &other) 
    {
        x /= other.x;
        y /= other.y;
        return *this;
    }

    FVector2D operator * (Scalar other) const 
    {
        return FVector2D(x * other, y * other);
    }

    FVector2D operator / (Scalar other) const 
    {
        return FVector2D(x / other, y / other);
    }

    FVector2D& operator *= (Scalar other) 
    {
        x *= other;
        y *= other;
        return *this;
    }

    FVector2D& operator /= (Scalar other) 
    {
        x /= other;
        y /= other;
        return *this;
    }

    FVector2D Abs() const
    {
        return FVector2D(FScalar::Abs(x), FScalar::Abs(y));
    }

    inline void Lerp(const FVector2D& a, const FVector2D& b, Scalar t)
    {
        *this = a + (b - a) * t;
        return;
    }

    inline static FVector2D Min(const FVector2D& a, const FVector2D& b)
    {
        return FVector2D(FScalar::Min(a.x, b.x), FScalar::Min(a.y, b.y));
    }

    inline static FVector2D Max(const FVector2D& a, const FVector2D& b)
    {
        return FVector2D(FScalar::Max(a.x, b.x), FScalar::Max(a.y, b.y));
    }

    constexpr bool operator == (const FVector2D &other) const { return x == other.x && y == other.y; }
    constexpr bool operator != (const FVector2D &other) const { return x != other.x || y == other.y; }
};

class FVector3D
{
public:
	Scalar x, y, z;

              FVector3D() {}
    constexpr FVector3D(Scalar _x, Scalar _y, Scalar _z) : x(_x), y(_y), z(_z) {}

    inline Scalar Length() const
    {
        return FScalar::Sqrt(x*x + y*y + z*z);
    }

	inline void Normalize()
    {
        Scalar il = FScalar::InvSqrt(x*x + y*y + z*z);

        x *= il;
        y *= il;
        z *= il;

        return;
    }

    inline FVector3D Cross(const FVector3D& other) const
    {
        return FVector3D(y * other.z - z * other.y,
                         z * other.x - x * other.z,
                         x * other.y - y * other.x);
    }

    inline Scalar Dot(const FVector3D& other) const
    {
        return x * other.x + y * other.y + z * other.z;
    }

    FVector3D operator - () const
    {
        return FVector3D(-x, -y, -z);
    }

	FVector3D operator + (const FVector3D &other) const 
	{
        return FVector3D(x + other.x, y + other.y, z + other.z);
    }

    FVector3D operator - (const FVector3D &other) const 
	{
        return FVector3D(x - other.x, y - other.y, z - other.z);
    }

    FVector3D operator * (const FVector3D &other) const 
    {
        return FVector3D(x * other.x, y * other.y, z * other.z);
    }

    FVector3D operator / (const FVector3D &other) const 
    {
        return FVector3D(x / other.x, y / other.y, z / other.z);
    }

    FVector3D& operator += (const FVector3D &other) 
    {
        x += other.x;
        y += other.y;
        z += other.z;
        return *this;
    }

    FVector3D& operator -= (const FVector3D &other) 
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        return *this;
    }

    FVector3D& operator *= (const FVector3D& other) 
    {
        x *= other.x;
        y *= other.y;
        z *= other.z;
        return *this;
    }

    FVector3D& operator /= (const FVector3D &other) 
    {
        x /= other.x;
        y /= other.y;
        z /= other.z;
        return *this;
    }

    FVector3D operator * (Scalar other) const 
    {
        return FVector3D(x * other, y * other, z * other);
    }

    FVector3D operator / (Scalar other) const 
    {
        return FVector3D(x / other, y / other, z / other);
    }

    FVector3D& operator *= (Scalar other) 
    {
        x *= other;
        y *= other;
        z *= other;
        return *this;
    }

    FVector3D& operator /= (Scalar other) 
    {
        x /= other;
        y /= other;
        z /= other;
        return *this;
    }

    inline void Lerp(const FVector3D& a, const FVector3D& b, Scalar t)
    {
        *this = a + (b - a) * t;
        return;
    }

    inline FVector3D Abs() const
    {
        return FVector3D(FScalar::Abs(x), FScalar::Abs(y), FScalar::Abs(z));
    }

    inline static FVector3D Min(const FVector3D& a, const FVector3D& b)
    {
        return FVector3D(FScalar::Min(a.x, b.x), FScalar::Min(a.y, b.y), FScalar::Min(a.z, b.z));
    }

    inline static FVector3D Max(const FVector3D& a, const FVector3D& b)
    {
        return FVector3D(FScalar::Max(a.x, b.x), FScalar::Max(a.y, b.y), FScalar::Max(a.z, b.z));
    }

    constexpr bool operator == (const FVector3D &other) const { return x == other.x && y == other.y && z == other.z; }
    constexpr bool operator != (const FVector3D &other) const { return x != other.x || y == other.y || z == other.z; }
};

class FVector4D
{
public:
	Scalar x, y, z, w;

              FVector4D() {}
    constexpr FVector4D(Scalar _x, Scalar _y, Scalar _z, Scalar _w) : x(_x), y(_y), z(_z), w(_w) {}

    inline Scalar Dot(const FVector4D& other) const
    {
        return x * other.x + y * other.y + z * other.z + w * other.w;
    }

    inline Scalar Dot(const FVector3D& other) const
    {
        return x * other.x + y * other.y + z * other.z + w;
    }

    inline void NormalizePlane()
    {
        Scalar il = FScalar::InvSqrt(x*x + y*y + z*z);

        x *= il;
        y *= il;
        z *= il;
        w *= il;

        return;
    }

    inline void Normalize()
    {
        Scalar il = FScalar::InvSqrt(x*x + y*y + z*z + w*w);

        x *= il;
        y *= il;
        z *= il;
        w *= il;

        return;
    }

    FVector4D operator - () const
    {
        return FVector4D(-x, -y, -z, -w);
    }

	FVector4D operator + (const FVector4D &other) const 
	{
        return FVector4D(x + other.x, y + other.y, z + other.z, w + other.w);
    }

    FVector4D operator - (const FVector4D &other) const 
	{
        return FVector4D(x - other.x, y - other.y, z - other.z, w - other.w);
    }

    FVector4D operator * (const FVector4D &other) const 
    {
        return FVector4D(x * other.x, y * other.y, z * other.z, w * other.w);
    }

    FVector4D operator / (const FVector4D &other) const 
    {
        return FVector4D(x / other.x, y / other.y, z / other.z, w / other.w);
    }

    FVector4D& operator += (const FVector4D &other) 
    {
        x += other.x;
        y += other.y;
        z += other.z;
        w += other.w;
        return *this;
    }

    FVector4D& operator -= (const FVector4D &other) 
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        w -= other.w;
        return *this;
    }

    FVector4D& operator *= (const FVector4D& other) 
    {
        x *= other.x;
        y *= other.y;
        z *= other.z;
        w *= other.w;
        return *this;
    }

    FVector4D& operator /= (const FVector4D &other) 
    {
        x /= other.x;
        y /= other.y;
        z /= other.z;
        w /= other.w;
        return *this;
    }

    FVector4D operator * (Scalar other) const 
    {
        return FVector4D(x * other, y * other, z * other, w * other);
    }

    FVector4D operator / (Scalar other) const 
    {
        return FVector4D(x / other, y / other, z / other, w / other);
    }

    FVector4D& operator *= (Scalar other) 
    {
        x *= other;
        y *= other;
        z *= other;
        w *= other;
        return *this;
    }

    FVector4D& operator /= (Scalar other) 
    {
        x /= other;
        y /= other;
        z /= other;
        w /= other;
        return *this;
    }

    inline FVector4D Abs() const
    {
        return FVector4D(FScalar::Abs(x), FScalar::Abs(y), FScalar::Abs(z), FScalar::Abs(w));
    }

    bool SphereInFrustum(const FMatrix &Local, const FVector4D *Planes) const;

    static bool SphereInFrustum(const FVector3D &Center, Scalar Radius, const FVector4D *Planes);

    void Slerp(const FVector4D &a, const FVector4D &b, Scalar t);

    inline void Lerp(const FVector4D& a, const FVector4D& b, Scalar t)
    {
        *this = a + (b - a) * t;
        return;
    }

    FMatrix FromQuaterion() const;

    inline static FVector4D Min(const FVector4D& a, const FVector4D& b)
    {
        return FVector4D(FScalar::Min(a.x, b.x), FScalar::Min(a.y, b.y), FScalar::Min(a.z, b.z), FScalar::Min(a.w, b.w));
    }

    inline static FVector4D Max(const FVector4D& a, const FVector4D& b)
    {
        return FVector4D(FScalar::Max(a.x, b.x), FScalar::Max(a.y, b.y), FScalar::Max(a.z, b.z), FScalar::Max(a.w, b.w));
    }

    constexpr bool operator == (const FVector4D &other) const { return x == other.x && y == other.y && z == other.z && w == other.w; }
    constexpr bool operator != (const FVector4D &other) const { return x != other.x || y == other.y || z == other.z || w == other.w; }
};

class FVector4Di
{
public:
    sint32 x, y, z, w;

    FVector4Di() {}
    constexpr FVector4Di(sint32 _x, sint32 _y, sint32 _z, sint32 _w) : x(_x), y(_y), z(_z), w(_w) {}

    FVector4Di operator - () const
    {
        return FVector4Di(-x, -y, -z, -w);
    }

    FVector4Di operator + (const FVector4Di &other) const 
	{
        return FVector4Di(x + other.x, y + other.y, z + other.z, w + other.w);
    }

    FVector4Di operator - (const FVector4Di &other) const 
	{
        return FVector4Di(x - other.x, y - other.y, z - other.z, w - other.w);
    }

    FVector4Di operator * (const FVector4Di &other) const 
    {
        return FVector4Di(x * other.x, y * other.y, z * other.z, w * other.w);
    }

    FVector4Di operator / (const FVector4Di &other) const 
    {
        return FVector4Di(x / other.x, y / other.y, z / other.z, w / other.w);
    }

    FVector4Di& operator += (const FVector4Di &other) 
    {
        x += other.x;
        y += other.y;
        z += other.z;
        w += other.w;
        return *this;
    }

    FVector4Di& operator -= (const FVector4Di &other) 
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        w -= other.w;
        return *this;
    }

    FVector4Di& operator *= (const FVector4Di& other) 
    {
        x *= other.x;
        y *= other.y;
        z *= other.z;
        w *= other.w;
        return *this;
    }

    FVector4Di& operator /= (const FVector4Di &other) 
    {
        x /= other.x;
        y /= other.y;
        z /= other.z;
        w /= other.w;
        return *this;
    }
};

static_assert(sizeof(FVector2D) ==  8, "FVector2D must be 8 bytes !");
static_assert(sizeof(FVector3D) == 12, "FVector3D must be 12 bytes !");
static_assert(sizeof(FVector4D) == 16, "FVector4D must be 16 bytes !");
static_assert(sizeof(FVector4Di) == 16, "FVector4Di must be 16 bytes !");
