// VGA Video output
// Modified implementation, based on pico SDK

#ifndef SCANVIDEO_scanvideo_H_
#define SCANVIDEO_scanvideo_H_

#include "pico/types.h"
#include "hardware/pio.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef PICO_SCANVIDEO_SCANLINE_BUFFER_COUNT
#define PICO_SCANVIDEO_SCANLINE_BUFFER_COUNT 16
#endif

#ifndef PICO_SCANVIDEO_ENABLE_CLOCK_PIN
#define PICO_SCANVIDEO_ENABLE_CLOCK_PIN 0
#endif

#ifndef PICO_SCANVIDEO_ENABLE_DEN_PIN
#define PICO_SCANVIDEO_ENABLE_DEN_PIN 0
#endif

#ifndef PICO_SCANVIDEO_COLOR_PIN_BASE
#define PICO_SCANVIDEO_COLOR_PIN_BASE 0
#endif

#ifndef PICO_SCANVIDEO_COLOR_PIN_COUNT
#define PICO_SCANVIDEO_COLOR_PIN_COUNT 16
#endif

#ifndef PICO_SCANVIDEO_SYNC_PIN_BASE
#define PICO_SCANVIDEO_SYNC_PIN_BASE (PICO_SCANVIDEO_COLOR_PIN_BASE + PICO_SCANVIDEO_COLOR_PIN_COUNT)
#endif

#ifndef PICO_SCANVIDEO_MAX_SCANLINE_BUFFER_WORDS
#define PICO_SCANVIDEO_MAX_SCANLINE_BUFFER_WORDS 16
#endif

#define BPP 16

extern const uint32_t video_clock_freq;

typedef struct scanvideo_timing 
{
    uint32_t clock_freq;

    uint16_t h_active;
    uint16_t v_active;

    uint16_t h_front_porch;
    uint16_t h_pulse;
    uint16_t h_total;
    uint8_t  h_sync_polarity;

    uint16_t v_front_porch;
    uint16_t v_pulse;
    uint16_t v_total;
    uint8_t  v_sync_polarity;

    uint8_t  enable_clock;
    uint8_t  clock_polarity;

    uint8_t  enable_den;
} scanvideo_timing_t;

typedef struct scanvideo_pio_program scanvideo_pio_program_t;

typedef struct scanvideo_mode 
{
    const scanvideo_timing_t      *default_timing;
    const scanvideo_pio_program_t *pio_program;

    uint16_t width;
    uint16_t height;
    uint8_t  xscale; // 1 == normal, 2 == double wide etc. up to what pio timing allows (not sure I have an assert based on delays)
    uint16_t yscale; // same for y scale (except any yscale is possible)

    // if > 1 then yscale is divided by this to provide the effective yscale;
    // note that yscale must be > yscale_denominator; i.e. only stretching is supported
    uint16_t yscale_denominator;
} scanvideo_mode_t;

extern bool scanvideo_setup(const scanvideo_mode_t *mode);
extern bool scanvideo_setup_with_timing(const scanvideo_mode_t *mode, const scanvideo_timing_t *timing);
extern void scanvideo_timing_enable(bool enable);

typedef struct scanvideo_scanline_buffer 
{
    uint32_t scanline_id;
    uint32_t *data;
    uint16_t data_used;
    uint16_t data_max;
    void *user_data;
    uint8_t status;
} scanvideo_scanline_buffer_t;

enum 
{
    SCANLINE_OK = 1,
    SCANLINE_ERROR,
    SCANLINE_SKIPPED
};

// note frame numbers wrap
static inline uint16_t scanvideo_frame_number(uint32_t scanline_id) 
{
    return (uint16_t) (scanline_id >> 16u);
}

static inline uint16_t scanvideo_scanline_number(uint32_t scanline_id) 
{
    return (uint16_t) scanline_id;
}

/**
 * @return the current vga mode (if there is one)
 */
extern scanvideo_mode_t scanvideo_get_mode();

/**
 * @return the next scanline_id to be displayed (may be from the next frame)
 */
extern uint32_t scanvideo_get_next_scanline_id();

/**
 * @return true if in the vblank interval
 */
extern bool scanvideo_in_vblank();

/**
 * @return true if in the hblank interval (or more accurately scanline data is not currently being sent to the PIO, which roughly corresponds, but is not exact). Note also that in
 * yscale-d modes, there are height * yscale hblank intervals per frame.
 */
extern bool scanvideo_in_hblank();

extern void scanvideo_wait_for_vblank();

extern uint32_t scanvideo_wait_for_scanline_complete(uint32_t scanline_id);

/**
 * Acquire a scanline that needs generating. The scanline_id field indicates which scanline is required.
 *
 * This method may be called concurrently
 *
 * @param block true to block if the vga system is not ready to generate a new scanline
 * @return the scanline_buffer or NULL if block is false, and the vga system is not ready
 */
scanvideo_scanline_buffer_t *scanvideo_begin_scanline_generation(bool block);

/**
 * Return a scanline that has been generated / or at least the client is done with.
 *
 * The status field indicates whether the scanline was actually generated OK
 *
 * This method may be called concurrently (for different buffers)
 *
 * @param scanline_buffer \todo
 */
void scanvideo_end_scanline_generation(scanvideo_scanline_buffer_t *scanline_buffer);

typedef uint (*scanvideo_scanline_repeat_count_fn)(uint32_t scanline_id);
void scanvideo_set_scanline_repeat_fn(scanvideo_scanline_repeat_count_fn fn);

extern const scanvideo_timing_t vga_timing_640x480_60_default;
extern const scanvideo_mode_t vga_mode_320x240_60;

// mode implementation

struct scanvideo_pio_program 
{
    const pio_program_t *program;
    const uint8_t entry_point;
    bool (*adapt_for_mode)(const scanvideo_pio_program_t *program, const scanvideo_mode_t *mode, scanvideo_scanline_buffer_t *missing_scanline_buffer, uint16_t *modifiable_instructions);
    pio_sm_config (*configure_pio)(pio_hw_t *pio, uint sm, uint offset);
};

extern const scanvideo_pio_program_t video_24mhz_composable;

extern void scanvideo_default_configure_pio(pio_hw_t *pio, uint sm, uint offset, pio_sm_config *config, bool overlay);

#ifndef PICO_SPINLOCK_ID_VIDEO_SCANLINE_LOCK
#define PICO_SPINLOCK_ID_VIDEO_SCANLINE_LOCK 2
#endif

#ifndef PICO_SPINLOCK_ID_VIDEO_FREE_LIST_LOCK
#define PICO_SPINLOCK_ID_VIDEO_FREE_LIST_LOCK 3
#endif

#ifndef PICO_SPINLOCK_ID_VIDEO_DMA_LOCK
#define PICO_SPINLOCK_ID_VIDEO_DMA_LOCK 4
#endif

#ifndef PICO_SPINLOCK_ID_VIDEO_IN_USE_LOCK
#define PICO_SPINLOCK_ID_VIDEO_IN_USE_LOCK 5
#endif

#define PICO_SCANVIDEO_ALPHA_MASK (1u << PICO_SCANVIDEO_ALPHA_PIN)

#ifndef PICO_SCANVIDEO_PIXEL_FROM_RGB8
#define PICO_SCANVIDEO_PIXEL_FROM_RGB8(r, g, b) ((((b)>>3u)<<PICO_SCANVIDEO_PIXEL_BSHIFT)|(((g)>>3u)<<PICO_SCANVIDEO_PIXEL_GSHIFT)|(((r)>>3u)<<PICO_SCANVIDEO_PIXEL_RSHIFT))
#endif

#ifndef PICO_SCANVIDEO_PIXEL_FROM_RGB5
#define PICO_SCANVIDEO_PIXEL_FROM_RGB5(r, g, b) (((b)<<PICO_SCANVIDEO_PIXEL_BSHIFT)|((g)<<PICO_SCANVIDEO_PIXEL_GSHIFT)|((r)<<PICO_SCANVIDEO_PIXEL_RSHIFT))
#endif

#ifndef PICO_SCANVIDEO_R5_FROM_PIXEL
#define PICO_SCANVIDEO_R5_FROM_PIXEL(p) (((p)>>PICO_SCANVIDEO_PIXEL_RSHIFT)&0x1f)
#endif

#ifndef PICO_SCANVIDEO_G5_FROM_PIXEL
#define PICO_SCANVIDEO_G5_FROM_PIXEL(p) (((p)>>PICO_SCANVIDEO_PIXEL_GSHIFT)&0x1f)
#endif

#ifndef PICO_SCANVIDEO_B5_FROM_PIXEL
#define PICO_SCANVIDEO_B5_FROM_PIXEL(p) (((p)>>PICO_SCANVIDEO_PIXEL_BSHIFT)&0x1f)
#endif

#ifdef __cplusplus
}
#endif
#endif //_VIDEO_H
