// Note: import removed for standalone usage. React is expected as a global in the browser.
// Destructure the React hooks from the global React object for convenience in this file.
const { useCallback, useEffect, useMemo, useRef, useState } = React;


/*
PALETKOWO – single-file React build

SEKCJE (Krok 7 – porządek techniczny):
  A) UI PRIMITIVES (Card/Btn/Switch/Slider)
  B) UTILS (math/geometry/theme helpers)
  C) UI STYLES (shared)
  D) ARCADE UI (knobs + mechanical ribs)
  E) GAME CONSTANTS + MODECFG + THEMES
  F) AUDIO ENGINE (WebAudio)
  G) SELF-TESTS (console.assert)
  H) MAIN COMPONENT
     H1) State + refs
     H2) Audio glue (ensureAudio/syncMusic)
     H3) Game state init/reset
     H4) Input + AI
     H5) Physics (ball/paddles/scoring/game over)
     H6) Render (canvas)
     H7) UI (React layout)
*/

// =========================
// Minimal UI primitives (no external deps)
// =========================
function Card({ children, className = "", style }) {
  return (
    <div
      className={"rounded-none border backdrop-blur p-0 overflow-hidden " + className}
      style={{
        background: "var(--ui-soft)",
        borderColor: "var(--ui-border)",
        color: "var(--ui-text)",
        fontFamily: "var(--ui-font)",
        ...style,
      }}
    >
      {children}
    </div>
  );
}
function CardContent({ className = "", children }) {
  return <div className={className}>{children}</div>;
}
function Btn({
  variant = "default",
  size = "md",
  className = "",
  style,
  disabled = false,
  ...props
}) {
  const base =
    "inline-flex items-center justify-center rounded-none font-semibold select-none transition ";
  const sizes = {
    sm: "h-8 px-3 text-base",
    md: "h-9 px-3 text-base",
    lg: "h-10 px-4 text-base",
  };

  const vStyle = (() => {
    if (variant === "outline")
      return {
        background: "transparent",
        color: "var(--ui-text)",
        border: "1px solid var(--ui-border)",
      };
    if (variant === "secondary")
      return {
        background: "var(--ui-soft)",
        color: "var(--ui-text)",
        border: "1px solid var(--ui-border)",
      };
    return {
      background: disabled ? "var(--ui-border)" : "var(--ui-accent)",
      color: "var(--ui-accentText)",
      border: "1px solid var(--ui-accentBorder)",
    };
  })();

  return (
    <button
      className={
        [
          base,
          sizes[size] || sizes.md,
          className,
          disabled ? "opacity-40 cursor-not-allowed" : "cursor-pointer",
        ].join(" ")
      }
      style={{ fontFamily: "var(--ui-font)", ...vStyle, ...style }}
      disabled={disabled}
      {...props}
    />
  );
}
function Label({ children }) {
  return (
    <div
      className="text-base font-medium"
      style={{ fontFamily: "var(--ui-font)", color: "var(--ui-text)" }}
    >
      {children}
    </div>
  );
}
function Switch({ checked, onCheckedChange, disabled = false }) {
  return (
    <button
      type="button"
      onClick={() => {
        if (disabled) return;
        onCheckedChange?.(!checked);
      }}
      className={
        "relative inline-flex h-6 w-11 items-center rounded-none border transition" +
        (disabled ? " opacity-40 cursor-not-allowed" : "")
      }
      style={{
        background: checked ? "var(--ui-accent)" : "var(--ui-panel)",
        borderColor: checked ? "var(--ui-accentBorder)" : "var(--ui-border)",
      }}
      aria-pressed={checked}
    >
      <span
        className={
          "inline-block h-5 w-5 transform rounded-none shadow transition " +
          (checked ? "translate-x-5" : "translate-x-1")
        }
        style={{ background: "var(--ui-knob)" }}
      />
    </button>
  );
}
function Slider({ value, min, max, step, onValueChange, disabled = false }) {
  const v = Array.isArray(value) ? value[0] : value;
  const pct = ((v - min) / (max - min)) * 100;

  const onChange = (e) => {
    if (disabled) return;
    onValueChange?.([Number(e.target.value)]);
  };

  return (
    <div className="w-full" style={{ position: "relative", height: 18 }}>
      {/* Track */}
      <div
        style={{
          position: "absolute",
          left: 0,
          right: 0,
          top: 4,
          height: 10,
          border: "1px solid var(--ui-border)",
          background: "var(--ui-soft)",
        }}
      />

      {/* Filled (left) */}
      <div
        style={{
          position: "absolute",
          left: 0,
          top: 4,
          height: 10,
          width: `${clamp(pct, 0, 100)}%`,
          background: "var(--ui-accent)",
        }}
      />

      {/* Unfilled (right) */}
      <div
        style={{
          position: "absolute",
          left: `${clamp(pct, 0, 100)}%`,
          right: 0,
          top: 4,
          height: 10,
          background: "var(--ui-sliderRight)",
        }}
      />

      {/* Knob */}
      <div
        style={{
          position: "absolute",
          top: 1,
          left: `calc(${clamp(pct, 0, 100)}% - 6px)`,
          width: 12,
          height: 16,
          border: "1px solid var(--ui-border)",
          background: "var(--ui-knob)",
        }}
      />

      {/* Disabled overlay (blenda jak Switch) */}
      {disabled && (
        <div
          style={{
            position: "absolute",
            left: 0,
            right: 0,
            top: 0,
            bottom: 0,
            background: "rgba(0,0,0,0.25)",
            pointerEvents: "none",
          }}
        />
      )}

      {/* Native input (interaction layer) */}
      <input
        className="w-full"
        style={{
          position: "absolute",
          left: 0,
          top: 0,
          height: 18,
          margin: 0,
          opacity: 0,
          cursor: disabled ? "not-allowed" : "pointer",
        }}
        type="range"
        value={v}
        min={min}
        max={max}
        step={step}
        onChange={onChange}
        disabled={disabled}
      />
    </div>
  );
}

// =========================
// Utils (math, color, geometry)
// =========================
const clamp = (v, lo, hi) => Math.max(lo, Math.min(hi, v));

// AI helper: UI poziom 1..10 -> skill 0..1 (1 = najtrudniej)
const aiSkillFromUiLevel = (uiLevel) => {
  const uiLvl = clamp(Number(uiLevel) || 1, 1, 10);
  // 1 ma dawać NAJWIĘKSZĄ trudność
  return clamp((10 - uiLvl) / 9, 0, 1);
};

const blendHex = (hexA, hexB, t) => {
  const norm = (h) => String(h || "").replace("#", "");
  const a = norm(hexA);
  const b = norm(hexB);
  const toRgb = (h) => {
    const hh = h.length === 3 ? h.split("").map((c) => c + c).join("") : h;
    const n = parseInt(hh, 16);
    return { r: (n >> 16) & 255, g: (n >> 8) & 255, b: n & 255 };
  };
  if (!a || !b) return hexA || hexB || "#fff";
  const A = toRgb(a);
  const B = toRgb(b);
  const mix = (x, y) => Math.round(x + (y - x) * t);
  return `rgb(${mix(A.r, B.r)} ${mix(A.g, B.g)} ${mix(A.b, B.b)})`;
};

const intersects = (ax, ay, aw, ah, bx, by, bw, bh) =>
  ax < bx + bw && ax + aw > bx && ay < by + bh && ay + ah > by;

const ensureRoundRect = (ctx) => {
  if (typeof ctx.roundRect === "function") return;
  ctx.roundRect = function roundRect(x, y, w, h, r) {
    const rr = Math.min(r, w / 2, h / 2);
    this.beginPath();
    this.moveTo(x + rr, y);
    this.arcTo(x + w, y, x + w, y + h, rr);
    this.arcTo(x + w, y + h, x, y + h, rr);
    this.arcTo(x, y + h, x, y, rr);
    this.arcTo(x, y, x + w, y, rr);
    this.closePath();
    return this;
  };
};

// =========================
// UI shared styles
// =========================
const PANEL_TITLE_STYLE = {
  fontFamily: "var(--ui-font)",
  color: "var(--ui-text)",
  fontSize: 19,
};
const PANEL_MUTED_STYLE = {
  fontFamily: "var(--ui-font)",
  color: "var(--ui-muted)",
};
const DIVIDER_STYLE = {
  height: 1,
  background: "var(--ui-border)",
  opacity: 0.4,
  margin: "18px 0",
};
const CANVAS_FRAME_STYLE = {
  borderColor: "var(--ui-border)",
  background: "var(--ui-soft)",
};

function Divider({ style }) {
  return <div style={{ ...DIVIDER_STYLE, ...(style || {}) }} />;
}

// =========================
// Arcade UI (Krok 2)
// =========================
function ArcadeKnob({ angle = 0 }) {
  return (
    <div className="flex flex-col items-center gap-2 select-none" style={{ width: 150 }}>
      <div
        style={{
          width: 126,
          height: 126,
          borderRadius: 9999,
          border: "1px solid var(--ui-border)",
          background: "var(--ui-soft)",
          boxShadow: "inset 0 0 0 2px rgba(0,0,0,0.15)",
          position: "relative",
          overflow: "hidden",
          display: "flex",
          alignItems: "center",
          justifyContent: "center",
        }}
      >
        {/* karbowany rant (inspiracja: konsola PONG) */}
        <div
          style={{
            position: "absolute",
            inset: 2,
            borderRadius: 9999,
            background:
              "repeating-conic-gradient(from 0deg, rgba(255,255,255,0.10) 0deg 5deg, rgba(0,0,0,0.18) 5deg 10deg)",
            boxShadow: "inset 0 0 0 1px rgba(0,0,0,0.22)",
            opacity: 0.95,
          }}
        />

        {/* płaski korpus pokrętła */}
        <div
          style={{
            position: "absolute",
            inset: 18,
            borderRadius: 9999,
            background: "var(--ui-panel)",
            border: "1px solid rgba(0,0,0,0.22)",
            boxShadow: "inset 0 10px 18px rgba(0,0,0,0.22)",
          }}
        />

        {/* metalowy kapsel (mniej cieni) */}
        <div
          style={{
            position: "absolute",
            width: 34,
            height: 34,
            borderRadius: 9999,
            border: "1px solid rgba(255,255,255,0.18)",
            background:
              "radial-gradient(circle at 35% 30%, rgba(255,255,255,0.75), rgba(170,170,170,0.28) 50%, rgba(70,70,70,0.45) 100%)",
            boxShadow: "inset 0 2px 5px rgba(0,0,0,0.28)",
          }}
        />

        {/* wskaźnik obrotu: „notch” na rancie */}
        <div
          style={{
            position: "absolute",
            inset: 0,
            transform: `rotate(${angle}deg)`,
            transformOrigin: "50% 50%",
            pointerEvents: "none",
          }}
        >
          <div
            style={{
              position: "absolute",
              top: 6,
              left: "50%",
              width: 20,
              height: 8,
              transform: "translateX(-50%)",
              background: "var(--ui-accent)",
              boxShadow: "0 0 0 1px var(--ui-accentBorder)",
            }}
          />
        </div>
      </div>

      {/* no label displayed for knob */}
    </div>
  );
}

// Dekoracyjny „radiator” między planszą a panelem arcade (łatwy do edycji)
// Dekoracyjny „radiator” między planszą a panelem arcade (minimalna metalowa faktura, zależna od kolorystyki)
function MechanicalRibs({ h = 56, ribH = 18, sideInset = 14, className = "" }) {
  return (
    <div
      className={className}
      style={{
        height: h,
        display: "flex",
        alignItems: "center",
        justifyContent: "center",
      }}
    >
      <div
        style={{
          width: "100%",
          height: ribH,
          margin: `0 ${sideInset}px`,
          // liniowy wzór: poziome linie z rosnącymi odstępami; usuwamy imitację siatki
          background: `
            linear-gradient(180deg,
              rgba(255,255,255,0.10),
              rgba(0,0,0,0.25)
            ),
            linear-gradient(to bottom,
              var(--ui-border) 0px 1px,
              transparent 1px 3px,
              var(--ui-border) 3px 4px,
              transparent 4px 7px,
              var(--ui-border) 7px 8px,
              transparent 8px 12px,
              var(--ui-border) 12px 13px,
              transparent 13px
            )
          `,
          boxShadow: `
            inset 0 1px 0 rgba(255,255,255,0.18),
            inset 0 -1px 0 rgba(0,0,0,0.35)
          `,
        }}
      />
    </div>
  );
}

// =========================
// Game constants
// =========================
const GAME = {
  W: 900,
  H: 500,
  INSET_DEFAULT: 30,
  PADDLE_W: 14,
  PADDLE_H_BASE: 110,
  PADDLE_SPEED: 7,
  BALL_SIZE: 14,
  BALL_SPEED_START: 6,
  BALL_SPEED_MAX: 12,
};

// =========================
// Color themes
// =========================
const COLOR_THEMES = {
  green: {
    key: "green",
    name: "Green",
    bg: "#06140B",
    fg: "#D9FFE5",
    accent: "#2BB673",
    sliderRight: "#9CFFB0",
  },
  amber: {
    key: "amber",
    name: "Amber",
    bg: "#140F05",
    fg: "#FFE6B8",
    accent: "#FFB000",
    sliderRight: "#FFD08A",
  },
  mono: {
    key: "mono",
    name: "B&W",
    bg: "#0A0A0A",
    fg: "#F2F2F2",
    accent: "#8E8E8E",
    sliderRight: "#CFCFCF",
  },
};
const getColorTheme = (key) => COLOR_THEMES[key] || COLOR_THEMES.green;

// =========================
// Mode rules
// =========================
const MODECFG = {
  pong: {
    key: "pong",
    label: "PONG",
    blurb: [
      "Klasyk: lewy kontra prawy.",
      "Punkt, gdy piłka minie przeciwnika.",
      "Pierwszy do limitu wygrywa.",
    ],
    audioTrack: "pong",
    rules: {
      paddles: { left: 1, right: 1 },
      rightPlayable: true,
      scoreToWin: true,
      squashWallInset: 0,
      hockey: null,
      leftInset: GAME.INSET_DEFAULT,
      rightInset: GAME.INSET_DEFAULT,
    },
  },
  squosh: {
    key: "squosh",
    label: "SQUASH",
    blurb: [
      "Solo: grasz tylko lewą paletką.",
      "Punkty za odbicia od prawej ściany.",
      "Nie ma prawego gracza.",
    ],
    audioTrack: "squosh",
    rules: {
      paddles: { left: 1, right: 0 },
      rightPlayable: false,
      scoreToWin: false,
      squashWallInset: 26,
      hockey: null,
      leftInset: GAME.INSET_DEFAULT,
      rightInset: GAME.INSET_DEFAULT,
    },
  },
  hockey: {
    key: "hockey",
    label: "HOKEY",
    blurb: [
      "HOKEY: 2 paletki na gracza.",
      "Poruszają się równocześnie.",
      "Gole przez środkowe okno.",
    ],
    audioTrack: "hockey_techno",
    rules: {
      paddles: { left: 2, right: 2 },
      rightPlayable: true,
      scoreToWin: true,
      squashWallInset: 0,
      hockey: { goalOpen: 150, forwardFrac: 0.75, bandInset: 0 },
      leftInset: GAME.INSET_DEFAULT,
      rightInset: GAME.INSET_DEFAULT,
    },
  },
};

const MODELIST = Object.values(MODECFG);
const getModeCfg = (key) => MODECFG[key] || MODECFG.pong;

// =========================
// Ball helpers
// =========================
const serveBall = (W, H, dir, speed) => {
  const d = dir == null ? (Math.random() < 0.5 ? -1 : 1) : dir;
  const angle = (Math.random() * 0.6 - 0.3) * Math.PI;
  const vx = Math.cos(angle) * speed * d;
  const vy = Math.sin(angle) * speed;
  return { x: W / 2, y: H / 2, vx, vy };
};

const reflectFromPaddle = (ball, paddleY, paddleH, speedMax) => {
  const rel = clamp((ball.y - paddleY) / paddleH, 0, 1);
  const n = rel * 2 - 1;
  const maxAng = 0.75;
  const ang = n * maxAng;
  const speed = clamp(Math.hypot(ball.vx, ball.vy) + 0.25, 5.5, speedMax);
  const dir = ball.vx < 0 ? 1 : -1;
  return {
    ...ball,
    vx: Math.cos(ang) * speed * dir,
    vy: Math.sin(ang) * speed,
  };
};

// HUD helper (tekst pomocy na dole)
const getHudHelpText = (modeKey, s) => {
  if (modeKey === "squosh") {
    return s.squoshTwoPlayers
      ? "Gracz 1: W/S   |   Gracz 2: ↑/↓ (AI OFF)   |   R: serw   |   Spacja: pauza"
      : "Lewy: W/S   |   R: serw   |   Spacja: pauza";
  }
  return "Lewy: W/S   |   Prawy: ↑/↓ (AI OFF)   |   R: serw   |   Spacja: pauza";
};

// =========================
// Audio
// =========================
function createAudioEngine() {
  try {
    const AudioCtx = window.AudioContext || window.webkitAudioContext;
    if (!AudioCtx) return null;

    const ctx = new AudioCtx();
    const master = ctx.createGain();
    master.gain.value = 0.55;

    const musicBus = ctx.createGain();
    const sfxBus = ctx.createGain();
    musicBus.gain.value = 1;
    sfxBus.gain.value = 1;
    musicBus.connect(master);
    sfxBus.connect(master);
    master.connect(ctx.destination);

    const semToHz = (base, sem) => base * Math.pow(2, sem / 12);

    const TRACKS = {
      pong: {
        bpm: 152,
        baseHz: 220,
        lead: [
          [0, null, 7, null, 12, null, 7, null, 10, null, 14, null, 12, null, 7, null],
          [0, 3, 7, 10, 12, 10, 7, 3, 2, null, 9, null, 7, null, 5, null],
          [12, null, 10, null, 7, null, 5, null, 7, null, 10, null, 12, null, 14, null],
          [0, null, 5, null, 7, null, 10, null, 12, 10, 7, 5, 3, null, 2, null],
        ],
        bass: [
          [0, null, null, null, 0, null, null, null, -5, null, null, null, -5, null, null, null],
          [0, null, null, null, -3, null, null, null, -5, null, null, null, -7, null, null, null],
          [-5, null, null, null, -5, null, null, null, -2, null, null, null, 0, null, null, null],
          [0, null, null, null, -7, null, null, null, -5, null, null, null, -3, null, null, null],
        ],
        pad: [
          [0, 0, 0, 0, -3, -3, -3, -3, -2, -2, -2, -2, -5, -5, -5, -5],
          [0, 0, 0, 0, -5, -5, -5, -5, -7, -7, -7, -7, -3, -3, -3, -3],
          [-5, -5, -5, -5, -2, -2, -2, -2, 0, 0, 0, 0, -3, -3, -3, -3],
          [0, 0, 0, 0, -7, -7, -7, -7, -5, -5, -5, -5, -3, -3, -3, -3],
        ],
        drums: { hat: "even", snare: "4-12", kick: "0-8" },
        mix: { lead: 1, bass: 1, pad: 1, drums: 1 },
      },

      squosh: {
        bpm: 176,
        baseHz: 330,
        lead: [
          [0, null, null, 7, null, null, 12, null, 7, null, null, 10, null, null, 7, null],
          [12, null, null, 10, null, null, 7, null, 10, null, null, 12, null, null, 14, null],
          [7, null, null, 5, null, null, 7, null, 10, null, null, 12, null, null, 10, null],
          [0, null, null, 7, null, null, 10, null, 12, null, null, 10, null, null, 7, null],
        ],
        bass: [
          [0, null, -5, null, -7, null, -5, null, 0, null, -5, null, -7, null, -5, null],
          [-5, null, -7, null, -5, null, 0, null, -3, null, -5, null, -7, null, -5, null],
          [-7, null, -5, null, -3, null, -5, null, -7, null, -5, null, 0, null, -5, null],
          [-5, null, 0, null, -5, null, -7, null, -5, null, -3, null, -5, null, -7, null],
        ],
        pad: [
          [0, 0, -7, -7, 0, 0, -7, -7, -5, -5, -7, -7, -5, -5, -7, -7],
          [-5, -5, -7, -7, -5, -5, -7, -7, 0, 0, -7, -7, 0, 0, -7, -7],
          [0, 0, -5, -5, 0, 0, -5, -5, -7, -7, -5, -5, -7, -7, -5, -5],
          [-7, -7, -5, -5, -7, -7, -5, -5, 0, 0, -5, -5, 0, 0, -5, -5],
        ],
        drums: { hat: "all", snare: "12", kick: "0-6-8-14" },
        mix: { lead: 0.85, bass: 0.95, pad: 0.45, drums: 1.0 },
      },

      hockey_techno: {
        bpm: 128,
        baseHz: 110,
        lead: [
          [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null],
          [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null],
          [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null],
          [null, null, null, null, null, null, null, null, null, null, 0, 2, 3, 2, 0, -2],
        ],
        bass: [
          [0, null, 0, null, 0, null, 0, null, 0, null, 0, null, 0, null, 0, null],
          [0, null, 0, null, -2, null, -2, null, 0, null, 0, null, -5, null, -5, null],
          [0, null, 0, null, 0, null, 0, null, -2, null, -2, null, 0, null, 0, null],
          [0, null, 0, null, 0, null, 0, null, 0, null, -2, -2, -5, -5, -7, -7],
        ],
        pad: [
          [0, 0, 0, 0, 0, 0, 0, 0, -2, -2, -2, -2, 0, 0, 0, 0],
          [-2, -2, -2, -2, 0, 0, 0, 0, -5, -5, -5, -5, -2, -2, -2, -2],
          [0, 0, 0, 0, -2, -2, -2, -2, 0, 0, 0, 0, -5, -5, -5, -5],
          [-2, -2, -2, -2, -5, -5, -5, -5, -6, -6, -6, -6, -5, -5, -2, -2],
        ],
        drums: { hat: "off", snare: "4-12", kick: "four", fill: true },
        mix: { lead: 0.25, bass: 1.2, pad: 0.85, drums: 1.15 },
      },
    };

    const NOISE_HAT = new Float32Array(256);
    const NOISE_SNARE = new Float32Array(512);
    for (let i = 0; i < NOISE_HAT.length; i++) NOISE_HAT[i] = (Math.random() * 2 - 1) * 0.7;
    for (let i = 0; i < NOISE_SNARE.length; i++) NOISE_SNARE[i] = (Math.random() * 2 - 1) * 0.9;

    const playNoise = (bufArr, gain, hp = 2000, ms = 60) => {
      const b = ctx.createBuffer(1, bufArr.length, ctx.sampleRate);
      b.copyToChannel(bufArr, 0);
      const src = ctx.createBufferSource();
      src.buffer = b;

      const g = ctx.createGain();
      const f = ctx.createBiquadFilter();
      f.type = "highpass";
      f.frequency.value = hp;

      src.connect(f);
      f.connect(g);
      g.connect(sfxBus);

      const t0 = ctx.currentTime;
      const dur = ms / 1000;
      g.gain.setValueAtTime(0.0001, t0);
      g.gain.exponentialRampToValueAtTime(Math.max(0.0001, gain), t0 + 0.004);
      g.gain.exponentialRampToValueAtTime(0.0001, t0 + Math.max(0.01, dur));

      src.start();
      src.stop(t0 + dur + 0.02);
    };

    const playTone = (freq, ms, type, gain) => {
      const o = ctx.createOscillator();
      o.type = type;
      o.frequency.value = freq;
      const g = ctx.createGain();
      g.gain.value = 0.0001;
      o.connect(g);
      g.connect(sfxBus);

      const t0 = ctx.currentTime;
      const dur = ms / 1000;
      g.gain.setValueAtTime(0.0001, t0);
      g.gain.exponentialRampToValueAtTime(Math.max(0.0001, gain), t0 + 0.008);
      g.gain.exponentialRampToValueAtTime(0.0001, t0 + Math.max(0.01, dur));

      o.start();
      o.stop(t0 + dur + 0.03);
    };

    const drumHat = (i, mode) => {
      if (mode === "all") return true;
      if (mode === "even") return i % 2 === 0;
      if (mode === "off") return i === 2 || i === 6 || i === 10 || i === 14;
      return i % 2 === 0;
    };

    const drumKick = (i, mode) => {
      if (mode === "0-6-8-14") return i === 0 || i === 6 || i === 8 || i === 14;
      if (mode === "0-8") return i === 0 || i === 8;
      if (mode === "four") return i === 0 || i === 4 || i === 8 || i === 12;
      return i === 0 || i === 8;
    };

    const drumSnare = (i, mode) => {
      if (mode === "12") return i === 12;
      if (mode === "4-12") return i === 4 || i === 12;
      if (mode === "none") return false;
      return i === 4 || i === 12;
    };

    const music = {
      enabled: true,
      playing: false,
      timer: null,
      step: 0,
      trackKey: "pong",
      bpm: 152,
      baseHz: 220,
      leadOsc: null,
      bassOsc: null,
      padOsc: null,
      leadGain: ctx.createGain(),
      bassGain: ctx.createGain(),
      padGain: ctx.createGain(),
      lvlLead: 0.05,
      lvlBass: 0.032,
      lvlPad: 0.012,
      lvlDrums: 1.0,
      pendingStart: null,
      startToken: 0,
    };

    music.leadGain.gain.value = 0;
    music.bassGain.gain.value = 0;
    music.padGain.gain.value = 0;
    music.leadGain.connect(musicBus);
    music.bassGain.connect(musicBus);
    music.padGain.connect(musicBus);

    const stopInternal = (fadeSec = 0.16) => {
      music.startToken++;
      if (music.pendingStart) {
        clearTimeout(music.pendingStart);
        music.pendingStart = null;
      }
      if (!music.playing) return;
      music.playing = false;
      if (music.timer) clearInterval(music.timer);
      music.timer = null;

      const t = ctx.currentTime;
      [music.leadGain, music.bassGain, music.padGain].forEach((gg) => {
        const current = Math.max(0.0001, gg.gain.value);
        gg.gain.cancelScheduledValues(t);
        gg.gain.setValueAtTime(current, t);
        gg.gain.exponentialRampToValueAtTime(0.0001, t + fadeSec);
      });

      const stopAt = t + fadeSec + 0.02;
      try {
        music.leadOsc?.stop(stopAt);
      } catch {}
      try {
        music.bassOsc?.stop(stopAt);
      } catch {}
      try {
        music.padOsc?.stop(stopAt);
      } catch {}
      music.leadOsc = music.bassOsc = music.padOsc = null;
    };

    const startInternal = (trackKey) => {
      if (!music.enabled || ctx.state !== "running") return;
      if (music.pendingStart) {
        clearTimeout(music.pendingStart);
        music.pendingStart = null;
      }

      let key = trackKey || music.trackKey;
      if (!TRACKS[key]) key = "pong";

      if (music.playing && music.trackKey === key) return;

      if (music.playing && music.trackKey !== key) {
        const myToken = ++music.startToken;
        stopInternal(0.09);
        music.pendingStart = setTimeout(() => {
          if (myToken !== music.startToken) return;
          music.pendingStart = null;
          startInternal(key);
        }, 110);
        return;
      }

      const tr = TRACKS[key];
      music.trackKey = key;
      music.bpm = tr.bpm;
      music.baseHz = tr.baseHz;
      music.step = 0;

      const mix = tr.mix || { lead: 1, bass: 1, pad: 1, drums: 1 };
      music.lvlLead = 0.05 * mix.lead;
      music.lvlBass = 0.032 * mix.bass;
      music.lvlPad = 0.012 * mix.pad;
      music.lvlDrums = 1.0 * mix.drums;

      music.playing = true;

      const lead = ctx.createOscillator();
      const bass = ctx.createOscillator();
      const pad = ctx.createOscillator();
      lead.type = "square";
      bass.type = "triangle";
      pad.type = "sawtooth";
      lead.connect(music.leadGain);
      bass.connect(music.bassGain);
      pad.connect(music.padGain);
      lead.start();
      bass.start();
      pad.start();

      music.leadOsc = lead;
      music.bassOsc = bass;
      music.padOsc = pad;

      const t = ctx.currentTime;
      music.leadGain.gain.setValueAtTime(0.0001, t);
      music.bassGain.gain.setValueAtTime(0.0001, t);
      music.padGain.gain.setValueAtTime(0.0001, t);
      music.leadGain.gain.exponentialRampToValueAtTime(Math.max(0.0001, music.lvlLead), t + 0.22);
      music.bassGain.gain.exponentialRampToValueAtTime(Math.max(0.0001, music.lvlBass), t + 0.22);
      music.padGain.gain.exponentialRampToValueAtTime(Math.max(0.0001, music.lvlPad), t + 0.35);

      const stepSec = 60 / music.bpm / 4;

      music.timer = setInterval(() => {
        if (!music.playing || !music.leadOsc || !music.bassOsc || !music.padOsc) return;

        const tr2 = TRACKS[music.trackKey] || TRACKS.pong;
        const pat = Math.floor(music.step / 16) % 4;
        const i = music.step % 16;

        const barStep = music.step % 64;
        const wantsFill = !!tr2.drums?.fill;
        const isFillWindow = wantsFill && barStep >= 48;
        const usePat = isFillWindow ? 3 : pat;

        const duckPad = (amt = 0.55, rel = 0.14) => {
          const t0 = ctx.currentTime;
          const base = Math.max(0.0001, music.lvlPad);
          const low = Math.max(0.0001, base * (1 - amt));
          music.padGain.gain.cancelScheduledValues(t0);
          music.padGain.gain.setValueAtTime(low, t0);
          music.padGain.gain.exponentialRampToValueAtTime(Math.max(0.0001, base), t0 + rel);
        };

        const padSem = tr2.pad[usePat][i];
        if (padSem != null)
          music.padOsc.frequency.setValueAtTime(semToHz(music.baseHz, padSem), ctx.currentTime);

        const leadSem = tr2.lead[usePat][i];
        if (leadSem == null) {
          music.leadGain.gain.setTargetAtTime(0.0001, ctx.currentTime, 0.01);
        } else {
          music.leadOsc.frequency.setValueAtTime(semToHz(music.baseHz, leadSem), ctx.currentTime);
          const t0 = ctx.currentTime;
          music.leadGain.gain.cancelScheduledValues(t0);
          music.leadGain.gain.setValueAtTime(0.0001, t0);
          music.leadGain.gain.exponentialRampToValueAtTime(Math.max(0.0001, music.lvlLead), t0 + 0.01);
          music.leadGain.gain.exponentialRampToValueAtTime(0.0001, t0 + stepSec * 0.92);
        }

        const bassSem = tr2.bass[usePat][i];
        if (bassSem == null) {
          music.bassGain.gain.setTargetAtTime(0.0001, ctx.currentTime, 0.02);
        } else {
          music.bassOsc.frequency.setValueAtTime(semToHz(music.baseHz, bassSem), ctx.currentTime);
          const tb = ctx.currentTime;
          music.bassGain.gain.cancelScheduledValues(tb);
          music.bassGain.gain.setValueAtTime(0.0001, tb);
          music.bassGain.gain.exponentialRampToValueAtTime(Math.max(0.0001, music.lvlBass), tb + 0.02);
          music.bassGain.gain.exponentialRampToValueAtTime(0.0001, tb + stepSec * 1.6);
        }

        const d = tr2.drums;
        if (drumHat(i, d.hat)) playNoise(NOISE_HAT, 0.012 * music.lvlDrums, 3200);
        if (drumSnare(i, d.snare)) playNoise(NOISE_SNARE, 0.018 * music.lvlDrums, 1400);

        const kickNow = drumKick(i, d.kick);
        if (kickNow) playTone(78, 70, "square", 0.034 * music.lvlDrums);

        if (isFillWindow && (i === 13 || i === 14 || i === 15))
          playNoise(NOISE_HAT, 0.016 * music.lvlDrums, 3600);

        if (kickNow || drumSnare(i, d.snare)) duckPad(0.45, 0.12);

        music.step++;
      }, stepSec * 1000);
    };

    const setEnabled = (on) => {
      music.enabled = !!on;
      if (!music.enabled) stopInternal(0.12);
    };

    const setTrack = (key) => {
      if (!key) return;
      music.trackKey = key;
      if (music.playing) startInternal(key);
    };

    const duckMusic = (amount = 0.35, rel = 0.18) => {
      const t0 = ctx.currentTime;
      const base = 1;
      const low = Math.max(0.05, base * (1 - amount));
      musicBus.gain.cancelScheduledValues(t0);
      musicBus.gain.setValueAtTime(low, t0);
      musicBus.gain.exponentialRampToValueAtTime(base, t0 + rel);
    };

    return {
      ctx,
      resume: () => (ctx.state !== "running" ? ctx.resume() : Promise.resolve()),
      sfx: {
        paddle: () => playTone(420, 55, "square", 0.07),
        wall: () => playTone(240, 45, "square", 0.05),
        score: () => {
          duckMusic(0.32, 0.2);
          playTone(180, 70, "square", 0.06);
          setTimeout(() => playTone(300, 70, "square", 0.05), 60);
        },
        ui: () => playTone(520, 35, "square", 0.04),

        gameOverSquosh: () => {
          duckMusic(0.78, 0.85);
          playTone(196, 120, "square", 0.065);
          setTimeout(() => playTone(262, 130, "square", 0.062), 140);
          setTimeout(() => playTone(330, 140, "square", 0.058), 300);
          setTimeout(() => playTone(392, 160, "square", 0.055), 470);
          setTimeout(() => playTone(330, 140, "square", 0.052), 660);
          setTimeout(() => playTone(262, 180, "square", 0.05), 820);
        },

        gameOverPong: () => {
          duckMusic(0.72, 0.7);
          playTone(262, 110, "square", 0.06);
          setTimeout(() => playTone(330, 120, "square", 0.058), 140);
          setTimeout(() => playTone(392, 130, "square", 0.056), 300);
          setTimeout(() => playTone(523, 160, "square", 0.05), 520);
        },

        gameOverHockey: () => {
          duckMusic(0.75, 0.8);
          playTone(196, 140, "square", 0.062);
          setTimeout(() => playTone(220, 140, "square", 0.06), 170);
          setTimeout(() => playTone(262, 150, "square", 0.058), 340);
          setTimeout(() => playTone(330, 180, "square", 0.052), 520);
          setTimeout(() => playTone(262, 200, "square", 0.05), 740);
        },
      },
      music: {
        start: (trackKey) => startInternal(trackKey),
        stop: () => stopInternal(0.16),
        setTrack,
        setEnabled,
        isPlaying: () => !!music.playing,
      },
    };
  } catch {
    return null;
  }
}

// =========================
// Lightweight self-tests
// =========================
function runSelfTests() {
  console.assert(Number.isFinite(GAME.W) && GAME.W > 0, "GAME.W should be > 0");
  console.assert(Number.isFinite(GAME.H) && GAME.H > 0, "GAME.H should be > 0");
  console.assert(Number.isFinite(GAME.PADDLE_W) && GAME.PADDLE_W > 0, "GAME.PADDLE_W should be > 0");
  console.assert(Number.isFinite(GAME.BALL_SIZE) && GAME.BALL_SIZE > 0, "GAME.BALL_SIZE should be > 0");

  console.assert(getColorTheme("green").key === "green", "Theme green should exist");
  console.assert(getColorTheme("amber").key === "amber", "Theme amber should exist");
  console.assert(getColorTheme("mono").key === "mono", "Theme mono should exist");
  console.assert(getColorTheme("nope").key === "green", "Unknown theme should fall back to green");

  console.assert(getModeCfg("pong").key === "pong", "Mode pong should exist");
  console.assert(getModeCfg("squosh").key === "squosh", "Mode squosh should exist");
  console.assert(getModeCfg("hockey").key === "hockey", "Mode hockey should exist");
  console.assert(getModeCfg("nope").key === "pong", "Unknown mode should fall back to pong");

  console.assert(clamp(5, 0, 10) === 5, "Clamp in range");
  console.assert(clamp(-1, 0, 10) === 0, "Clamp low");
  console.assert(clamp(99, 0, 10) === 10, "Clamp high");

  console.assert(aiSkillFromUiLevel(1) === 1, "AI lvl 1 should be hardest (skill=1)");
  console.assert(aiSkillFromUiLevel(10) === 0, "AI lvl 10 should be easiest (skill=0)");
}

// =========================
// MAIN COMPONENT (H)
// =========================
function Paletkowo() {
  // -------------------------
  // H1) State + refs
  // -------------------------
  const canvasRef = useRef(null);
  const ctxRef = useRef(null);
  const rafRef = useRef(0);
  const lastTRef = useRef(0);

  const keysRef = useRef(new Set());
  const audioRef = useRef(null);

  const [paused, setPaused] = useState(false);

  // Krok 2: pokrętła (wizualizacja ruchu paletek)
  const [knobL, setKnobL] = useState(0);
  const [knobR, setKnobR] = useState(0);
  const knobRef = useRef({ aL: 0, aR: 0, lastYL: null, lastYR: null, lastUiT: 0 });


  // Krok 2: podświetlenie przycisków (klik + klawisze)
  const [arcadeActive, setArcadeActive] = useState({ pause: false, start: false, serve: false });
  const pulseArcade = useCallback((key) => {
    setArcadeActive((s) => ({ ...s, [key]: true }));
    window.setTimeout(() => {
      setArcadeActive((s) => ({ ...s, [key]: false }));
    }, 120);
  }, []);
  const [started, setStarted] = useState(false);
  const [scoreToWin, setScoreToWin] = useState(10);
  const [speedScale, setSpeedScale] = useState(1);
  const [aiEnabled, setAiEnabled] = useState(true);
  const [squoshTwoPlayers, setSquoshTwoPlayers] = useState(false);
  const [aiLevel, setAiLevel] = useState(6);
  const [paddleScale, setPaddleScale] = useState(1);
  const [mode, setMode] = useState("pong");
  const [colorTheme, setColorTheme] = useState(() => {
    try {
      const v = localStorage.getItem("paletkowo_theme");
      if (!v) return "green";
      return getColorTheme(v).key;
    } catch {
      return "green";
    }
  });
  const [musicEnabled, setMusicEnabled] = useState(() => {
    try {
      const v = localStorage.getItem("paletkowo_music");
      if (v === null) return true;
      return v === "1";
    } catch {
      return true;
    }
  });

  const settingsRef = useRef(null);
  settingsRef.current = {
    paused,
    started,
    scoreToWin,
    speedScale,
    aiEnabled,
    aiLevel,
    squoshTwoPlayers,
    paddleScale,
    mode,
    musicEnabled,
    colorTheme,
  };

  const G = GAME;
  const W = G.W;
  const H = G.H;

  const uiVars = useMemo(() => {
    const th = getColorTheme(colorTheme);
    const blend = (c1, c2, t) => blendHex(c1, c2, t);

    const uiBg = blend(th.bg, "#ffffff", 0.06);
    const uiPanel = blend(th.bg, "#ffffff", 0.12);
    const uiBorder = blend(th.accent, th.fg, 0.62);
    const uiText = th.fg;
    const uiMuted = blend(th.fg, th.bg, 0.55);
    const uiSoft = blend(th.accent, th.bg, 0.82);

    const uiAccent = th.accent;
    const uiAccentBorder = blend(th.accent, th.fg, 0.35);
    const uiAccentText = th.key === "mono" ? blend(th.fg, th.bg, 0.15) : blend(th.bg, th.fg, 0.15);
    const uiKnob = th.key === "mono" ? blend(th.fg, th.bg, 0.35) : blend(th.fg, "#ffffff", 0.1);

    return {
      "--ui-bg": uiBg,
      "--ui-panel": uiPanel,
      "--ui-border": uiBorder,
      "--ui-text": uiText,
      "--ui-muted": uiMuted,
      "--ui-soft": uiSoft,
      "--ui-accent": uiAccent,
      "--ui-accentBorder": uiAccentBorder,
      "--ui-accentText": uiAccentText,
      "--ui-knob": uiKnob,
      "--ui-sliderRight": th.sliderRight || blend(th.fg, "#ffffff", 0.12),
      "--ui-font": '"Tiny5", "Press Start 2P", "VT323", monospace',
    };
  }, [colorTheme]);

  const themeRef = useRef(getColorTheme(colorTheme));
  useEffect(() => {
    themeRef.current = getColorTheme(colorTheme);
  }, [colorTheme]);

  const TITLE_UI = useMemo(
    () => ({
      fontFamily: '"Tiny5", "Press Start 2P", "VT323", monospace',
      titleSize: 90,
      modeSize: 54,
      blurbSize: 21,
      hintSize: 20,
      gapTitleToMode: 22,
      gapModeToBlurb: 18,
      gapBlurbToBtn: 26,
      gapBtnToHint: 26,
      btnW: 260,
      btnH: 56,
      btnRadius: 0,
      btnTextSize: 30,
    }),
    []
  );

  const HUD_UI = useMemo(
    () => ({
      fontFamily: TITLE_UI.fontFamily,
      scoreSize: 51,
      helpSize: 23,
      overlaySize: 32,
      gameOverSize: 34,
    }),
    [TITLE_UI]
  );

  const PADDLE_W = G.PADDLE_W;
  const PADDLE_H_BASE = G.PADDLE_H_BASE;
  const PADDLE_SPEED = G.PADDLE_SPEED;
  const BALL_SIZE = G.BALL_SIZE;
  const BALL_SPEED_START = G.BALL_SPEED_START;
  const BALL_SPEED_MAX = G.BALL_SPEED_MAX;

  const getPaddleH = () => Math.round(PADDLE_H_BASE * settingsRef.current.paddleScale);

  // -------------------------
  // H2) Audio glue (ensureAudio/syncMusic)
  // -------------------------
  const ensureAudio = useCallback(async () => {
    // WebAudio bywa "zablokowane" dopóki nie nastąpi gest użytkownika.
    // Dodatkowo (zwłaszcza na iOS) pomaga krótki, prawie niesłyszalny "tick" po resume.
    if (!audioRef.current) audioRef.current = createAudioEngine();
    const a = audioRef.current;
    if (!a?.ctx) return false;

    try {
      if (a.ctx.state !== "running") {
        await a.ctx.resume();
      }

      // "unlock tick" – bardzo krótki i bardzo cichy
      const o = a.ctx.createOscillator();
      const g = a.ctx.createGain();
      o.type = "sine";
      o.frequency.value = 60;
      g.gain.value = 0.00001;
      o.connect(g);
      g.connect(a.ctx.destination);
      const t0 = a.ctx.currentTime;
      o.start(t0);
      o.stop(t0 + 0.01);

      return true;
    } catch {
      return false;
    }
  }, []);

  const syncMusic = useCallback((modeOverride) => {
    const s = settingsRef.current;
    const a = audioRef.current;
    if (!a?.music) return;

    const desiredMode = modeOverride || s.mode || "pong";
    const track = getModeCfg(desiredMode).audioTrack;

    a.music.setEnabled(!!s.musicEnabled);
    if (!s.musicEnabled) return a.music.stop();

    a.music.start(track);
  }, []);

    // -------------------------
  // H3) Game state init/reset
  // -------------------------

  const aiLastUpdateRef = useRef(0);
  const aiAimRef = useRef(0);

  const gameRef = useRef({
    left: { x: 30, y: H / 2 - PADDLE_H_BASE / 2 },
    right: { x: W - 30 - PADDLE_W, y: H / 2 - PADDLE_H_BASE / 2 },
    left2: { x: 30, y: H / 2 - PADDLE_H_BASE / 2 + 140 },
    right2: { x: W - 30 - PADDLE_W, y: H / 2 - PADDLE_H_BASE / 2 + 140 },
    ball: serveBall(W, H, null, BALL_SPEED_START),
    leftScore: 0,
    rightScore: 0,
    gameOver: false,
    __lastPaddleH: PADDLE_H_BASE,
    fx: { goalFlashUntil: 0, goalFlashSide: 0 },
    lastTouch: 0,
  });

  const makeInitialState = (mKey) => {
    const cfg = getModeCfg(mKey);
    const paddleH = getPaddleH();

    const leftX = cfg.rules.leftInset;
    const rightX = W - cfg.rules.rightInset - PADDLE_W;
    const baseY = H / 2 - paddleH / 2;

    const hockey = cfg.rules.hockey;
    const f = hockey?.forwardFrac ?? 0.75;
    const midL = clamp(Math.floor(W * f - PADDLE_W / 2), 0, W - PADDLE_W);
    const midR = clamp(Math.floor(W * (1 - f) - PADDLE_W / 2), 0, W - PADDLE_W);

    return {
      left: { x: leftX, y: baseY },
      right: { x: rightX, y: baseY },
      left2: { x: hockey ? midL : leftX, y: baseY },
      right2: { x: hockey ? midR : rightX, y: baseY },
      ball: serveBall(W, H, null, BALL_SPEED_START),
      leftScore: 0,
      rightScore: 0,
      gameOver: false,
      __lastPaddleH: paddleH,
      fx: { goalFlashUntil: 0, goalFlashSide: 0 },
      lastTouch: 0,
    };
  };

  const resetGame = useCallback(() => {
    const m = settingsRef.current.mode;
    gameRef.current = makeInitialState(m);
    lastTRef.current = 0;
    aiLastUpdateRef.current = 0;
    aiAimRef.current = 0;
  }, []);

  const newRound = (dir) => (gameRef.current.ball = serveBall(W, H, dir, BALL_SPEED_START));

  const isInHockeyGoalMouth = (y, goalOpen) => {
    const top = H / 2 - goalOpen / 2;
    const bot = H / 2 + goalOpen / 2;
    return y >= top && y <= bot;
  };

  const flashGoal = (side) => {
    const g = gameRef.current;
    const now = performance.now();
    g.fx.goalFlashSide = side;
    g.fx.goalFlashUntil = now + 220;
  };

  useEffect(() => {
    runSelfTests();

    // Awaryjny "unlock" audio: jeśli środowisko wymaga bezpośredniego gestu,
    // pierwszy pointerdown uruchomi resume (raz).
    const onFirstGesture = () => {
      ensureAudio().then(() => {
        // Muzyka ma działać także na planszy startowej (po pierwszym geście użytkownika).
        if (settingsRef.current.musicEnabled) syncMusic();
      });
    };
    window.addEventListener("pointerdown", onFirstGesture, { once: true });
    return () => window.removeEventListener("pointerdown", onFirstGesture);
  }, []);

  useEffect(() => {
    const g = gameRef.current;
    const newH = Math.round(PADDLE_H_BASE * paddleScale);
    const lastH = g.__lastPaddleH ?? newH;

    const recenter = (p) => (p.y = clamp(p.y + lastH / 2 - newH / 2, 0, H - newH));
    recenter(g.left);
    recenter(g.right);
    recenter(g.left2);
    recenter(g.right2);
    g.__lastPaddleH = newH;
  }, [paddleScale]);

  useEffect(() => {
    // Zmiana trybu nie powinna ucinać muzyki.
    // Jeśli gra była już uruchomiona, zostajemy w stanie STARTED i tylko resetujemy stan gry.
    const wasStarted = !!settingsRef.current.started;

    resetGame();
    setPaused(false);
    setStarted(wasStarted);

    ensureAudio().then(() => syncMusic(mode));
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [mode]);

  useEffect(() => {
    // Persist ustawień UI
    try {
      localStorage.setItem("paletkowo_music", musicEnabled ? "1" : "0");
      localStorage.setItem("paletkowo_theme", colorTheme);
    } catch {}
  }, [musicEnabled, colorTheme]);

  useEffect(() => {
    // Muzyka ma działać i w grze, i na planszy startowej.
    // AudioContext wystartuje dopiero po geście użytkownika — ensureAudio() może wtedy zadziałać.
    if (!musicEnabled) {
      try {
        audioRef.current?.music?.setEnabled?.(false);
        audioRef.current?.music?.stop?.();
      } catch {}
      return;
    }

    ensureAudio().then(() => {
      syncMusic();
    });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [musicEnabled, mode]);

  useEffect(() => {
    const onKeyDown = (e) => {
      const isGameOver = gameRef.current.gameOver;

      const isSpace = e.key === " " || e.code === "Space";
      if (isGameOver && !isSpace) return;

      keysRef.current.add(e.key);
      if (isSpace) {
        e.preventDefault();
        ensureAudio().then(() => {
          syncMusic();

          if (gameRef.current.gameOver) {
            // KROK 1: po game-over SPACJA wraca do planszy startowej (nie restartuje meczu)
            gameRef.current = makeInitialState(settingsRef.current.mode);
            gameRef.current.lastTouch = 0;
            lastTRef.current = 0;
            aiLastUpdateRef.current = 0;
            aiAimRef.current = 0;

            setPaused(false);
            setStarted(false);

            audioRef.current?.sfx?.ui?.();
            return;
          }

          if (!settingsRef.current.started) {
            pulseArcade("start");
            setStarted(true);
            setPaused(false);
            audioRef.current?.sfx?.ui?.();
            return;
          }

          // pauza / wznowienie
          pulseArcade("pause");
          setPaused((p) => !p);
          audioRef.current?.sfx?.ui?.();
        });
      }

      if (!isGameOver && String(e.key).toLowerCase() === "r") {
        pulseArcade("serve");
        ensureAudio().then(() => {
          gameRef.current.ball = serveBall(W, H, null, BALL_SPEED_START);
          audioRef.current?.sfx?.ui?.();
        });
      }
    };

    const onKeyUp = (e) => keysRef.current.delete(e.key);

    window.addEventListener("keydown", onKeyDown, { passive: false });
    window.addEventListener("keyup", onKeyUp);
    return () => {
      window.removeEventListener("keydown", onKeyDown);
      window.removeEventListener("keyup", onKeyUp);
    };
  }, []);

    // -------------------------
  // H4) Input + AI
  // -------------------------

  const stepAI = (paddleH) => {
    const g = gameRef.current;
    const s = settingsRef.current;

    const skill = aiSkillFromUiLevel(s.aiLevel);

    const now = performance.now();
    const reactionMs = 220 - skill * 190;

    if (now - aiLastUpdateRef.current > reactionMs) {
      aiLastUpdateRef.current = now;

      const lookAhead = 10 + skill * 55;
      let aimY = g.ball.y + g.ball.vy * lookAhead;

      const r = BALL_SIZE / 2;
      while (aimY < r || aimY > H - r) {
        aimY = aimY < r ? r + (r - aimY) : H - r - (aimY - (H - r));
      }

      aimY += (Math.random() * 2 - 1) * (1 - skill) * 24;
      aiAimRef.current = clamp(aimY - paddleH / 2, 0, H - paddleH);
    }

    const maxSpeed = PADDLE_SPEED * s.speedScale * (0.65 + 0.55 * skill);
    const diff = aiAimRef.current - g.right.y;
    const dead = 2 + (1 - skill) * 6;
    if (Math.abs(diff) <= dead) return;

    g.right.y = clamp(g.right.y + clamp(diff, -maxSpeed, maxSpeed), 0, H - paddleH);
    if (s.mode === "hockey") g.right2.y = g.right.y;
  };

  const stepAISquoshMate = (paddleH) => {
    const g = gameRef.current;
    const s = settingsRef.current;

    const skill = aiSkillFromUiLevel(s.aiLevel);

    const now = performance.now();
    const reactionMs = 220 - skill * 190;

    if (now - aiLastUpdateRef.current > reactionMs) {
      aiLastUpdateRef.current = now;

      const lookAhead = 10 + skill * 55;
      let aimY = g.ball.y + g.ball.vy * lookAhead;

      const r = BALL_SIZE / 2;
      while (aimY < r || aimY > H - r) {
        aimY = aimY < r ? r + (r - aimY) : H - r - (aimY - (H - r));
      }

      aimY += (Math.random() * 2 - 1) * (1 - skill) * 24;
      aiAimRef.current = clamp(aimY - paddleH / 2, 0, H - paddleH);
    }

    const maxSpeed = PADDLE_SPEED * s.speedScale * (0.65 + 0.55 * skill);
    const diff = aiAimRef.current - g.left2.y;
    const dead = 2 + (1 - skill) * 6;
    if (Math.abs(diff) <= dead) return;

    g.left2.y = clamp(g.left2.y + clamp(diff, -maxSpeed, maxSpeed), 0, H - paddleH);
  };

  const applyInput = (cfg) => {
    if (gameRef.current.gameOver) return;
    const s = settingsRef.current;
    const g = gameRef.current;
    const paddleH = getPaddleH();
    const keys = keysRef.current;

    const paddleSpeed = PADDLE_SPEED * s.speedScale;

    if (keys.has("w") || keys.has("W")) g.left.y -= paddleSpeed;
    if (keys.has("s") || keys.has("S")) g.left.y += paddleSpeed;
    g.left.y = clamp(g.left.y, 0, H - paddleH);

    if (cfg.key === "hockey" && cfg.rules.paddles.left === 2) g.left2.y = g.left.y;

    if (cfg.key === "squosh" && s.squoshTwoPlayers) {
      g.left2.x = g.left.x + PADDLE_W + 10;
      if (!g.lastTouch) g.lastTouch = 1;

      if (s.aiEnabled) {
        stepAISquoshMate(paddleH);
      } else {
        if (keys.has("ArrowUp")) g.left2.y -= paddleSpeed;
        if (keys.has("ArrowDown")) g.left2.y += paddleSpeed;
        g.left2.y = clamp(g.left2.y, 0, H - paddleH);
      }
    }

    if (cfg.rules.rightPlayable) {
      if (s.aiEnabled) stepAI(paddleH);
      else {
        if (keys.has("ArrowUp")) g.right.y -= paddleSpeed;
        if (keys.has("ArrowDown")) g.right.y += paddleSpeed;
        g.right.y = clamp(g.right.y, 0, H - paddleH);
        if (cfg.rules.paddles.right === 2) g.right2.y = g.right.y;
      }
    }

    // -- Aktualizacja kątów pokręteł na podstawie położenia paletek (potencjometr zakresowy) --
    // Dla lewego pokrętła: mapuj aktualne Y paletki na przedział 0–360°.
    {
      const range = H - paddleH;
      if (range > 0) {
        const fracL = g.left.y / range;
        const angleL = fracL * 360;
        knobRef.current.aL = Math.max(0, Math.min(360, angleL));
        setKnobL(knobRef.current.aL);
      }
    }
    // Dla prawego pokrętła: analogicznie mapuj pozycję odpowiedniej paletki.
    // W trybie "squosh" z dwoma graczami prawa paletka to w istocie lewa2; inaczej używamy g.right.
    {
      const range = H - paddleH;
      if (range > 0) {
        // Wybierz źródło Y dla prawego pokrętła w zależności od trybu.
        let sourceY;
        if (cfg.key === 'squosh' && s.squoshTwoPlayers) {
          sourceY = g.left2?.y;
        } else {
          sourceY = g.right?.y;
        }
        if (typeof sourceY === 'number') {
          const fracR = sourceY / range;
          const angleR = fracR * 360;
          knobRef.current.aR = Math.max(0, Math.min(360, angleR));
          setKnobR(knobRef.current.aR);
        }
      }
    }
  };

    // -------------------------
  // H5) Physics (ball/paddles/scoring/game over)
  // -------------------------

  const advanceBall = (dt) => {
    const s = settingsRef.current;
    const g = gameRef.current;

    const frameScale = s.speedScale * (dt / (1000 / 60));
    g.ball.x += g.ball.vx * frameScale;
    g.ball.y += g.ball.vy * frameScale;

    const r = BALL_SIZE / 2;

    if (g.ball.y - r <= 0) {
      g.ball.y = r;
      g.ball.vy *= -1;
      audioRef.current?.sfx?.wall?.();
    } else if (g.ball.y + r >= H) {
      g.ball.y = H - r;
      g.ball.vy *= -1;
      audioRef.current?.sfx?.wall?.();
    }
  };

  const resolveScoringAndWalls = (cfg) => {
    const g = gameRef.current;
    const r = BALL_SIZE / 2;

    if (cfg.key === "pong") {
      if (g.ball.x + r < 0) {
        g.rightScore++;
        flashGoal(-1);
        newRound(-1);
        audioRef.current?.sfx?.score?.();
      } else if (g.ball.x - r > W) {
        g.leftScore++;
        flashGoal(1);
        newRound(1);
        audioRef.current?.sfx?.score?.();
      }
    }

    if (cfg.key === "squosh") {
      const wallX = W - cfg.rules.squashWallInset;

      if (g.ball.x + r >= wallX) {
        g.ball.x = wallX - r;
        g.ball.vx *= -1;

        if (settingsRef.current.squoshTwoPlayers) {
          const lt = g.lastTouch || 1;
          if (lt === 1) g.leftScore++;
          else if (lt === 2) g.rightScore++;
        } else {
          g.leftScore++;
        }

        audioRef.current?.sfx?.wall?.();
      }

      if (g.ball.x - r <= 0) {
        if (settingsRef.current.squoshTwoPlayers) {
          const lt = g.lastTouch;
          if (lt === 1) g.rightScore++;
          else if (lt === 2) g.leftScore++;
          g.lastTouch = 0;
          g.ball = serveBall(W, H, 1, BALL_SPEED_START);
        } else {
          g.leftScore = 0;
          g.ball = serveBall(W, H, 1, BALL_SPEED_START);
        }
        audioRef.current?.sfx?.score?.();
      }

      if (!settingsRef.current.squoshTwoPlayers) g.rightScore = 0;
    }

    if (cfg.key === "hockey") {
      const goalOpen = cfg.rules.hockey.goalOpen;
      const hitLeft = g.ball.x - r <= 0;
      const hitRight = g.ball.x + r >= W;

      if (hitLeft) {
        if (isInHockeyGoalMouth(g.ball.y, goalOpen)) {
          g.rightScore++;
          flashGoal(-1);
          newRound(1);
          audioRef.current?.sfx?.score?.();
        } else {
          g.ball.x = r;
          g.ball.vx *= -1;
          audioRef.current?.sfx?.wall?.();
        }
      }

      if (hitRight) {
        if (isInHockeyGoalMouth(g.ball.y, goalOpen)) {
          g.leftScore++;
          flashGoal(1);
          newRound(-1);
          audioRef.current?.sfx?.score?.();
        } else {
          g.ball.x = W - r;
          g.ball.vx *= -1;
          audioRef.current?.sfx?.wall?.();
        }
      }
    }
  };

  const resolvePaddles = (cfg) => {
    const g = gameRef.current;
    const paddleH = getPaddleH();
    const r = BALL_SIZE / 2;

    const ballX = g.ball.x - r;
    const ballY = g.ball.y - r;

    const hitPaddle = (p, dir) => {
      if (dir < 0) g.ball.x = p.x + PADDLE_W + r;
      else g.ball.x = p.x - r;
      g.ball = reflectFromPaddle(g.ball, p.y, paddleH, BALL_SPEED_MAX);

      if (cfg.key === "squosh" && settingsRef.current.squoshTwoPlayers) {
        if (p === g.left) g.lastTouch = 1;
        else if (p === g.left2) g.lastTouch = 2;
      }

      audioRef.current?.sfx?.paddle?.();
    };

    if (g.ball.vx < 0) {
      if (intersects(ballX, ballY, BALL_SIZE, BALL_SIZE, g.left.x, g.left.y, PADDLE_W, paddleH))
        hitPaddle(g.left, -1);
      else if (
        (cfg.rules.paddles.left === 2 || (cfg.key === "squosh" && settingsRef.current.squoshTwoPlayers)) &&
        intersects(ballX, ballY, BALL_SIZE, BALL_SIZE, g.left2.x, g.left2.y, PADDLE_W, paddleH)
      )
        hitPaddle(g.left2, -1);
    } else if (cfg.rules.rightPlayable) {
      if (intersects(ballX, ballY, BALL_SIZE, BALL_SIZE, g.right.x, g.right.y, PADDLE_W, paddleH))
        hitPaddle(g.right, 1);
      else if (
        cfg.rules.paddles.right === 2 &&
        intersects(ballX, ballY, BALL_SIZE, BALL_SIZE, g.right2.x, g.right2.y, PADDLE_W, paddleH)
      )
        hitPaddle(g.right2, 1);
    }
  };

  const resolveGameOver = (cfg) => {
    const s = settingsRef.current;
    const g = gameRef.current;
    if (g.gameOver) return;

    if (cfg.rules.scoreToWin && s.scoreToWin && (g.leftScore >= s.scoreToWin || g.rightScore >= s.scoreToWin)) {
      g.gameOver = true;
      // KROK 5: wyraźne zakończenie rundy – wygaszenie muzyki + jingle
      audioRef.current?.music?.stop?.();
      if (cfg.key === "pong") audioRef.current?.sfx?.gameOverPong?.();
      else if (cfg.key === "hockey") audioRef.current?.sfx?.gameOverHockey?.();
      return;
    }

    if (cfg.key === "squosh" && s.scoreToWin) {
      const target = s.scoreToWin;
      const reached = s.squoshTwoPlayers
        ? g.leftScore >= target || g.rightScore >= target
        : g.leftScore >= target;

      if (reached) {
        g.gameOver = true;
        // KROK 5: wyraźne zakończenie rundy – wygaszenie muzyki + jingle
        audioRef.current?.music?.stop?.();
        audioRef.current?.sfx?.gameOverSquosh?.();
        return;
      }
    }
  };

  const step = (dt) => {
    const s = settingsRef.current;
    if (!s.started) return;
    const g = gameRef.current;
    if (g.gameOver) return;

    const cfg = getModeCfg(s.mode);
    applyInput(cfg);
    advanceBall(dt);
    resolveScoringAndWalls(cfg);
    resolvePaddles(cfg);
    resolveGameOver(cfg);
  };

    // -------------------------
  // H6) Render (canvas)
  // -------------------------

  const drawField = (ctx, mKey) => {
    const th = themeRef.current;
    const cfg = getModeCfg(mKey);
    ctx.fillStyle = th.bg;
    ctx.fillRect(0, 0, W, H);

    if (cfg.key !== "squosh") {
      ctx.fillStyle = th.accent;
      for (let y = 0; y < H; y += 28) ctx.fillRect(W / 2 - 2, y, 4, 16);
    }

    if (cfg.key === "squosh") {
      const wallX = W - cfg.rules.squashWallInset;
      ctx.fillRect(wallX - 3, 0, 3, H);
    }

    if (cfg.key === "hockey") {
      const top = H / 2 - cfg.rules.hockey.goalOpen / 2;
      const bot = H / 2 + cfg.rules.hockey.goalOpen / 2;
      ctx.strokeStyle = th.accent;
      ctx.lineWidth = 4;
      ctx.beginPath();
      ctx.moveTo(0, top);
      ctx.lineTo(22, top);
      ctx.moveTo(0, bot);
      ctx.lineTo(22, bot);
      ctx.moveTo(W, top);
      ctx.lineTo(W - 22, top);
      ctx.moveTo(W, bot);
      ctx.lineTo(W - 22, bot);
      ctx.stroke();
    }
  };

  const getTitleLayout = (m) => {
    const ui = TITLE_UI;
    const hTitle = ui.titleSize + 10;
    const hMode = ui.modeSize + 10;
    const hBlurb = ui.blurbSize * 2 + 14;
    const hBtn = ui.btnH;
    const hHint = ui.hintSize + 6;

    const total =
      hTitle +
      ui.gapTitleToMode +
      hMode +
      ui.gapModeToBlurb +
      hBlurb +
      ui.gapBlurbToBtn +
      hBtn +
      ui.gapBtnToHint +
      hHint;

    let y = Math.round(H / 2 - total / 2);

    const titleY = y + hTitle - 10;
    y += hTitle + ui.gapTitleToMode;

    const modeY = y + hMode - 10;
    y += hMode + ui.gapModeToBlurb;

    const blurbY1 = y + ui.blurbSize;
    const blurbY2 = y + ui.blurbSize * 2 + 8;
    y += hBlurb + ui.gapBlurbToBtn;

    const btnX = W / 2 - ui.btnW / 2;
    const btnY = y;
    y += hBtn + ui.gapBtnToHint;

    const hintY = y + ui.hintSize;

    return { titleY, modeY, blurbY1, blurbY2, btnX, btnY, btnW: ui.btnW, btnH: ui.btnH, hintY };
  };

  const drawTitleOnly = (ctx, m) => {
    const th = themeRef.current;
    ctx.fillStyle = th.bg;
    ctx.fillRect(0, 0, W, H);
    ctx.textAlign = "center";

    const ui = TITLE_UI;
    const pix = ui.fontFamily;
    const L = getTitleLayout(m);

    ctx.font = `900 ${ui.titleSize}px ${pix}`;
    ctx.fillStyle = th.accent;
    ctx.fillText("PALETKOWO", W / 2 + 3, L.titleY + 3);
    ctx.fillStyle = th.fg;
    ctx.fillText("PALETKOWO", W / 2, L.titleY);

    ctx.fillStyle = th.accent;
    ctx.font = `700 ${ui.modeSize}px ${pix}`;
    ctx.fillText(getModeCfg(m).label || String(m).toUpperCase(), W / 2, L.modeY);

    const bl = getModeCfg(m).blurb || ["", ""];
    ctx.fillStyle = th.fg;
    ctx.font = `${ui.blurbSize}px ${pix}`;
    ctx.fillText(bl[0], W / 2, L.blurbY1);
    ctx.fillText(bl[1], W / 2, L.blurbY2);

    ctx.fillStyle = th.accent;
    ctx.roundRect(L.btnX, L.btnY, L.btnW, L.btnH, ui.btnRadius);
    ctx.fill();

    ctx.fillStyle = th.bg;
    ctx.font = `bold ${ui.btnTextSize}px ${pix}`;
    ctx.fillText("START", W / 2, L.btnY + 38);

    ctx.fillStyle = th.fg;
    ctx.font = `bold ${ui.hintSize}px ${pix}`;
    ctx.fillText("Kliknij START lub naciśnij SPACJĘ", W / 2, L.hintY);
  };

    const drawOverlayPause = (ctx, th) => {
    ctx.fillStyle = "rgba(0,0,0,0.35)";
    ctx.fillRect(0, 0, W, H);
    ctx.fillStyle = th.fg;
    ctx.textAlign = "center";
    ctx.font = `${HUD_UI.overlaySize}px ${HUD_UI.fontFamily}`;
    ctx.fillText("Pauza", W / 2, H / 2);
  };

  const drawOverlayGameOver = (ctx, th, m, s, g) => {
    ctx.fillStyle = "rgba(0,0,0,0.5)";
    ctx.fillRect(0, 0, W, H);

    ctx.fillStyle = th.fg;
    ctx.textAlign = "center";

    ctx.font = `${HUD_UI.gameOverSize + 22}px ${HUD_UI.fontFamily}`;
    ctx.fillText("GAME OVER", W / 2, H / 2 - 24);

    ctx.font = `${HUD_UI.gameOverSize + 18}px ${HUD_UI.fontFamily}`;
    const scoreLine =
      m === "squosh"
        ? s.squoshTwoPlayers
          ? `${g.leftScore}:${g.rightScore}`
          : String(g.leftScore)
        : `${g.leftScore}:${g.rightScore}`;
    ctx.fillText(scoreLine, W / 2, H / 2 + 28);

    ctx.fillStyle = th.accent;
    ctx.font = `${HUD_UI.helpSize}px ${HUD_UI.fontFamily}`;
    ctx.fillText("SPACJA → MENU", W / 2, H / 2 + 70);
  };

  const draw = () => {
    const ctx = ctxRef.current;
    if (!ctx) return;

    const s = settingsRef.current;
    const g = gameRef.current;
    const m = s.mode;

    ctx.clearRect(0, 0, W, H);

    if (!s.started) return drawTitleOnly(ctx, m);

    drawField(ctx, m);

    const paddleH = getPaddleH();
    const r = BALL_SIZE / 2;

    const th = themeRef.current;
    const p1Col = th.fg;
    const p2Col = blendHex(th.accent, th.fg, 0.35);

    const rr = (x, y) => ctx.fillRect(x, y, PADDLE_W, paddleH);

    ctx.fillStyle = p1Col;
    rr(g.left.x, g.left.y);

    if (m === "hockey") rr(g.left2.x, g.left2.y);

    if (m === "squosh" && s.squoshTwoPlayers) {
      ctx.fillStyle = p2Col;
      rr(g.left2.x, g.left2.y);
      ctx.fillStyle = p1Col;
    }

    if (m !== "squosh") {
      ctx.fillStyle = p2Col;
      rr(g.right.x, g.right.y);
      if (m === "hockey") rr(g.right2.x, g.right2.y);
      ctx.fillStyle = p1Col;
    }

    ctx.fillRect(g.ball.x - r, g.ball.y - r, BALL_SIZE, BALL_SIZE);

    ctx.font = `bold ${HUD_UI.scoreSize}px ${HUD_UI.fontFamily}`;
    ctx.textAlign = "center";

    if (m === "squosh") {
      if (s.squoshTwoPlayers) {
        ctx.fillStyle = p1Col;
        ctx.fillText(String(g.leftScore), W / 2 - 70, 56);
        ctx.fillStyle = p2Col;
        ctx.fillText(String(g.rightScore), W / 2 + 70, 56);
      } else {
        ctx.fillStyle = p1Col;
        ctx.fillText(String(g.leftScore), W / 2, 56);
      }
    } else {
      ctx.fillStyle = p1Col;
      ctx.fillText(String(g.leftScore), W / 2 - 70, 56);
      ctx.fillStyle = p2Col;
      ctx.fillText(String(g.rightScore), W / 2 + 70, 56);
    }

    ctx.fillStyle = th.accent;
    ctx.font = `${HUD_UI.helpSize}px ${HUD_UI.fontFamily}`;
    const help = getHudHelpText(m, s);
    ctx.fillText(help, W / 2, H - 12);
    if (g.gameOver) {
      drawOverlayGameOver(ctx, th, m, s, g);
    } else if (s.paused) {
      drawOverlayPause(ctx, th);
    }
  };

  useEffect(() => {
    const canvas = canvasRef.current;
    if (!canvas) return;

    const ctx = canvas.getContext("2d");
    if (!ctx) return;

    ensureRoundRect(ctx);
    ctxRef.current = ctx;

    const loop = (t) => {
      if (!lastTRef.current) lastTRef.current = t;
      const dt = t - lastTRef.current;
      lastTRef.current = t;

      if (!settingsRef.current.paused) step(dt);
      draw();

      // Krok 2: aktualizacja pokręteł na podstawie ruchu paletek (również gdy AI steruje prawą)
      // - PONG/HOKEY: prawa paletka = g.right
      // - SQUASH 2P: "prawy" gracz = g.left2
      // - SQUASH solo: prawe pokrętło stoi
      const g = gameRef.current;
      const k = knobRef.current;

      const yL = g.left.y;
      const yR =
        settingsRef.current.mode === "squosh" && settingsRef.current.squoshTwoPlayers
          ? g.left2.y
          : settingsRef.current.mode === "squosh"
            ? 0
            : g.right.y;

      if (k.lastYL == null) {
        k.lastYL = yL;
        k.lastYR = yR;
      }

      const dyL = yL - (k.lastYL ?? yL);
      const dyR = yR - (k.lastYR ?? yR);
      k.lastYL = yL;
      k.lastYR = yR;

      k.aL = (k.aL + dyL * 1.4) % 360;
      k.aR = (k.aR + dyR * 1.4) % 360;

      if (t - (k.lastUiT || 0) > 50) {
        k.lastUiT = t;
        setKnobL(k.aL);
        setKnobR(k.aR);
      }

      rafRef.current = requestAnimationFrame(loop);
    };

    rafRef.current = requestAnimationFrame(loop);
    return () => {
      cancelAnimationFrame(rafRef.current);
      ctxRef.current = null;
      try {
        audioRef.current?.music?.stop?.();
      } catch {}
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const onCanvasPointerDown = useCallback(
    (e) => {
      const s = settingsRef.current;

      if (!s.started) {
        const canvas = canvasRef.current;
        if (!canvas) return;
        const rect = canvas.getBoundingClientRect();
        const scaleX = W / rect.width;
        const scaleY = H / rect.height;
        const x = (e.clientX - rect.left) * scaleX;
        const y = (e.clientY - rect.top) * scaleY;

        const L = getTitleLayout(s.mode);
        if (x >= L.btnX && x <= L.btnX + L.btnW && y >= L.btnY && y <= L.btnY + L.btnH) {
          ensureAudio().then(() => {
            setStarted(true);
            setPaused(false);
            audioRef.current?.sfx?.ui?.();
            syncMusic();
          });
          return;
        }
      }

      ensureAudio().then(() => syncMusic());
    },
    [W, ensureAudio, syncMusic]
  );

  const onTogglePause = useCallback(() => {
    pulseArcade("pause");
    ensureAudio().then(() => {
      syncMusic();
      if (!settingsRef.current.started) setStarted(true);
      setPaused((p) => !p);
      audioRef.current?.sfx?.ui?.();
    });
  }, [ensureAudio, syncMusic, pulseArcade]);

  const onStart = useCallback(() => {
    pulseArcade("start");
    ensureAudio().then(() => {
      syncMusic();
      if (!settingsRef.current.started) {
        setStarted(true);
        setPaused(false);
        audioRef.current?.sfx?.ui?.();
        return;
      }
      // START = wznów (nie pauzuje)
      setPaused(false);
      audioRef.current?.sfx?.ui?.();
    });
  }, [ensureAudio, syncMusic, pulseArcade]);

  const onServe = useCallback(() => {
    pulseArcade("serve");
    ensureAudio().then(() => {
      syncMusic();
      const g = gameRef.current;
      if (g.gameOver) return;
      if (!settingsRef.current.started) {
        setStarted(true);
        setPaused(false);
      }
      g.ball = serveBall(W, H, null, BALL_SPEED_START);
      audioRef.current?.sfx?.ui?.();
    });
  }, [ensureAudio, syncMusic, W, H, BALL_SPEED_START, pulseArcade]);

  const onReset = useCallback(() => {
    ensureAudio().then(() => {
      resetGame();
      setStarted(false);
      setPaused(false);
      audioRef.current?.sfx?.ui?.();
      syncMusic();
    });
  }, [ensureAudio, resetGame, syncMusic]);

    // -------------------------
  // H7) UI (React layout)
  // -------------------------

  return (
    <div className="min-h-[100vh]" style={{ background: "var(--ui-bg)", ...uiVars }}>
      <style>{`@import url('https://fonts.googleapis.com/css2?family=Tiny5&display=swap');`}</style>

      <div className="p-4 max-w-6xl mx-auto">
        <div className="grid gap-4 md:grid-cols-[1fr_360px]">
          <Card className="h-full">
            <CardContent className="p-3 flex flex-col h-full">

              {/* ROZDZIELENIE: panel gry + separator + panel arcade */}

              {/* Panel gry */}
              <div style={{ display: "flex", flexDirection: "column", flex: 1, minHeight: 0 }}>
                <div
                  onPointerDown={onCanvasPointerDown}
                  className="select-none border rounded-none overflow-hidden"
                  style={CANVAS_FRAME_STYLE}
                >
                  <canvas
                    ref={canvasRef}
                    width={W}
                    height={H}
                    className="block"
                    style={{ width: "100%", height: "auto" }}
                  />
                </div>
              </div>

              {/* Separator jak między panelami (prawy/lewy) */}
              <div style={{ marginTop: 21, marginBottom: 12 }}>
                <Divider style={{ margin: "0 0 10px 0" }} />
                <MechanicalRibs className="" h={44} ribH={16} sideInset={14} />
                <Divider style={{ margin: "10px 0 0 0" }} />
              </div>

              {/* KROK 3: panel arcade – dopracowanie układu */}
              <div
                className="mt-3 border rounded-none overflow-hidden"
                style={{
                  borderColor: "var(--ui-border)",
                  background: "var(--ui-panel)",
                  height: 190,
                  position: "relative",
                  display: "flex",
                  flexDirection: "column",
                  justifyContent: "space-between",
                  padding: "12px 14px",
                  gap: 10,
                }}
              >
                {/* dekoracyjne śrubki w narożnikach */}
                <div
                  style={{
                    position: "absolute",
                    top: 4,
                    left: 4,
                    width: 12,
                    height: 12,
                    borderRadius: "50%",
                    border: "1px solid rgba(0,0,0,0.4)",
                    background:
                      "radial-gradient(circle at 30% 30%, rgba(255,255,255,0.7), rgba(150,150,150,0.4) 60%, rgba(60,60,60,0.6) 100%)",
                    boxShadow: "inset 1px 1px 2px rgba(255,255,255,0.3)",
                  }}
                />
                <div
                  style={{
                    position: "absolute",
                    top: 4,
                    right: 4,
                    width: 12,
                    height: 12,
                    borderRadius: "50%",
                    border: "1px solid rgba(0,0,0,0.4)",
                    background:
                      "radial-gradient(circle at 30% 30%, rgba(255,255,255,0.7), rgba(150,150,150,0.4) 60%, rgba(60,60,60,0.6) 100%)",
                    boxShadow: "inset 1px 1px 2px rgba(255,255,255,0.3)",
                  }}
                />
                <div
                  style={{
                    position: "absolute",
                    bottom: 4,
                    left: 4,
                    width: 12,
                    height: 12,
                    borderRadius: "50%",
                    border: "1px solid rgba(0,0,0,0.4)",
                    background:
                      "radial-gradient(circle at 30% 30%, rgba(255,255,255,0.7), rgba(150,150,150,0.4) 60%, rgba(60,60,60,0.6) 100%)",
                    boxShadow: "inset 1px 1px 2px rgba(255,255,255,0.3)",
                  }}
                />
                <div
                  style={{
                    position: "absolute",
                    bottom: 4,
                    right: 4,
                    width: 12,
                    height: 12,
                    borderRadius: "50%",
                    border: "1px solid rgba(0,0,0,0.4)",
                    background:
                      "radial-gradient(circle at 30% 30%, rgba(255,255,255,0.7), rgba(150,150,150,0.4) 60%, rgba(60,60,60,0.6) 100%)",
                    boxShadow: "inset 1px 1px 2px rgba(255,255,255,0.3)",
                  }}
                />

                {/* pasek tytułu i lampki sygnalizacyjne usunięto na życzenie użytkownika */}

                {/* sekcja główna z pokrętłami i przyciskami */}
                <div
                  style={{
                    display: "flex",
                    alignItems: "center",
                    justifyContent: "space-between",
                    width: "100%",
                    gap: 10,
                    flexWrap: "wrap",
                    flex: 1,
                  }}
                >
                  <ArcadeKnob angle={knobL} />

                  <div className="flex flex-col items-center gap-2" style={{ minWidth: 240, flex: 1 }}>
                                    <div className="flex items-center gap-2">
                    <Btn
                      variant={(paused || arcadeActive.pause) ? "default" : "secondary"}
                      size="md"
                      onClick={onTogglePause}
                      style={{ minWidth: 92 }}
                    >
                      PAUZA
                    </Btn>
                    <Btn
                      variant={((started && !paused) || arcadeActive.start) ? "default" : "outline"}
                      size="md"
                      onClick={onStart}
                      style={{ minWidth: 92 }}
                    >
                      START
                    </Btn>
                    <Btn
                      variant={(arcadeActive.serve) ? "default" : "outline"}
                      size="md"
                      onClick={onServe}
                      style={{ minWidth: 92 }}
                    >
                      SERW
                    </Btn>
                  </div>
                  {/* prawe pokrętło */}
                  <ArcadeKnob angle={knobR} />
                </div>

              </div>
            </CardContent>
          </Card>

          <Card className="h-full">
            <CardContent className="p-4 space-y-4">
              <div>
                <div className="font-semibold" style={PANEL_TITLE_STYLE}>
                  TRYB GRY
                </div>
                <div
                  className="grid gap-2 mt-2"
                  style={{ gridTemplateColumns: "repeat(3, minmax(0, 1fr))" }}
                >
                  {MODELIST.map((mIt) => (
                    <Btn
                      key={mIt.key}
                      variant={mode === mIt.key ? "default" : "outline"}
                      size="sm"
                      onClick={() => {
                        const nextMode = mIt.key;
                        setMode(nextMode);
                        // KROK 2: na planszy startowej przełączanie trybu nie może ucinać muzyki.
                        if (!settingsRef.current.started && settingsRef.current.musicEnabled) {
                          ensureAudio().then(() => syncMusic(nextMode));
                        }
                      }}
                      style={{ width: "100%" }}
                    >
                      {mIt.label}
                    </Btn>
                  ))}
                </div>
                <div className="mt-2 text-sm" style={PANEL_MUTED_STYLE}>
                  {(getModeCfg(mode).blurb || []).slice(0, 3).map((line, idx) => (
                    <div key={idx}>{line}</div>
                  ))}
                </div>
              </div>

              <Divider />

              <div>
                <div className="font-semibold" style={PANEL_TITLE_STYLE}>
                  KOLORYSTYKA
                </div>
                <div
                  className="grid gap-2 mt-2"
                  style={{ gridTemplateColumns: "repeat(3, minmax(0, 1fr))" }}
                >
                  {Object.values(COLOR_THEMES).map((t) => (
                    <Btn
                      key={t.key}
                      variant={colorTheme === t.key ? "default" : "outline"}
                      size="sm"
                      onClick={() => setColorTheme(t.key)}
                      style={{ width: "100%" }}
                    >
                      {t.name}
                    </Btn>
                  ))}
                </div>
              </div>

              <Divider />

              <div>
                <div className="flex items-center justify-between mt-2">
                  <Label>Muzyka ON/OFF</Label>
                  <Switch
                    checked={musicEnabled}
                    onCheckedChange={(v) => {
                      setMusicEnabled(!!v);
                      ensureAudio().then(() => syncMusic());
                    }}
                  />
                </div>

                {/* Stała wysokość panelu po prawej: sekcja „Gra dla dwóch” zawsze zajmuje miejsce,
                    a poza Squash jest tylko niewidoczna (żeby tryby miały identyczną wysokość układu). */}
                <div
                  className="flex items-center justify-between mt-2"
                  style={mode === "squosh" ? undefined : { visibility: "hidden" }}
                  aria-hidden={mode === "squosh" ? undefined : true}
                >
                  <Label>Gra dla dwóch</Label>
                  <Switch
                    checked={squoshTwoPlayers}
                    disabled={mode !== "squosh"}
                    onCheckedChange={(v) => {
                      if (mode !== "squosh") return;
                      setSquoshTwoPlayers(!!v);
                      if (!v) setAiEnabled(false);
                    }}
                  />
                </div>

                <Divider />

                <div className="flex items-center justify-between mt-2">
                  <Label>Gra z komputerem</Label>
                  <Switch
                    checked={aiEnabled}
                    disabled={mode === "squosh" && !squoshTwoPlayers}
                    onCheckedChange={(v) => setAiEnabled(!!v)}
                  />
                </div>

                <div className="mt-3">
                  <Label>Poziom inteligencji komputera: {aiLevel}</Label>
                  <Slider
                    value={[aiLevel]}
                    min={1}
                    max={10}
                    step={1}
                    disabled={mode === "squosh" && !squoshTwoPlayers}
                    onValueChange={(v) => setAiLevel(v[0])}
                  />
                  <Divider />
                </div>

                <div className="mt-3">
                  <Label>Długość paletek: {paddleScale.toFixed(2)}</Label>
                  <Slider value={[paddleScale]} min={0.7} max={1.5} step={0.05} onValueChange={(v) => setPaddleScale(v[0])} />
                </div>

                <div className="mt-3">
                  <Label>Szybkość: {speedScale.toFixed(2)}</Label>
                  <Slider value={[speedScale]} min={0.6} max={1.8} step={0.05} onValueChange={(v) => setSpeedScale(v[0])} />
                </div>

                <div className="mt-3">
                  <Label>Limit punktów: {scoreToWin}</Label>
                  <Slider value={[scoreToWin]} min={3} max={25} step={1} onValueChange={(v) => setScoreToWin(v[0])} />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
